///////////////////////////////////////////////////////////////////////////////
//
//  File:       sssbase.h
//
//  Class:      SSStateBase - Abstract
//  
//  Hierarchy:  SSStateBase - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      Replay's selection screen uses a State Pattern to allow for easy
//      flexibility and extensibility of game options and this class serves
//      as a base class for all of the states of the selection screen.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _sssbase_h_
#define _sssbase_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "canvas.h"

//  Forward class declarations.
class SelectScreen;
class Keyboard;
class Bitmap;
class GfxSet;
class GameInfo;
class Game;
class ColourTable;

///////////////////////////////////////////////////////////////////////////////
//  SSStateBase Class Definition
///////////////////////////////////////////////////////////////////////////////
class SSStateBase : public RepBase
{
    public:

        //  The possible actions that a state can instruct the screen to do.
        typedef enum
        {
            SSS_CONTINUE,
            SSS_RUN,
            SSS_QUIT
        }
        Action;


        //  Creation/Deletion.
        SSStateBase ( 
            const KString& iName, SelectScreen* pSelectScreen, Canvas* pCanvas 
        );
        virtual ~SSStateBase ( );

        //  Interface.
        virtual Action      execute     ( ColourTable* pColourTable ) = 0;
        virtual Bitmap*     getScreen   ( ) = 0;
        virtual void        fillHelp    ( 
            Bitmap* pBitmap, Canvas::Font eFont, ColourTable* pColourTable 
        ) = 0;
        virtual void        setGameInfo ( GameInfo* pGameInfo );
        virtual void        setState    ( );
        void                setPrevious ( SSStateBase* pPreviousState );
        SSStateBase*        getPrevious ( ) const;

    protected:

        //  The possible default key actions.
        enum
        {
            DEF_NONE       = 0x0000,
            DEF_MAIN_MENU  = 0x0001,
            DEF_GAME_RUN   = 0x0002, 
            DEF_GAME_UP    = 0x0004,
            DEF_GAME_DOWN  = 0x0008
        };

        //  Utility.
        void     drawHelp    (
            Bitmap*      pBitmap,
            Canvas::Font eFont,
            ColourTable* pColourTable,
            char**       ppstrHelpText 
        );
        Action   checkKeys   ( DWord dwActions = 0xffff );

        //  Member Data.
        SelectScreen*       m_pSelectScreen;
        Canvas*             m_pCanvas;
        Keyboard*           m_pKeyboard;
        GameInfo*           m_pGameInfo;
        SSStateBase*        m_pPreviousState;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  SSStateBase Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
SSStateBase::setPrevious(
    SSStateBase* pPreviousState
)
{
    m_pPreviousState = pPreviousState;
}

inline
SSStateBase*
SSStateBase::getPrevious(
) const
{
    return( m_pPreviousState );
}

#endif
