///////////////////////////////////////////////////////////////////////////////
//
//  File:    ssscfg.cpp
//
//  Class:   SSStateControlConfig
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for allowing the
//      user to configure the game controls.  It allows for setting
//      both the global Replay game controls and for setting the current
//      game's controls.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "ssscfg.h"
#include "select.h"
#include "canvas.h"
#include "keyb.h"
#include "clock.h"
#include "registry.h"
#include "bitmap.h"
#include "ctrlmap.h"
#include "gameinfo.h"
#include "game.h"


///////////////////////////////////////////////////////////////////////////////
//  File Constants.
///////////////////////////////////////////////////////////////////////////////
//  Some strings.
static const char* LOCAL_HEADING  = "Local Definitions:";
static const char* GLOBAL_HEADING = "Global Definitions:";


///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateControlConfig
//
//  Description:
//
//      This is the main constructor for the selection screen control
//      configuration state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateControlConfig::SSStateControlConfig(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateList         ( iName, pSelectScreen, pCanvas, FALSE ),
    m_bGlobal           ( FALSE ),
    m_bWaiting          ( FALSE ),
    m_bControlsChanged  ( FALSE ),
    m_dwLastSwitchOn    ( DigitalController::SWITCH_NONE ),
    m_resourcePostfix   ( ),
    m_pController       ( NULL ),
    m_eAssoc            ( CtrlMap::ASSOC_COUNT )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateControlConfig
//
//  Description:
//
//      This is the destructor for the selection screen control configuration 
//      state object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateControlConfig::~SSStateControlConfig(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setGameInfo
//
//  Description:
//
//      This is called when a new game has been selected.  It is overridden
//      so that we can clear out our list of lines from the previous game
//      information.
//
//  Parameters:
//
//      pGameInfo (input)
//          The information for the new game.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateControlConfig::setGameInfo(
    GameInfo* pGameInfo
)
{
    //  Call the base class.
    SSStateBase::setGameInfo( pGameInfo );

    //  Add the control maps for either the global or local config.
    addItems( );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the control configuration state.  The base
//      method is used to display the controls available for redefinition 
//      and supports scrolling.  This function allows the user to 
//      choose a new control for the job.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateControlConfig::execute(
    ColourTable* pColourTable 
)
{
    //  Call the base class.
    SSStateList::execute( pColourTable );

    //  If we are waiting then we keep checking until the last
    //  switch pressed changes so that we know what the new switch is.
    if( m_bWaiting )
    {
        //  ESC aborts the new switch selection.
        if( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
        { 
            m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );

            m_bWaiting = FALSE;
            updateCurItem( );
        }
        else
        {
            //  This indicates the switch that the user has selected.
            DWord dwSwitchSelected = DigitalController::SWITCH_NONE;

            ASSERT( m_pController != NULL );

            //  If the last switch on has changed then use that as the new 
            //  switch.
            if( m_pController->lastSwitchOn( ) != m_dwLastSwitchOn )
            {
                dwSwitchSelected = m_pController->lastSwitchOn( );
            }

            //  If a switch has been selected then update the control map.
            if( dwSwitchSelected != DigitalController::SWITCH_NONE )
            {
                //  Update the affected control maps with the selected switch.
                updateCtrlMaps( dwSwitchSelected );
    
                m_bWaiting = FALSE;
                updateCurItem( );
            }
        }

        return( SSS_CONTINUE );
    }
    else
    //  Exit?  Go back to the previous state.
    if( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );

        //  If the controls have changed then we need to set them
        //  to the new values.
        if( m_bControlsChanged )
        {
            ASSERT( m_pSelectScreen->getGame( ) != NULL );
            m_pSelectScreen->getGame( )->resetControls( );
        }

        //  Move back to the previous state.
        m_pSelectScreen->setState( SelectScreen::STATE_PREVIOUS );
    }
    else
    //  Reset the current control?
    if( m_pKeyboard->switchOn( Keyboard::KEY__BS ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__BS );

        //  Reset the Control Maps involved.
        resetCtrlMaps( );

        //  Update the current item.
        updateCurItem( );
    }
    else
    //  Global<->Local?
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__TAB ) )
    {
        //  Toggle the screen displayed.
        m_bGlobal = m_bGlobal ? FALSE : TRUE;
        addItems( );
        m_pSelectScreen->update( );
    }
    else
    //  Change Control?  Only allowed if there is a current control.
    if( 
        ( m_pKeyboard->switchOn( Keyboard::KEY__ENTER ) ) &&
        ( m_lineList.entries( ) > 0 )
    )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ENTER );

        //  Clear the last switch that was on and hold on to the cleared value.
        m_pController->clearLastSwitchOn( );
        m_dwLastSwitchOn = m_pController->lastSwitchOn( );

        //  We are now waiting for a key.
        m_bWaiting = TRUE;

        //  Update the current item.
        updateCurItem( );
    }
    else
    {
        //  Let the base class check for scrolling.
        SSStateList::checkForScroll( );
    }
    

    //  Allow for the standard keys (like rotate, flip, etc.).
    SSStateBase::checkKeys( DEF_NONE );


    //  The game should not be started yet.
    return( SSS_CONTINUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addItems
//
//  Description:
//
//      This function is responsible for adding the current control mappings
//      to the list of lines to be displayed.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateControlConfig::addItems( 
)
{
    //  Clear the header and lines.
    clearHeader( );
    clearLines( );
    
    //  Are we displaying the global or local?
    if( m_bGlobal )
    {
        //  Add the global header in.
        m_headerList.add( new KString( GLOBAL_HEADING ) );
        
        //  Display the controls that can be defined.
        for( DWord dwI = 0 ; dwI < CtrlMap::GLOBAL_CONTROL_COUNT ; dwI += 1 )
        {
            //  Add the item.
            m_lineList.add( 
                createItem( 
                    CtrlMap::s_getGlobalControl( 
                        ( CtrlMap::GlobalControl )dwI 
                    ), 
                    dwI
                )
            );
        }
    }
    else
    {
        //  Retrieve the list of control mappings for the current game.
        ASSERT( m_pSelectScreen->getGame( ) != NULL );
        const KPtrList<CtrlMap>& ctrlMapList = 
            m_pSelectScreen->getGame( )->getCtrlMapList( );
    
        //  Add the local header in.
        m_headerList.add( new KString( LOCAL_HEADING ) );
    
        //  Display the controls that can be defined.
        for( DWord dwI = 0 ; dwI < ctrlMapList.entries( ) ; dwI += 1 )
        {
            //  Add the item.  
            m_lineList.add( createItem( ctrlMapList[ dwI ], dwI ) );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateCurItem
//
//  Description:
//
//      This function is responsible for updating the current item.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateControlConfig::updateCurItem( 
)
{
    //  The new item.
    KString* pNewItem;


    //  Is this global or local?
    if( m_bGlobal )
    {
        ASSERT( m_dwCurLine < m_lineList.entries( ) );
        ASSERT( m_dwCurLine < CtrlMap::GLOBAL_CONTROL_COUNT );

        //  Get the new item.
        pNewItem = createItem( 
            CtrlMap::s_getGlobalControl( 
                ( CtrlMap::GlobalControl )m_dwCurLine 
            ),
            m_dwCurLine
        ); 
    }
    else
    {
        //  Retrieve the list of control mappings for the current game.
        ASSERT( m_pSelectScreen->getGame( ) != NULL );
        const KPtrList<CtrlMap>& ctrlMapList = 
            m_pSelectScreen->getGame( )->getCtrlMapList( );
    
        ASSERT( m_dwCurLine < m_lineList.entries( ) );
        ASSERT( m_dwCurLine < ctrlMapList.entries( ) );

        //  Get the new item.
        pNewItem = createItem( ctrlMapList[ m_dwCurLine ], m_dwCurLine );
    }
        
    //   Place the new item in the list.
    *( m_lineList[ m_dwCurLine ] ) = *pNewItem;

    //  All done with the new item.
    delete pNewItem;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: createItem
//
//  Description:
//
//      This function is responsible for creating the specified item in the
//      displayed list of controls.
//
//  Parameters:
//
//      pCtrlMap (input)
//          A pointer to the control mapping to draw.
//
//      dwIndex (input)
//          The index of the specified control within the list.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString*
SSStateControlConfig::createItem(
    const CtrlMap* pCtrlMap,
    const DWord    dwIndex
)
{
    //  The current line begins with the name of the control.
    KString* pCurLine = new KString( pCtrlMap->getInstanceName( ) );
    KString& rCurLine = *pCurLine;


    //  Pad with spaces between the description and the switch def.
    rCurLine.pad( ' ', 18 - rCurLine.length( ) );

    //  If we are waiting for the user to designate a switch for this
    //  control then we display "???", otherwise we display the name of 
    //  the current switch assigned.
    if( ( m_bWaiting == TRUE ) && ( dwIndex == m_dwCurLine ) )
    {
        rCurLine += "    ???";
    }
    else
    {
        //  We mark the key as follows:
        //      (U) - User specified.
        //      (G) - Global default.
        //      (D) - Game Default.
        switch( pCtrlMap->getSource( m_eAssoc ) )
        {
            case CtrlMap::FROM_USER:   rCurLine += "(U) "; break;
            case CtrlMap::FROM_GLOBAL: rCurLine += "(G) "; break;
            case CtrlMap::FROM_GAME:   rCurLine += "(D) "; break;
            default:
            {
                fatalError( 
                    "Unknown Control Map Source %d", 
                    pCtrlMap->getSource( m_eAssoc )
                );
                break;
            }
        }
        rCurLine += m_pController->switchName( pCtrlMap->get( m_eAssoc ) );
    }

    //  Return back the new line.
    return( pCurLine );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateCtrlMaps
//
//  Description:
//
//      This function is used to set the current control map to the 
//      specified switch and if the control map is global it will update
//      any local controls that default to the global.
//
//  Parameters:
//
//      dwSelectedSwitch (input)
//          The switch that was selected by the user for the current control.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateControlConfig::updateCtrlMaps(
    DWord dwSwitchSelected
)
{
    //  The current control map.
    CtrlMap* pCtrlMap( NULL );

    //  Get the registries that contain the settings.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    Registry* pLocalSettings = 
        m_pSelectScreen->getGame( )->getSettingsRegistry( );
    Registry* pGlobalSettings = 
        Configuration::s_instance( ).getSettingsRegistry( );

    //  Retrieve the list of control mappings for the current game.
    const KPtrList<CtrlMap>& ctrlMapList = 
        m_pSelectScreen->getGame( )->getCtrlMapList( );
    


    //  Get the current control map.
    if( m_bGlobal )
    {
        pCtrlMap = CtrlMap::s_getGlobalControl( 
            ( CtrlMap::GlobalControl )m_dwCurLine 
        );
    }
    else
    {
        pCtrlMap = ctrlMapList[ m_dwCurLine ];
    }


    //  Set the control map to use the new switch and set the source to indicate
    //  that the control is user chosen.
    pCtrlMap->set( m_eAssoc, dwSwitchSelected );
    pCtrlMap->setSource( m_eAssoc, CtrlMap::FROM_USER );

    //  Is the control a global control?  If so we process it slightly
    //  differently.
    if( m_bGlobal )
    {
        //  Save the global control to the settings registry.
        KString resourceName = pCtrlMap->getInstanceName( );
        resourceName += m_resourcePostfix;
        pGlobalSettings->setValue( 
            "replay", 
            resourceName,
            ( Byte* )&dwSwitchSelected,
            sizeof( dwSwitchSelected )
        );

        //  Now search the local list of controls and modify any matching
        //  control maps to the new switch.
        for( DWord dwI = 0 ; dwI < ctrlMapList.entries( ) ; dwI += 1 )
        {
            //  If the local control map matches the current control map
            //  then set its switch to the same as the current control.
            //  Of course, the local control must be sourced from the global
            //  control.
            if( 
                ( 
                    ctrlMapList[ dwI ]->getSource( m_eAssoc ) == 
                    CtrlMap::FROM_GLOBAL 
                ) &&
                ( ctrlMapList[ dwI ]->getGlobal( ) == pCtrlMap->getGlobal( ) )
            )
            {
                //  Set the switch.  No need to set the source since it is 
                //  still coming from the global setting.
                ctrlMapList[ dwI ]->set( m_eAssoc, dwSwitchSelected );

                //  Note that the controls need setting.
                m_bControlsChanged = TRUE;
            }
        }
    }
    else
    {
        //  Save the local control to the settings registry.
        KString resourceName = pCtrlMap->getInstanceName( );
        resourceName += m_resourcePostfix;
        pLocalSettings->setValue( 
            m_pGameInfo->getGameId( ),
            resourceName,
            ( Byte* )&dwSwitchSelected,
            sizeof( dwSwitchSelected )
        );

        //  Note that the controls need setting.
        m_bControlsChanged = TRUE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: resetCtrlMaps
//
//  Description:
//
//      This function is used to reset the current control map to the 
//      default and if the control map is global it will reset
//      any local controls that default to the global.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateControlConfig::resetCtrlMaps(
)
{
    //  The current control map.
    CtrlMap* pCtrlMap( NULL );

    //  Get the registry that contains the settings.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    Registry* pLocalSettings = 
        m_pSelectScreen->getGame( )->getSettingsRegistry( );
    Registry* pGlobalSettings = 
        Configuration::s_instance( ).getSettingsRegistry( );

    //  Retrieve the list of control mappings for the current game.
    const KPtrList<CtrlMap>& ctrlMapList = 
        m_pSelectScreen->getGame( )->getCtrlMapList( );
    


    //  Is the control a global control?  If so we process it slightly
    //  differently.
    if( m_bGlobal )
    {
        //  Get the current control map.
        pCtrlMap = CtrlMap::s_getGlobalControl( 
            ( CtrlMap::GlobalControl )m_dwCurLine 
        );

        //  Remove the global control from the settings registry.
        KString resourceName( pCtrlMap->getInstanceName( ) );
        resourceName += m_resourcePostfix;
        pGlobalSettings->deleteValue( "replay", resourceName );

        //  Set the global control back to the default switch.
        pCtrlMap->set( m_eAssoc, pCtrlMap->getDefault( m_eAssoc ) );
        pCtrlMap->setSource( m_eAssoc, CtrlMap::FROM_GLOBAL );

        //  Now search the local list of controls and modify any matching
        //  control maps.
        for( DWord dwI = 0 ; dwI < ctrlMapList.entries( ) ; dwI += 1 )
        {
            //  If the local control map is sourced from the global control
            //  and matches the current global control map
            //  then set its key to the new key of the global control map.
            if( 
                ( 
                    ctrlMapList[ dwI ]->getSource( m_eAssoc ) == 
                    CtrlMap::FROM_GLOBAL 
                ) &&
                ( ctrlMapList[ dwI ]->getGlobal( ) == pCtrlMap->getGlobal( ) )
            )
            {
                //  Reset the key.  No need to set the source since it is still
                //  coming from the global setting.
                ctrlMapList[ dwI ]->set( m_eAssoc, pCtrlMap->get( m_eAssoc ) );

                //  Note that the local controls need setting.
                m_bControlsChanged = TRUE;
            }
        }
    }
    else
    {
        //  Get the current control map.
        pCtrlMap = ctrlMapList[ m_dwCurLine ];

        //  Remove the local control from the settings registry.
        KString resourceName( pCtrlMap->getInstanceName( ) );
        resourceName += m_resourcePostfix;
        pLocalSettings->deleteValue( 
            m_pGameInfo->getGameId( ), resourceName
        );

        //  Either reset the control to the global setting if it is mapped
        //  to a global control or to the default setting otherwise.
        if( pCtrlMap->getGlobal( ) != CtrlMap::NO_GLOBAL )
        {
            pCtrlMap->set(
                m_eAssoc,
                CtrlMap::s_getGlobalControl( 
                    pCtrlMap->getGlobal( )
                )->get( m_eAssoc )
            );
            pCtrlMap->setSource( m_eAssoc, CtrlMap::FROM_GLOBAL );
        }
        else
        {
            pCtrlMap->set( m_eAssoc, pCtrlMap->getDefault( m_eAssoc ) );
            pCtrlMap->setSource( m_eAssoc, CtrlMap::FROM_GAME );
        }

        //  Note that the controls need setting.
        m_bControlsChanged = TRUE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Exit
//          UP  Prev
//          DN  Next
//          RET Set
//          BS  Reset
//          TAB Glob/Loc
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateControlConfig::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpGlobalText[ ] = 
    {
        "ESC Exit",
        "UP  Prev",
        "DN  Next", 
        "RET Set",  
        "BS  Reset",
        "TAB Local", 
        NULL
    };
    static char* ppstrHelpLocalText[ ] = 
    {
        "ESC Exit",
        "UP  Prev",
        "DN  Next", 
        "RET Set",  
        "BS  Reset",
        "TAB Global", 
        NULL
    };

    //  Draw the text.
    if( m_bGlobal )
    {
        drawHelp( pBitmap, eFont, pColourTable, ppstrHelpGlobalText );
    }
    else
    {
        drawHelp( pBitmap, eFont, pColourTable, ppstrHelpLocalText );
    }
}
