///////////////////////////////////////////////////////////////////////////////
//
//  File:    sssinit.cpp
//
//  Class:   SSStateInitialize
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for initializing
//      a game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <stdlib.h>

//  Application Headers.
#include "sssinit.h"
#include "select.h"
#include "canvas.h"
#include "clock.h"
#include "bitmap.h"
#include "keyb.h"
#include "gameinfo.h"
#include "game.h"
#include "ctable.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data Initialization.
///////////////////////////////////////////////////////////////////////////////

//  The time delay is 3 seconds before the current game is initialized.
const DWord SSStateInitialize::sm_dwTimeDelay = 3000000;



///////////////////////////////////////////////////////////////////////////////
//  File Constants.
///////////////////////////////////////////////////////////////////////////////
//  Some strings.
static const char* INIT_MESSAGE      = "Booting...";

//  The screen attributes.
static const DWord INIT_MESSAGE_HEIGHT = 8;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateInitialize
//
//  Description:
//
//      This is the main constructor for the selection screen initialization 
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateInitialize::SSStateInitialize(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateBase     ( iName, pSelectScreen, pCanvas ),
    m_bNewGame      ( TRUE ),
    m_bWaiting      ( TRUE ),
    m_dwStartTime   ( 0 ),
    m_pScreenBitmap ( NULL ),
    m_pClock        ( NULL )
{
    //  We use the default screen bitmap.
    m_pScreenBitmap = pSelectScreen->getDefStateBitmap( );

    //  Create the bounding box for the text we are going to draw.
    m_clipping.m_nMinX = 0;
    m_clipping.m_nMaxX = m_pScreenBitmap->getWidth( ) - 1;
    m_clipping.m_nMinY = 
        ( m_pScreenBitmap->getHeight( ) - INIT_MESSAGE_HEIGHT ) / 2;
    m_clipping.m_nMaxY = m_pScreenBitmap->getHeight( ) - 1;

    //  Get the clock.
    m_pClock = Replay::s_instance( ).getClock( );
    ASSERT( m_pClock != NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateInitialize
//
//  Description:
//
//      This is the destructor for the selection screen initialization state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateInitialize::~SSStateInitialize(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen initialize state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStateInitialize::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStateInitialize" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setGameInfo
//
//  Description:
//
//      This is called when a new game has been selected.  It performs
//      initialization to indicate that a new game has been selected 
//      and therefore the initial delay should be invoked.
//
//  Parameters:
//
//      pGameInfo (input)
//          The information for the new game.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateInitialize::setGameInfo(
    GameInfo* pGameInfo
)
{
    //  Call the base class to ensure proper processing.
    SSStateBase::setGameInfo( pGameInfo );

    //  The initial delay should be invoked.
    m_bNewGame = TRUE;
    m_bWaiting = TRUE;
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the initialization state.  It basically
//      makes a call to the current games startup method until the game
//      has finished the startup procedure.
//
//      If the game starts successfully then we move on to the demo state,
//      otherwise we move on to the missing files state.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateInitialize::execute(
    ColourTable* pColourTable 
)
{
    //  We should have a selection screen specified by now.
    ASSERT( m_pSelectScreen != NULL );
    ASSERT( m_pGameInfo     != NULL );
 
    //  This indicates whether the game's startup function needs to be 
    //  called again.
    Byte bCallAgain;
    
   
    //  If the game does not have the required files available then go
    //  directly to the missing state.
    if( m_pGameInfo->requiredFilesAvailable( ) == FALSE )
    {
        m_pSelectScreen->setState( SelectScreen::STATE_MISSING_FILES );
        return( SSS_CONTINUE );
    }

    //  Clear the screen bitmap.
    m_pScreenBitmap->clear( ( *pColourTable )[ 0 ] );

    //  If we are waiting then show static, otherwise show a "booting" type
    //  message.
    if( m_bWaiting )
    {
        for( DWord dwY = 0 ; dwY < m_pScreenBitmap->getHeight( ) ; dwY += 1 )
        {
            for( DWord dwX = 0 ; dwX < m_pScreenBitmap->getWidth( ) ; dwX += 1 )
            {
                m_pScreenBitmap->setPixel( 
                    dwX, dwY, ( *pColourTable )[ rand( ) % 2 ]
                );
            }
        }
    }
    else
    {
        m_pCanvas->drawText( 
            INIT_MESSAGE,
            m_pScreenBitmap,
            Canvas::FONT_SMALL,
            &m_clipping,
            *pColourTable,
            TRUE
        );
    }

    //  So things don't get overwhelming, we wait for a frame.
    m_pClock->frameWait( );

    //  If a new game has been selected then get the starting clock value.
    if( m_bNewGame == TRUE )
    {
        m_dwStartTime = m_pClock->getTime( );
        m_bNewGame = FALSE;
    }

    //  If we are waiting then check if the time delay has expired.
    if( m_bWaiting == TRUE )
    {
        //  If the time has expired then we're no longer waiting.
        if( ( m_pClock->getTime( ) - m_dwStartTime ) > sm_dwTimeDelay )
        {
            m_bWaiting = FALSE;
        }
    }
    else
    {
        //  If there isn't a game yet, then create one.
        if( m_pSelectScreen->getGame( ) == NULL )
        {
            m_pSelectScreen->setGame( );
        }

        //  Continue the startup.
        bCallAgain = m_pSelectScreen->getGame( )->startUp( );

        //  If the game startup is complete then we need to move on to the 
        //  demo state.
        if( bCallAgain == FALSE )
        {
            m_bNewGame = TRUE;
            m_bWaiting = TRUE;
            m_pSelectScreen->setState( SelectScreen::STATE_DEMO );
            return( SSS_CONTINUE );
        }
    }

    //  Return the result of checking the default keys.
    return( checkKeys( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Main Menu
//          UP  Prev Game
//          DN  Next Game
//          RET Run  Game
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateInitialize::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpText[ ] = 
    {
        "ESC Main Menu",
        "UP  Prev Game",
        "DN  Next Game",
        "RET Run  Game",
        NULL
    };

    //  Draw the text.
    drawHelp( pBitmap, eFont, pColourTable, ppstrHelpText );
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getScreen
//
//  Description:
//
//      This returns the screen bitmap used by the initialization state.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      A pointer to the screen bitmap of the state.
//
///////////////////////////////////////////////////////////////////////////////
Bitmap* 
SSStateInitialize::getScreen ( 
)
{
    return( m_pScreenBitmap );
} 
