///////////////////////////////////////////////////////////////////////////////
//
//  File:    ssslist.cpp
//
//  Class:   SSStateList
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract selection screen state that can be derived from
//      to create states that display list of items that can be scrolled.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "ssslist.h"
#include "replay.h"
#include "select.h"
#include "canvas.h"
#include "keyb.h"
#include "clock.h"
#include "bitmap.h"
#include "gfxset.h"
#include "ctable.h"
#include "clip.h"


///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateList
//
//  Description:
//
//      This is the main constructor for the selection screen list
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//      bScrollLock (input)
//          Indicates whether the list should behave with "Locked" scrolling.
//          this means that the list always scrolls with up/down whereas
//          "Unlocked" means that the list only scrolls when the movement
//          goes off the top or bottom of the displayed list.
//
//      bCenter (input)
//          Indicates whether or not to center the items in the list.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateList::SSStateList(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas,
    Byte           bScrollLock /* = FALSE */,
    Byte           bCenter     /* = FALSE */
)
:
    SSStateBase     ( iName, pSelectScreen, pCanvas ),
    m_pScreenBitmap ( NULL ),
    m_headerList    ( 2 ),
    m_lineList      ( 20 ),
    m_bScrollLock   ( bScrollLock ),
    m_bCenter       ( bCenter ),
    m_dwTopLine     ( 0 ),
    m_dwBotLine     ( 0 ),
    m_dwCurLine     ( 0 ),
    m_pFont         ( NULL ),
    m_nFontHeight   ( 0 ),
    m_nFontWidth    ( 0 )
{
    //  Initialize
    m_pFont = &( m_pCanvas->getFont( Canvas::FONT_SMALL ) );
    m_nFontHeight = ( int32 )( *m_pFont )[ 0 ]->getHeight( );
    m_nFontWidth  = ( int32 )( *m_pFont )[ 0 ]->getWidth( );

    //  We use the default screen bitmap.
    m_pScreenBitmap = pSelectScreen->getDefStateBitmap( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateList
//
//  Description:
//
//      This is the destructor for the selection screen list state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateList::~SSStateList(
)
{
    //  Clear out the header and the lines.
    clearHeader( );
    clearLines( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clearHeader
//
//  Description:
//
//      This is used to clear out the list of header lines.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateList::clearHeader(
)
{
    //  Clear out the list of header lines.
    for( DWord dwI = 0 ; dwI < m_headerList.entries( ) ; dwI += 1 )
    {
        delete m_headerList[ dwI ];
    }
    m_headerList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clearLines
//
//  Description:
//
//      This is used to clear out the list of lines.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateList::clearLines(
)
{
    //  Clear out the list of lines.
    for( DWord dwI = 0 ; dwI < m_lineList.entries( ) ; dwI += 1 )
    {
        delete m_lineList[ dwI ];
    }
    m_lineList.clear( );

    //  Since the list is empty, reset the indices.
    m_dwCurLine = 0;
    m_dwTopLine = 0;
    m_dwBotLine = 0;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the list state.  It displays the list
//      of lines and indicates if there are lines that are available
//      either above or below the current visible area.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateList::execute(
    ColourTable* pColourTable 
)
{
    //  The colour table used for highlighting.
    ColourTable highlightTable( "Highlight CTable", 2 );

    //  The clipping table.
    Clipping clipping(
        m_nFontWidth, 
        m_pScreenBitmap->getWidth( ) - m_nFontWidth - 1,
        m_nFontHeight,
        m_pScreenBitmap->getHeight( ) - m_nFontHeight - 1
    );

    //  We should have a selection screen and a game specified by now.
    ASSERT( m_pSelectScreen != NULL );
    ASSERT( m_pGameInfo     != NULL );
 

    //  Clear the screen bitmap.
    m_pScreenBitmap->clear( ( *pColourTable )[ 0 ] );


    //  Assign the colours to the colour table used to highlight an item.
    highlightTable[ 0 ] = ( *pColourTable )[ 0 ];
    highlightTable[ 1 ] = ( *pColourTable )[ 2 ];


    //  First of all, draw the header.
    for( DWord dwI = 0 ; dwI < m_headerList.entries( ) ; dwI += 1 )
    {
        //  Draw the current header line into the bitmap.
        m_pCanvas->drawText(
            *( m_headerList[ dwI ] ),
            m_pScreenBitmap,
            Canvas::FONT_SMALL,
            &clipping,
            *pColourTable,
            m_bCenter
        );
        clipping.m_nMinY += 8;
    }


    //  Calculate the bottom line based on the top line and the space
    //  remaining in the clipping area.
    m_dwBotLine = 
        m_dwTopLine + 
        ( clipping.m_nMaxY - clipping.m_nMinY + 1) / m_nFontHeight - 1;
    if( m_dwBotLine >= m_lineList.entries( ) )
    {
        m_dwBotLine = m_lineList.entries( ) - 1;
    }
    

    //  If we are not at the top of the line list then draw the up indicator.
    if( m_dwTopLine != 0 )
    {
        m_pScreenBitmap->blit(
            ( *m_pFont )[ 0 ],
            m_pScreenBitmap->getWidth( ) - 2 * m_nFontWidth,
            m_pScreenBitmap->getHeight( ) - m_nFontHeight,
            *pColourTable,
            FALSE,
            TRUE,
            m_pScreenBitmap->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE,
            0
        ); 
    }

    //  If we are not at the bottom of the line list then draw the down
    //  indicator.
    if( m_dwBotLine < m_lineList.entries( ) - 1 )
    {
        m_pScreenBitmap->blit(
            ( *m_pFont )[ 0 ],
            m_pScreenBitmap->getWidth( ) - m_nFontWidth,
            m_pScreenBitmap->getHeight( ) - m_nFontHeight,
            *pColourTable,
            FALSE,
            FALSE,
            m_pScreenBitmap->getFullClipping( ),
            Bitmap::TRANSPARENCY_NONE,
            0
        ); 
    }


    //  To distinguish the lines from the header we indent them.
    clipping.m_nMinX += m_nFontWidth;
    clipping.m_nMaxX -= m_nFontWidth;


    //  Draw the lines.
    for( 
        DWord dwI = m_dwTopLine ; 
        ( dwI < m_lineList.entries( ) ) && ( dwI <= m_dwBotLine ) ; 
        dwI += 1 
    )
    {
        //  Draw the current line into the bitmap.  If this is the current
        //  line and scroll lock isn't on then highlight the item.
        if( ( dwI == m_dwCurLine ) && ( m_bScrollLock == FALSE ) )
        {
            m_pCanvas->drawText(
                *( m_lineList[ dwI ] ),
                m_pScreenBitmap,
                Canvas::FONT_SMALL,
                &clipping,
                highlightTable,
                m_bCenter
            );
        }
        else
        {
            m_pCanvas->drawText(
                *( m_lineList[ dwI ] ),
                m_pScreenBitmap,
                Canvas::FONT_SMALL,
                &clipping,
                *pColourTable,
                m_bCenter
            );
        }
        clipping.m_nMinY += 8;
    }

    //  So things don't get overwhelming, we wait for a frame.
    Replay::s_instance( ).getClock( )->frameWait( );

    //  Return with continue (we let the derived class decide what to really
    //  return).
    return( SSS_CONTINUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: checkForScroll
//
//  Description:
//
//      This is used to check the keys associated with the list state.
//      The only keys processed are the up and down keys for changing 
//      the current position within the list.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateList::checkForScroll(
)
{
    //  We process differently depending on the scroll lock status.
    if( m_bScrollLock == FALSE )
    {
        //  Move up?  Move up one line.
        if( m_pKeyboard->switchOnWait( Keyboard::KEY__UP ) ) 
        {
            //  Find the previous non-null line.
            //  Don't be alarmed at the check in the for loop.  The reason
            //  why we don't check for >= 0 is that once we subtract 1 from
            //  0 we don't go negative (since this is unsigned) we go to 
            //  some large value.
            for( 
                DWord dwI = m_dwCurLine - 1 ; 
                dwI < m_lineList.entries( ) ;
                dwI -= 1
            )
            {
                //  If we've found a non-null line then assign the current line
                //  to it and quit the loop.
                if(  m_lineList[ dwI ]->length( ) > 0 )
                {
                    m_dwCurLine = dwI;
                    break;
                }
            }
    
            //  If we've moved past the top line then adjust the top.
            if( m_dwCurLine < m_dwTopLine )
            {
                m_dwTopLine -= 1;
            }
        }
        else
        //  Move down?  Move down one line.
        if( m_pKeyboard->switchOnWait( Keyboard::KEY__DOWN ) )
        {
            //  Find the next non-null line.
            for( 
                DWord dwI = m_dwCurLine + 1 ; 
                dwI < m_lineList.entries( ) ;
                dwI += 1
            )
            {
                //  If we've found a non-null line then assign the current line
                //  to it and quit the loop.
                if( m_lineList[ dwI ]->length( ) > 0 )
                {
                    m_dwCurLine = dwI;
                    break;
                }
            }
    
            //  If we've moved past the bottom line then adjust the top (the
            //  bottom will be adjusted later).
            if( m_dwCurLine > m_dwBotLine )
            {
                m_dwTopLine += 1;
            }
        }
    }
    else
    {
        //  Move up?  Move up one line.
        if( m_pKeyboard->switchOnWait( Keyboard::KEY__UP ) ) 
        {
            //  If we're not at the top line then move up.
            if( m_dwTopLine > 0 )
            {
                m_dwTopLine -= 1;
                m_dwCurLine = m_dwTopLine;
            }
        }
        else
        //  Move down?  Move down one line.
        if( m_pKeyboard->switchOnWait( Keyboard::KEY__DOWN ) )
        {
            //  If we're not at the bottom line then move down (adjust the
            //  top and the bottom will follow).
            if( m_dwBotLine < m_lineList.entries( ) - 1 )
            {
                m_dwTopLine += 1;
                m_dwCurLine = m_dwTopLine;
            }
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getScreen
//
//  Description:
//
//      This returns the screen bitmap used by the game info state.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      A pointer to the screen bitmap of the state.
//
///////////////////////////////////////////////////////////////////////////////
Bitmap* 
SSStateList::getScreen ( 
)
{
    return( m_pScreenBitmap );
} 
