///////////////////////////////////////////////////////////////////////////////
//
//  File:    sssmiss.cpp
//
//  Class:   SSStateMissingFiles
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for displaying
//      the required files for the current game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "sssmiss.h"
#include "select.h"
#include "canvas.h"
#include "bitmap.h"
#include "gameinfo.h"
#include "game.h"


///////////////////////////////////////////////////////////////////////////////
//  File Constants.
///////////////////////////////////////////////////////////////////////////////
//  Some strings.
static const char* MISS_MESSAGE_1 = "Required Files Missing!";
static const char* MISS_MESSAGE_2 = "The following are needed:";


///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateMissingFiles
//
//  Description:
//
//      This is the main constructor for the selection screen missing files 
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateMissingFiles::SSStateMissingFiles(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateList     ( iName, pSelectScreen, pCanvas, TRUE )
{
    //  Add the header lines to the header list.
    m_headerList.add( new KString( MISS_MESSAGE_1 ) );
    m_headerList.add( new KString( MISS_MESSAGE_2 ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateMissingFiles
//
//  Description:
//
//      This is the destructor for the selection screen missing files state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateMissingFiles::~SSStateMissingFiles(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen missing files state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStateMissingFiles::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStateMissingFiles" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setGameInfo
//
//  Description:
//
//      This is called when a new game has been selected.  It is overridden
//      so that we can clear out our list of lines from the previous game
//      ROMs.
//
//  Parameters:
//
//      pGameInfo (input)
//          The information for the new game.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateMissingFiles::setGameInfo(
    GameInfo* pGameInfo
)
{
    //  Call the base class.
    SSStateBase::setGameInfo( pGameInfo );

    //  Clear out the list of lines.
    clearLines( );

    //  Add the names of the ROMs for the game.
    for( 
        DWord dwI = 0 ; 
        m_pGameInfo->getReqdFile( dwI ) != KStringNULL ; 
        dwI += 1 
    )
    {
        //  Add the game to the list.
        m_lineList.add( new KString( m_pGameInfo->getReqdFile( dwI ) ) );
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the missing files state.  The base method
//      displays the missing files on screen.  Since up/down are used
//      to move between games we currently don't allow scrolling.  This
//      shouldn't be a problem since games generally don't have that many
//      ROMs.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateMissingFiles::execute(
    ColourTable* pColourTable 
)
{
    //  Call the base class.
    SSStateList::execute( pColourTable );

    //  Return the result of checking the default keys.  We disallow run
    //  though since there are files missing.
    return( checkKeys( DEF_MAIN_MENU | DEF_GAME_UP | DEF_GAME_DOWN ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Main Menu
//          UP  Prev Game
//          DN  Next Game
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateMissingFiles::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpText[ ] = 
    {
        "ESC Main Menu",
        "UP  Prev Game",
        "DN  Next Game",
        NULL
    };

    //  Draw the text.
    drawHelp( pBitmap, eFont, pColourTable, ppstrHelpText );
}
