///////////////////////////////////////////////////////////////////////////////
//
//  File:    sssmsg.cpp
//
//  Class:   SSStateMessage
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for displaying an
//      informational message to the user.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "sssmsg.h"
#include "select.h"
#include "canvas.h"
#include "keyb.h"
#include "bitmap.h"



///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
SSStateMessage::Message SSStateMessage::sm_eMessage = SSStateMessage::MSG_NONE;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStateMessage
//
//  Description:
//
//      This is the main constructor for the selection screen message
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateMessage::SSStateMessage(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateList     ( iName, pSelectScreen, pCanvas, TRUE ),
    m_eMessage      ( MSG_NONE )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStateMessage
//
//  Description:
//
//      This is the destructor for the selection screen message state
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStateMessage::~SSStateMessage(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen message state
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStateMessage::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStateMessage" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_setMessage
//
//  Description:
//
//      This is a static method that indicates what the new message should be.
//
//  Parameters:
//
//      eMessage (input)
//          The new message to display.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateMessage::s_setMessage(
    const Message eMessage
)
{
    sm_eMessage = eMessage;
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: setMessage
//
//  Description:
//
//      This is called to indicate that a new message is to be displayed.
//
//  Parameters:
//
//      eMessage (input)
//          The new message to display.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateMessage::setMessage(
)
{
    //  Clear out the list of lines.
    clearLines( );

    //  Add the desired message to the list of lines.
    switch( m_eMessage )
    {
        case MSG_NONE:
        {
            break;
        }
        case MSG_SAVE_GOOD:
        {
            m_lineList.add( new KString( "Save successful." ) ); break;
        }
        case MSG_SAVE_BAD:
        {
            m_lineList.add( new KString( "Error during save!" ) ); break;
        }
        case MSG_LOAD_GOOD:
        {
            m_lineList.add( new KString( "Load successful." ) ); break;
        }
        case MSG_LOAD_BAD:
        {
            m_lineList.add( new KString( "Error during load!" ) ); break;
        }
        case MSG_RECORD_GOOD:
        {
            m_lineList.add( new KString( "Recording started." ) ); break;
        }
        case MSG_RECORD_BAD:
        {
            m_lineList.add( new KString( "Error in recording!" ) ); break;
        }
        case MSG_RECORD_STOP:
        {
            m_lineList.add( new KString( "Recording stopped!" ) ); break;
        }
        case MSG_PLAY_GOOD:
        {
            m_lineList.add( new KString( "Playback started." ) ); break;
        }
        case MSG_PLAY_BAD:
        {
            m_lineList.add( new KString( "Error in playback!" ) ); break;
        }
        case MSG_PLAY_STOP:
        {
            m_lineList.add( new KString( "Playback stopped!" ) ); break;
        }
        case MSG_NET_INUSE:
        {
            m_lineList.add( new KString( "Net Game already exists!" ) ); break;
        }
        case MSG_NET_NOSERV:
        {
            m_lineList.add( new KString( "Net Game not found!" ) ); break;
        }
        case MSG_NET_STOPPED:
        {
            m_lineList.add( new KString( "Net Game Terminated!" ) ); break;
        }
        case MSG_NET_LEFT:
        {
            m_lineList.add( new KString( "Disconnect from Net Game!" ) ); break;
        }
        case MSG_NET_MISSING:
        {
            m_lineList.add( new KString( "Missing Net Game Files!" ) ); break;
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the message state.  The base method
//      is used to display the lines and allow the user to scroll, this
//      is only responsible for exiting the state.
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStateMessage::execute(
    ColourTable* pColourTable 
)
{
    //  If the message has changed then set the new message for display.
    if( m_eMessage != sm_eMessage )
    {
        m_eMessage = sm_eMessage;
        setMessage( );
    }

    //  Call the base class.
    SSStateList::execute( pColourTable );

    //  Allow the user to scroll the lines.
    SSStateList::checkForScroll( );

    //  Allow for the standard keys (like rotate, flip, etc.).
    SSStateBase::checkKeys( DEF_NONE );

    //  Exit?  Go back to the previous state.
    if( m_pKeyboard->switchOn( Keyboard::KEY__ESC ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ESC );
        m_pSelectScreen->setState( SelectScreen::STATE_PREVIOUS );
    }
    else
    //  Exit?  Go back to the previous state.
    if( m_pKeyboard->switchOn( Keyboard::KEY__ENTER ) )
    {
        m_pKeyboard->waitUntilOff( Keyboard::KEY__ENTER );
        m_pSelectScreen->setState( SelectScreen::STATE_PREVIOUS );
    }

    //  The game should not be started yet.
    return( SSS_CONTINUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Exit
//          RET Exit
//          UP  Up Line
//          DN  Down Line
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStateMessage::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpText[ ] = 
    {
        "ESC Exit",
        "RET Exit",
        "UP  Up Line",
        "DN  Down Line",
        NULL
    };

    //  Draw the text.
    drawHelp( pBitmap, eFont, pColourTable, ppstrHelpText );
}
