///////////////////////////////////////////////////////////////////////////////
//
//  File:    ssspause.cpp
//
//  Class:   SSStatePause
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is the selection screen state responsible for showing a
//      paused display of the current game while allowing options to be set.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "ssspause.h"
#include "select.h"
#include "keyb.h"
#include "clock.h"
#include "game.h"
#include "screen.h"
#include "bitmap.h"
#include "ctable.h"
#include "clip.h"


///////////////////////////////////////////////////////////////////////////////
//  File Constants.
///////////////////////////////////////////////////////////////////////////////
//  Some strings.
static const char* PAUSE_MESSAGE     = "Paused!";


//  The screen attributes.
static const DWord TEXT_WIDTH          = strlen( PAUSE_MESSAGE ) * 16;
static const DWord TEXT_HEIGHT         = 32;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: SSStatePause
//
//  Description:
//
//      This is the main constructor for the selection screen pause
//      state object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pSelectScreen (input)
//          The selection screen the state belongs to.
//
//      pCanvas (input)
//          The canvas used by the selection screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStatePause::SSStatePause(
    const KString& iName,
    SelectScreen*  pSelectScreen,
    Canvas*        pCanvas
)
:
    SSStateBase     ( iName, pSelectScreen, pCanvas ),
    m_pClock        ( NULL )
{
    //  Get the clock.
    m_pClock = Replay::s_instance( ).getClock( );
    ASSERT( m_pClock != NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~SSStatePause
//
//  Description:
//
//      This is the destructor for the selection screen pause state 
//      object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
SSStatePause::~SSStatePause(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the select screen pause state object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
SSStatePause::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "SSStatePause" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This is used to execute the pause state.  We simply check for
//      game-related keys being pressed (e.g. F1 for Game Info, etc.).
//
//  Parameters:
//
//      pColourTable (input)
//          The colour table to draw with (ignored).
//
//  Returns:
//
//      An action indicating what the selection screen should do.
//
///////////////////////////////////////////////////////////////////////////////
SSStateBase::Action
SSStatePause::execute(
    ColourTable* pColourTable
)
{
    //  We should have a selection screen and a game specified by now.
    ASSERT( m_pSelectScreen             != NULL );
    ASSERT( m_pGameInfo                 != NULL );
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
 
    //  So things don't get overwhelming, we wait for a frame.
    m_pClock->frameWait( );
        
    //  Create the bounding box for the text we are going to draw.
    m_clipping.m_nMinX = ( getScreen( )->getWidth( ) - TEXT_WIDTH ) / 2;
    m_clipping.m_nMaxX = m_clipping.m_nMinX + TEXT_WIDTH;
    m_clipping.m_nMinY = ( getScreen( )->getHeight( ) - TEXT_HEIGHT ) / 2;
    m_clipping.m_nMaxY = m_clipping.m_nMinY + TEXT_HEIGHT;

    //  Draw the message over top of the game screen.
    m_pCanvas->drawText(
        PAUSE_MESSAGE,
        getScreen( ),
        Canvas::FONT_LARGE,
        &m_clipping,
        *pColourTable,
        TRUE
    );
    
    //  Check the default keys.
    return( checkKeys( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: fillHelp
//
//  Description:
//
//      This is used to fill in the help area for this state.  The following
//      keys are allowed during this state:
//          ESC Main Menu
//          UP  Prev Game
//          DN  Next Game
//          RET Run Game
//
//  Parameters:
//
//      pBitmap (input)
//          The help bitmap.
//
//      eFont (input)
//          The font to draw with.
//
//      pColourTable (input)
//          The colour table to draw with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
SSStatePause::fillHelp(
    Bitmap*      pBitmap,
    Canvas::Font eFont,
    ColourTable* pColourTable 
)
{
    //  Check the arguments.
    ASSERT( pBitmap       != NULL );
    ASSERT( pColourTable  != NULL );

    //  The help text.
    static char* ppstrHelpText[ ] = 
    {
        "ESC Main Menu",
        "UP  Prev Game",
        "DN  Next Game",
        "Ret Run  Game",
        NULL
    };

    //  Draw the text.
    drawHelp( pBitmap, eFont, pColourTable, ppstrHelpText );
}
        


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getScreen
//
//  Description:
//
//      This returns the bitmap of the current game currently paused.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      A pointer to the screen bitmap of the state.
//
///////////////////////////////////////////////////////////////////////////////
Bitmap* 
SSStatePause::getScreen ( 
)
{
    //  Return the screen of the current game.
    ASSERT( m_pSelectScreen->getGame( ) != NULL );
    return( m_pSelectScreen->getGame( )->getScreen( )->getBitmap( ) );
} 
