///////////////////////////////////////////////////////////////////////////////
//
//  File:    ay8910.cpp
//
//  Class:   AY8910
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is responsible for emulating the AY-3-8910/2/3.
//      This chip is a register oriented Programmable Sound Generator (PSG)
//      from General Instruments.
//
//      The only difference between the three chips is that the 8910 has
//      I/O ports A&B, the 8912 has I/O port A and the 8913 has no I/O ports.
//
//      Ideally to be completely object oriented, this class would be
//      broken up into the following hierarchy:
//
//          AY8910 - AY8912 - AY8913 - SoundDevice - RepBase
//
//      However, then the update() member of the AY8910 (the most commonly
//      used version of the chip) would have to traverse up to AY8913
//      for most of its functionality.  This is a couple of extra
//      function calls and a hit I don't want to take at the moment.
//
//      Thanks go to Ville Hallik/Michael Cuddy for the original code and
//      the Mame team for modifications.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "ay8910.h"
#include "replay.h"
#include "sound.h"
#include "clock.h"
#include "sample.h"
#include "game.h"
#include "schedule.h"
#include "appfile.h"



///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
//  The scaling factor used to allow us to work in integers.
const DWord AY8910::sm_dwScale   = 0x10000;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: AY8910
//
//  Description:
//
//      This is the main constructor for a AY-3-8910 chip object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwChipClock (input)
//          The clock speed of the chip.
//
//      dwChipVolume (input)
//          The volume setting for the chip.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
AY8910::AY8910(
    const KString& iName,
    const DWord    dwChipClock,
    const DWord    dwChipVolume
)
:
    SoundDevice       ( iName ),
    m_dwChipClock     ( dwChipClock ),
    m_dwChipVolume    ( dwChipVolume ),
    m_bRegister       ( 0 ),
    m_pSample         ( NULL ),
    m_nUpdatePos      ( 0 ),
    m_dwUpdateStep    ( 0 ),
    m_nPeriodA        ( 0 ),
    m_nPeriodB        ( 0 ),
    m_nPeriodC        ( 0 ),
    m_nPeriodN        ( 0 ),
    m_nPeriodE        ( 0 ),
    m_nCountA         ( 0 ),
    m_nCountB         ( 0 ),
    m_nCountC         ( 0 ),
    m_nCountN         ( 0 ),
    m_nCountE         ( 0 ),
    m_dwVolumeA       ( 0 ),
    m_dwVolumeB       ( 0 ),
    m_dwVolumeC       ( 0 ),
    m_dwVolumeE       ( 0 ),
    m_bEnvelopeA      ( 0 ),
    m_bEnvelopeB      ( 0 ),
    m_bEnvelopeC      ( 0 ),
    m_bOutputA        ( 0 ),
    m_bOutputB        ( 0 ),
    m_bOutputC        ( 0 ),
    m_bOutputN        ( 0xff ),
    m_nCountEnv       ( 0 ),
    m_bHold           ( 0 ),
    m_bAlternate      ( 0 ),
    m_bAttack         ( 0 ),
    m_bHolding        ( 0 ),
    m_nRNG            ( 1 ),
    m_pReadHandlerA   ( NULL ),
    m_pReadHandlerB   ( NULL ),
    m_pWriteHandlerA  ( NULL ),
    m_pWriteHandlerB  ( NULL )
{
    //  Chip volume shouldn't exceed 16 bits (8 for volume, 8 for gain).
    ASSERT( m_dwChipVolume < 0x10000 );

    //  Create the sample where the output will be placed.
    m_pSample = new Sample( 
        "PSG", 
        m_pSound->getSampleRate( ) / 
            Replay::s_instance( ).getClock( )->getFPS( )
    );

    //  Calculate the emulation rate.
    m_dwEmulationRate = 
        m_pSample->getSize( ) * Replay::s_instance( ).getClock( )->getFPS( );

    //  Set the sample parameters.
    m_pSample->setVolume( m_dwChipVolume & 0xff ); 
    m_pSample->setFrequency( m_dwEmulationRate );
    m_pSample->setResolution( 8 );
    m_pSample->setChannel( m_pSound->getChannel( ) );

    //  Set the clock.
    setClock( m_dwChipClock, m_dwEmulationRate );
 
    //  Set the output gain of this chip.
    setGain( );

    //  Initialize the registers.
    for( Byte bI = 0 ; bI < NUM_8910_REGISTERS ; bI += 1 )
    {
        m_abRegisters[ bI ] = 0;
    }
    for( Byte bI = 0 ; bI < PORT_A ; bI += 1 )
    {
        m_bRegister = bI; 
        doWriteRegister( 0x00 );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~AY8910
//
//  Description:
//
//      This is the destructor for a AY8910 object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
AY8910::~AY8910(
)
{
    //  Free up the resources.
    ASSERT( m_pSample != NULL );
    delete m_pSample;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the AY-3-8910 class.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
AY8910::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "AY8910" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the 8910 to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the input to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
AY8910::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    SoundDevice::save( pSaveFile );

    //  Allow the sample to save itself.
    m_pSample->save( pSaveFile );

    //  Save the member data that keeps the state.
    pSaveFile->clearTotal( );
    pSaveFile->write( &m_bRegister,              sizeof( m_bRegister ) );
    pSaveFile->write( &( m_abRegisters[ 0 ] ),   sizeof( m_abRegisters ) );
    pSaveFile->write( ( Byte* )&m_nUpdatePos,    sizeof( m_nUpdatePos ) );
    pSaveFile->write( ( Byte* )&m_dwUpdateStep,  sizeof( m_dwUpdateStep ) );
    pSaveFile->write( ( Byte* )&m_nPeriodA,      sizeof( m_nPeriodA ) );
    pSaveFile->write( ( Byte* )&m_nPeriodB,      sizeof( m_nPeriodB ) );
    pSaveFile->write( ( Byte* )&m_nPeriodC,      sizeof( m_nPeriodC ) );
    pSaveFile->write( ( Byte* )&m_nPeriodN,      sizeof( m_nPeriodN ) );
    pSaveFile->write( ( Byte* )&m_nPeriodE,      sizeof( m_nPeriodE ) );
    pSaveFile->write( ( Byte* )&m_nCountA,       sizeof( m_nCountA ) );
    pSaveFile->write( ( Byte* )&m_nCountB,       sizeof( m_nCountB ) );
    pSaveFile->write( ( Byte* )&m_nCountC,       sizeof( m_nCountC ) );
    pSaveFile->write( ( Byte* )&m_nCountN,       sizeof( m_nCountN ) );
    pSaveFile->write( ( Byte* )&m_nCountE,       sizeof( m_nCountE ) );
    pSaveFile->write( ( Byte* )&m_dwVolumeA,     sizeof( m_dwVolumeA ) );
    pSaveFile->write( ( Byte* )&m_dwVolumeB,     sizeof( m_dwVolumeB ) );
    pSaveFile->write( ( Byte* )&m_dwVolumeC,     sizeof( m_dwVolumeC ) );
    pSaveFile->write( ( Byte* )&m_dwVolumeE,     sizeof( m_dwVolumeE ) );
    pSaveFile->write( &m_bEnvelopeA,             sizeof( m_bEnvelopeA ) );
    pSaveFile->write( &m_bEnvelopeB,             sizeof( m_bEnvelopeB ) );
    pSaveFile->write( &m_bEnvelopeC,             sizeof( m_bEnvelopeC ) );
    pSaveFile->write( &m_bOutputA,               sizeof( m_bOutputA ) );
    pSaveFile->write( &m_bOutputB,               sizeof( m_bOutputB ) );
    pSaveFile->write( &m_bOutputC,               sizeof( m_bOutputC ) );
    pSaveFile->write( &m_bOutputN,               sizeof( m_bOutputN ) );
    pSaveFile->write( ( Byte* )&m_nCountEnv,     sizeof( m_nCountEnv ) );
    pSaveFile->write( &m_bHold,                  sizeof( m_bHold ) );
    pSaveFile->write( &m_bAlternate,             sizeof( m_bAlternate ) );
    pSaveFile->write( &m_bAttack,                sizeof( m_bAttack ) );
    pSaveFile->write( &m_bHolding,               sizeof( m_bHolding ) );
    pSaveFile->write( ( Byte* )&m_nRNG,          sizeof( m_nRNG ) );
    pSaveFile->write( 
        ( Byte* )&( m_adwVolumeTable[ 0 ] ),     sizeof( m_adwVolumeTable ) 
    );

    //  Was it saved successfully?
    return( pSaveFile->total( ) != 225 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the AY8910 from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the input from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
AY8910::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    SoundDevice::load( pLoadFile );

    //  Allow the sample to load itself.
    m_pSample->load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->clearTotal( );
    pLoadFile->clearTotal( );
    pLoadFile->read( &m_bRegister,              sizeof( m_bRegister ) );
    pLoadFile->read( &( m_abRegisters[ 0 ] ),   sizeof( m_abRegisters ) );
    pLoadFile->read( ( Byte* )&m_nUpdatePos,    sizeof( m_nUpdatePos ) );
    pLoadFile->read( ( Byte* )&m_dwUpdateStep,  sizeof( m_dwUpdateStep ) );
    pLoadFile->read( ( Byte* )&m_nPeriodA,      sizeof( m_nPeriodA ) );
    pLoadFile->read( ( Byte* )&m_nPeriodB,      sizeof( m_nPeriodB ) );
    pLoadFile->read( ( Byte* )&m_nPeriodC,      sizeof( m_nPeriodC ) );
    pLoadFile->read( ( Byte* )&m_nPeriodN,      sizeof( m_nPeriodN ) );
    pLoadFile->read( ( Byte* )&m_nPeriodE,      sizeof( m_nPeriodE ) );
    pLoadFile->read( ( Byte* )&m_nCountA,       sizeof( m_nCountA ) );
    pLoadFile->read( ( Byte* )&m_nCountB,       sizeof( m_nCountB ) );
    pLoadFile->read( ( Byte* )&m_nCountC,       sizeof( m_nCountC ) );
    pLoadFile->read( ( Byte* )&m_nCountN,       sizeof( m_nCountN ) );
    pLoadFile->read( ( Byte* )&m_nCountE,       sizeof( m_nCountE ) );
    pLoadFile->read( ( Byte* )&m_dwVolumeA,     sizeof( m_dwVolumeA ) );
    pLoadFile->read( ( Byte* )&m_dwVolumeB,     sizeof( m_dwVolumeB ) );
    pLoadFile->read( ( Byte* )&m_dwVolumeC,     sizeof( m_dwVolumeC ) );
    pLoadFile->read( ( Byte* )&m_dwVolumeE,     sizeof( m_dwVolumeE ) );
    pLoadFile->read( &m_bEnvelopeA,             sizeof( m_bEnvelopeA ) );
    pLoadFile->read( &m_bEnvelopeB,             sizeof( m_bEnvelopeB ) );
    pLoadFile->read( &m_bEnvelopeC,             sizeof( m_bEnvelopeC ) );
    pLoadFile->read( &m_bOutputA,               sizeof( m_bOutputA ) );
    pLoadFile->read( &m_bOutputB,               sizeof( m_bOutputB ) );
    pLoadFile->read( &m_bOutputC,               sizeof( m_bOutputC ) );
    pLoadFile->read( &m_bOutputN,               sizeof( m_bOutputN ) );
    pLoadFile->read( ( Byte* )&m_nCountEnv,     sizeof( m_nCountEnv ) );
    pLoadFile->read( &m_bHold,                  sizeof( m_bHold ) );
    pLoadFile->read( &m_bAlternate,             sizeof( m_bAlternate ) );
    pLoadFile->read( &m_bAttack,                sizeof( m_bAttack ) );
    pLoadFile->read( &m_bHolding,               sizeof( m_bHolding ) );
    pLoadFile->read( ( Byte* )&m_nRNG,          sizeof( m_nRNG ) );
    pLoadFile->read( 
        ( Byte* )&( m_adwVolumeTable[ 0 ] ),     sizeof( m_adwVolumeTable ) 
    );

    //  Was it loaded successfully?
    return( pLoadFile->total( ) != 225 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called to allow the AY8910 to update itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AY8910::update(
)
{
    //  Since the video frame is complete, we update until the end of the
    //  buffer.
    updateBuffer( TRUE );

    //  Reset the update position.
    m_nUpdatePos = 0;

    //  Play the sample.
    m_pSound->playStreamed( m_pSample );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateBuffer
//
//  Description:
//
//      This member is called to allow the AY8910 to update a portion of
//      its audio stream.
//
//  Parameters:
//
//      bToEnd (input)
//          If TRUE the complete buffer will be updated, otherwise, the buffer
//          will be updated to the point that represents the same percentage
//          complete as the current frame progress.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AY8910::updateBuffer(
    const Byte bToEnd
)
{
    //  A pointer to the sample buffer.
    Byte* pBuffer = m_pSample->getBuffer( ) + m_nUpdatePos;

    //  The length to update.  This is the number of bytes since the end of
    //  the last update.
    int32 nLength;
    
    //  Calculate the length to update.
    if( bToEnd )
    {
        nLength = m_pSample->getSize( ) - m_nUpdatePos;
    }
    else
    {
        //  This position is the position we want to fill the sample buffer to.
        nLength = ( int32 )
            Game::sm_pGame->getScheduler( )->getFramePercent( ) *
            m_pSample->getSize( ) - 
            m_nUpdatePos;
        if( ( DWord )( m_nUpdatePos + nLength ) > m_pSample->getSize( ) )
        {
            nLength = m_pSample->getSize( ) - m_nUpdatePos;
        }
    }
    if( 
        ( nLength < 10 ) && 
        ( ( DWord )( m_nUpdatePos + nLength ) < m_pSample->getSize( ) )
    )
    {
        return;
    }
    m_nUpdatePos += nLength;

    //  We need to mix each of the three tone generators with the noise 
    //  generator before going to the DAC.  The mixing formula is:
    //      ( ToneOn | ToneDisable ) & ( NoiseOn | NoiseDisable )
    //  Note that if both tone and noise are disabled, the output will be 1.
    if( m_abRegisters[ ENABLE ] & 0x01 )
    {
        m_nCountA  = nLength * sm_dwScale + 1;
        m_bOutputA = 1;
    }
    if( m_abRegisters[ ENABLE ] & 0x02 )
    {
        m_nCountB  = nLength * sm_dwScale + 1;
        m_bOutputB = 1;
    }
    if( m_abRegisters[ ENABLE ] & 0x04 )
    {
        m_nCountC  = nLength * sm_dwScale + 1;
        m_bOutputC = 1;
    }
    if( ( m_abRegisters[ ENABLE ] & 0x38 ) == 0x38 )
    {
        m_nCountN = nLength * sm_dwScale + 1;
    }
    Byte bOutputN = m_bOutputN | m_abRegisters[ ENABLE ];


    //  Loop until we have filled the desired length.
    while( nLength )
    {
        //  Some volumes.
        int32 nVolumeA = 0;
        int32 nVolumeB = 0;
        int32 nVolumeC = 0;

        //  The amount left.
        int32 nLeft = sm_dwScale;

        do
        {
            //  The next event.
            int32 nNextEvent;

            if( m_nCountN < nLeft )
            {
                nNextEvent = m_nCountN;
            }
            else
            {
                nNextEvent = nLeft;
            }

            if( bOutputN & 0x08 )
            {
                if( m_bOutputA )
                {
                    nVolumeA += m_nCountA;
                }
                m_nCountA -= nNextEvent;
 
                //  m_nPeriodA is the half period of the square wave.  Here,
                //  in each loop m_nPeriodA is added twice, so that at the
                //  end of the loop the square wave is in the same state (0,1)
                //  as it was at the start.  nVolumeA is also incremented by
                //  m_nPeriodA since the wave has been 1 exactly half of the
                //  time, regardless of the initial position.  If we exit the
                //  loop in the middle, m_bOutputA has to be inverted and
                //  nVolumeA incremented only if the exit state of the square
                //  wave is 1.
                while( m_nCountA <= 0 )
                {
                    m_nCountA += m_nPeriodA;
                    if( m_nCountA > 0 )
                    {
                        m_bOutputA ^= 1;
                        if( m_bOutputA )
                        {
                            nVolumeA += m_nPeriodA;
                        }
                        break;
                    }
                    m_nCountA += m_nPeriodA;
                    nVolumeA += m_nPeriodA;
                }
                if( m_bOutputA )
                {
                    nVolumeA -= m_nCountA;
                }
            } 
            else
            {
                m_nCountA -= nNextEvent;
                while( m_nCountA <= 0 )
                {
                    m_nCountA += m_nPeriodA;
                    if( m_nCountA > 0 )
                    {
                        m_bOutputA ^= 1;
                        break;
                    }
                    m_nCountA += m_nPeriodA;
                }
            }

            if( bOutputN & 0x10 )
            {
                if( m_bOutputB )
                {
                    nVolumeB += m_nCountB;
                }
                m_nCountB -= nNextEvent;
                while( m_nCountB <= 0 )
                {
                    m_nCountB += m_nPeriodB;
                    if( m_nCountB > 0 )
                    {
                        m_bOutputB ^= 1;
                        if( m_bOutputB )
                        {
                            nVolumeB += m_nPeriodB;
                        }
                        break;
                    }
                    m_nCountB += m_nPeriodB;
                    nVolumeB += m_nPeriodB;
                }
                if( m_bOutputB )
                {
                    nVolumeB -= m_nCountB;
                }
            } 
            else
            {
                m_nCountB -= nNextEvent;
                while( m_nCountB <= 0 )
                {
                    m_nCountB += m_nPeriodB;
                    if( m_nCountB > 0 )
                    {
                        m_bOutputB ^= 1;
                        break;
                    }
                    m_nCountB += m_nPeriodB;
                }
            }

            if( bOutputN & 0x20 )
            {
                if( m_bOutputC )
                {
                    nVolumeC += m_nCountC;
                }
                m_nCountC -= nNextEvent;
                while( m_nCountC <= 0 )
                {
                    m_nCountC += m_nPeriodC;
                    if( m_nCountC > 0 )
                    {
                        m_bOutputC ^= 1;
                        if( m_bOutputC )
                        {
                            nVolumeC += m_nPeriodC;
                        }
                        break;
                    }
                    m_nCountC += m_nPeriodC;
                    nVolumeC += m_nPeriodC;
                }
                if( m_bOutputC )
                {
                    nVolumeC -= m_nCountC;
                }
            } 
            else
            {
                m_nCountC -= nNextEvent;
                while( m_nCountC <= 0 )
                {
                    m_nCountC += m_nPeriodC;
                    if( m_nCountC > 0 )
                    {
                        m_bOutputC ^= 1;
                        break;
                    }
                    m_nCountC += m_nPeriodC;
                }
            }

            m_nCountN -= nNextEvent;
            if( m_nCountN <= 0 )
            {
                //  Is the noise output going to change?
                if( ( m_nRNG + 1 ) & 0x02 )
                {
                    m_bOutputN = ~m_bOutputN;
                    bOutputN = ( m_bOutputN | m_abRegisters[ ENABLE ] );
                }

                //  The random number generator of the 8910 is a 17-bit
                //  shift register.  The input to the shift register is
                //  bit0 XOR bit2 (bit0 is the output).
                //  The following is a fast way to compute 
                //  bit 17 = bit0^bit2.  Instead of doing all the logic
                //  operations, we only check bit0, relying on the fact
                //  that after two shifts of the register, what now is bit 2
                //  will become bit 0, and will invert, if necessary,
                //  bit 16, which previously was bit 18. 
                if( m_nRNG & 0x01 )
                {
                    m_nRNG ^= 0x28000;
                }
                m_nRNG >>= 1;
                m_nCountN += m_nPeriodN;
            }

            nLeft -= nNextEvent;
        }
        while( nLeft > 0 );

        //  Update the envelope.
        if( m_bHolding == 0 )
        {
            m_nCountE -= sm_dwScale;
            if( m_nCountE <= 0 )
            {
                do
                {
                    m_nCountEnv -= 1;
                    if( m_nCountEnv < 0 )
                    {
                        m_nCountEnv = 0x1f;

                        if( m_bAlternate )
                        {
                            m_bAttack ^= 0x1f;
                        }
 
                        if( m_bHold != 0 )
                        {
                            m_bHolding = 1;
                            m_nCountEnv = 0;
                        }
                    }

                    m_dwVolumeE = m_adwVolumeTable[ m_nCountEnv ^ m_bAttack ];
                    if( m_bEnvelopeA )
                    {
                        m_dwVolumeA = m_dwVolumeE;
                    }
                    if( m_bEnvelopeB )
                    {
                        m_dwVolumeB = m_dwVolumeE;
                    }
                    if( m_bEnvelopeC )
                    {
                        m_dwVolumeC = m_dwVolumeE;
                    }

                    m_nCountE += m_nPeriodE;
                } 
                while( ( m_nCountE <= 0 ) && ( m_bHolding == 0 ) );
            }
        }

        nVolumeA *= m_dwVolumeA;
        nVolumeB *= m_dwVolumeB;
        nVolumeC *= m_dwVolumeC;

        *pBuffer = ( nVolumeA + nVolumeB + nVolumeC ) / ( sm_dwScale * 256 );

        pBuffer += 1;

        nLength -= 1;
    } 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setGain
//
//  Description:
//
//      This member is called to set the gain for the chip.  Gain is
//      expressed in 0.2dB * the gain value.  Sounds that already play at
//      full volume will not be affected by gain.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AY8910::setGain(
)
{
    //  The maximum output.
    static const double fMax = 0x7fff / 3;

    //  The step increment (based on 0.2dB per step).
    static const double fIncrement = 1.023292992;

    //  The step decrement (based on 1.5dB per step).
    static const double fDecrement = 1.188502227;

    //  The output.
    double fOutput;


    //  The gain is based on the volume of the chip.
    int nGain = ( m_dwChipVolume >> 8 ) & 0xff;


    //  Calculate the starting output value by increasing 0.2dB per step.
    fOutput = fMax;
    for( ; nGain > 0 ; nGain -= 1 )
    {
        fOutput *= fIncrement;
    }

    //  Now we fill in the volume to voltage conversion table.  We use 32
    //  levels even though the AY-3-891x only has 16 levels so that derived
    //  classes that use the upper 16 for envelope generation will be
    //  satisfied.
    for( DWord dwI = 31 ; dwI > 0 ; dwI -= 1 )
    {
        //  Limit the volume if it's greater than the max, otherwise use
        //  the output value.
        m_adwVolumeTable[ dwI ] = 
            ( DWord )( ( fOutput > fMax ) ? fMax : fOutput );
    
        //  Trim the output for the next value.
        fOutput /= fDecrement;
    }
    m_adwVolumeTable[ 0 ] = 0;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: doWriteRegister
//
//  Description:
//
//      This member is called to write a specified value to a specified
//      register.  This is for primary control of the chip.
//
//  Parameters:
//
//      bValue (input)
//          The value to write.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AY8910::doWriteRegister(
    const Byte bValue
)
{
    //  Used as a temporary holding spot.
    int32 nTemp;


    ASSERT( m_bRegister < NUM_8910_REGISTERS );

    //  Cache the value.
    m_abRegisters[ m_bRegister ] = bValue; 

    //  OK, process the value.
    switch( m_bRegister )
    {
        case A_COARSE_PITCH:
        {
            m_abRegisters[ A_COARSE_PITCH ] &= 0x0f;

            //  Fall through.
        }
        case A_FINE_PITCH:
        {
            nTemp = m_nPeriodA;

            m_nPeriodA = ( 
                ( m_abRegisters[ A_COARSE_PITCH ] << 8 ) + 
                m_abRegisters[ A_FINE_PITCH ]
            ) * m_dwUpdateStep;
            if( m_nPeriodA == 0 )
            {
                m_nPeriodA = m_dwUpdateStep;
            }
            
            m_nCountA += m_nPeriodA - nTemp;
            if( m_nCountA <= 0 )
            {
                m_nCountA = 1;
            }

            break;
        }

        case B_COARSE_PITCH:
        {
            m_abRegisters[ B_COARSE_PITCH ] &= 0x0f;

            //  Fall through.
        }
        case B_FINE_PITCH:
        {
            nTemp = m_nPeriodB;

            m_nPeriodB = ( 
                ( m_abRegisters[ B_COARSE_PITCH ] << 8 ) + 
                m_abRegisters[ B_FINE_PITCH ]
            ) * m_dwUpdateStep;
            if( m_nPeriodB == 0 )
            {
                m_nPeriodB = m_dwUpdateStep;
            }
            
            m_nCountB += m_nPeriodB - nTemp;
            if( m_nCountB <= 0 )
            {
                m_nCountB = 1;
            }

            break;
        }

        case C_COARSE_PITCH:
        {
            m_abRegisters[ C_COARSE_PITCH ] &= 0x0f;

            //  Fall through.
        }
        case C_FINE_PITCH:
        {
            nTemp = m_nPeriodC;

            m_nPeriodC = ( 
                ( m_abRegisters[ C_COARSE_PITCH ] << 8 ) + 
                m_abRegisters[ C_FINE_PITCH ]
            ) * m_dwUpdateStep;
            if( m_nPeriodC == 0 )
            {
                m_nPeriodC = m_dwUpdateStep;
            }
            
            m_nCountC += m_nPeriodC - nTemp;
            if( m_nCountC <= 0 )
            {
                m_nCountC = 1;
            }

            break;
        }

        case NOISE_PITCH:
        {
            m_abRegisters[ NOISE_PITCH ] &= 0x1f;

            nTemp = m_nPeriodN;

            m_nPeriodN = m_abRegisters[ NOISE_PITCH ] * m_dwUpdateStep; 
            if( m_nPeriodN == 0 )
            {
                m_nPeriodN = m_dwUpdateStep;
            }

            m_nCountN += m_nPeriodN - nTemp;
            if( m_nCountN <= 0 )
            {
                m_nCountN = 1;
            }

            break;
        }

        case A_VOLUME:
        {
            m_abRegisters[ A_VOLUME ] &= 0x1f;

            m_bEnvelopeA = m_abRegisters[ A_VOLUME ] & 0x10;
            m_dwVolumeA = m_bEnvelopeA ?  m_dwVolumeE : m_adwVolumeTable[ 
                m_abRegisters[ A_VOLUME ] ? 
                    m_abRegisters[ A_VOLUME ] * 2 + 1 : 0 
            ];
           
            break;
        }

        case B_VOLUME:
        {
            m_abRegisters[ B_VOLUME ] &= 0x1f;

            m_bEnvelopeB = m_abRegisters[ B_VOLUME ] & 0x10;
            m_dwVolumeB = m_bEnvelopeB ?  m_dwVolumeE : m_adwVolumeTable[ 
                m_abRegisters[ B_VOLUME ] ? 
                    m_abRegisters[ B_VOLUME ] * 2 + 1 : 0 
            ];
           
            break;
        }

        case C_VOLUME:
        {
            m_abRegisters[ C_VOLUME ] &= 0x1f;

            m_bEnvelopeC = m_abRegisters[ C_VOLUME ] & 0x10;
            m_dwVolumeC = m_bEnvelopeC ?  m_dwVolumeE : m_adwVolumeTable[ 
                m_abRegisters[ C_VOLUME ] ? 
                    m_abRegisters[ C_VOLUME ] * 2 + 1 : 0 
            ];
           
            break;
        }
             
        case E_FINE_DURATION:
        case E_COARSE_DURATION:
        {
            nTemp = m_nPeriodE;

            m_nPeriodE = ( 
                ( m_abRegisters[ E_COARSE_DURATION ] << 8 ) + 
                m_abRegisters[ E_FINE_DURATION ]
            ) * m_dwUpdateStep;
            if( m_nPeriodE == 0 )
            {
                m_nPeriodE = m_dwUpdateStep / 2;
            }
            
            m_nCountE += m_nPeriodE - nTemp;
            if( m_nCountE <= 0 )
            {
                m_nCountE = 1;
            }

            break;
        }

        case E_SHAPE:
        {
            //  The following shapes are available for the envelope:
            //
            //  0 0 x x    \_____________
            //  0 1 x x    /|____________
            //  1 0 0 0    \|\|\|\|\|\|\|
            //  1 0 0 1    \_____________
            //  1 0 1 0    \/\/\/\/\/\/\/
            //  1 0 1 1    \|------------
            //  1 1 0 0    /|/|/|/|/|/|/|
            //  1 1 0 1    /-------------
            //  1 1 1 0    /\/\/\/\/\/\/\  
            //  1 1 1 1    /|____________

            //  We use 32 steps instead of the normal 16 steps of
            //  the 8910 to handle derived classes.  
            m_abRegisters[ E_SHAPE ] &= 0x0f;
            m_bAttack = ( m_abRegisters[ E_SHAPE ] & 0x04 ) ? 0x1f : 0x00;
            if( ( m_abRegisters[ E_SHAPE ] & 0x08 ) == 0 )
            {
                m_bHold      = 1;
                m_bAlternate = m_bAttack;
            }
            else
            {
                m_bHold      = m_abRegisters[ E_SHAPE ] & 0x01;
                m_bAlternate = m_abRegisters[ E_SHAPE ] & 0x02;
            }

            m_nCountE   = m_nPeriodE;
            m_nCountEnv = 0x1f;
            m_bHolding  = 0;
            m_dwVolumeE = m_adwVolumeTable[ m_nCountEnv ^ m_bAttack ];
            if( m_bEnvelopeA )
            {
                m_dwVolumeA = m_dwVolumeE;
            }
            if( m_bEnvelopeB )
            {
                m_dwVolumeB = m_dwVolumeE;
            }
            if( m_bEnvelopeC )
            {
                m_dwVolumeC = m_dwVolumeE;
            }

            break;
        }
 
        case PORT_A:
        {
            CHECK0( 
                ( m_abRegisters[ ENABLE ] & 0x40 ) != 0,
                "Write to AY8910:A not expected."
            );

            if( m_pWriteHandlerA != NULL )
            {
                ( *( m_pWriteHandlerA->getHandler( ) ) )( 
                    0, bValue, m_pWriteHandlerA 
                );
            }
 
            break;
        }

        case PORT_B:
        {
            CHECK0( 
                ( m_abRegisters[ ENABLE ] & 0x80 ) != 0,
                "Write to AY8910:B not expected."
            );

            if( m_pWriteHandlerA != NULL )
            {
                ( *( m_pWriteHandlerB->getHandler( ) ) )( 
                    1, bValue, m_pWriteHandlerB
                );
            }

            break;
        }

        default:
        {
            break;
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readRegister
//
//  Description:
//
//      This member is called to read the value of a specified register.
//      This is for primary control of the chip.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
AY8910::readRegister(
)
{
    ASSERT( m_bRegister < NUM_8910_REGISTERS );
    
    //  OK, process the value.
    switch( m_bRegister )
    {
        case PORT_A:
        {
            ASSERT( ( m_abRegisters[ ENABLE ] & 0x40 ) == 0 );

            if( m_pReadHandlerA != NULL )
            {
                m_abRegisters[ PORT_A ] = 
                    ( *( m_pReadHandlerA->getHandler( ) ) )( 
                        0, m_pReadHandlerA 
                    );
            }
            else
            {
                m_abRegisters[ m_bRegister ] = 0x00;
            }
 
            break;
        }

        case PORT_B:
        {
            ASSERT( ( m_abRegisters[ ENABLE ] & 0x80 ) == 0 );

            if( m_pReadHandlerA != NULL )
            {
                m_abRegisters[ PORT_B ] = 
                    ( *( m_pReadHandlerB->getHandler( ) ) )( 
                        1, m_pReadHandlerB 
                    );
            }
            else
            {
                m_abRegisters[ m_bRegister ] = 0x00;
            }
 
            break;
        }

        default:
        {
            break;
        }
    }
             
    //  Return the value of the register.
    return( m_abRegisters[ m_bRegister ] );
}
