///////////////////////////////////////////////////////////////////////////////
//
//  File:       ay8910.h
//
//  Class:      AY8910 - Concrete
//  
//  Hierarchy:  AY8910 - SoundDevice - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class is responsible for emulating the AY-3-8910/2/3.
//      This chip is a register oriented Programmable Sound Generator (PSG)
//      from General Instruments.
//
//      The only difference between the three chips is that the 8910 has
//      I/O ports A&B, the 8912 has I/O port A and the 8913 has no I/O ports.
//
//      Ideally to be completely object oriented, this class would be 
//      broken up into the following hierarchy:
//
//          AY8910 - AY8912 - AY8913 - SoundDevice - RepBase
//
//      However, then the update() member of the AY8910 (the most commonly
//      used version of the chip) would have to traverse up to AY8913
//      for most of its functionality.  This is a couple of extra
//      function calls and a hit I don't want to take at the moment.
//
//      Thanks go to Ville Hallik/Michael Cuddy and the Mame team for 
//      code on which this object is based.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _ay8910_h_
#define _ay8910_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "repbase.h"
#include "kstring.h"
#include "sdev.h"
#include "cpurh.h"
#include "cpuwh.h"

//  Forward class declarations.
class Sample;


///////////////////////////////////////////////////////////////////////////////
//  AY8910 Class Definition
///////////////////////////////////////////////////////////////////////////////
class AY8910 : public SoundDevice
{
    public:
        
        //  Creation/Deletion.
        AY8910 ( 
            const KString& iName,
            const DWord    dwChipClock,
            const DWord    dwChipVolume
        );
        virtual ~AY8910 ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Persistence.
        virtual Byte save ( AppFile* pSaveFile );
        virtual Byte load ( AppFile* pLoadFile );

        //  Interface.
        virtual void update ( );

        //  Chip Manipulation.
        void       controlRegister     ( const Byte bRegister );
        void       writeRegister       ( const Byte bValue );
        const Byte readRegister        ( );
        void       setClock            ( 
            const DWord dwClockRate, const DWord dwSampleRate
        );
        
        //  Setup.
        void setPortAReadHandler ( 
            ReadHandler::ReadFunc pfnHandler, void* pData 
        );
        void setPortBReadHandler ( 
            ReadHandler::ReadFunc pfnHandler, void* pData 
        );
        void setPortAWriteHandler ( 
            WriteHandler::WriteFunc pfnHandler, void* pData 
        );
        void setPortBWriteHandler ( 
            WriteHandler::WriteFunc pfnHandler, void* pData 
        );

        
    protected:

        //  The following enum defines the available registers in the
        //  AY-3-8910.
        enum
        {
            A_FINE_PITCH = 0x00,
            A_COARSE_PITCH,
            B_FINE_PITCH,
            B_COARSE_PITCH,
            C_FINE_PITCH,
            C_COARSE_PITCH,
            NOISE_PITCH,
            ENABLE,
            A_VOLUME,
            B_VOLUME,
            C_VOLUME,
            E_FINE_DURATION,
            E_COARSE_DURATION,
            E_SHAPE,
            PORT_A,
            PORT_B,

            NUM_8910_REGISTERS
        };
             

        //  Utility.
        void setGain         ( );
        void updateBuffer    ( const Byte bToEnd );
        void doWriteRegister ( const Byte bValue );


        //  Member Data.
        Byte           m_bInitialized;
        DWord          m_dwEmulationRate;
        DWord          m_dwChipClock;
        DWord          m_dwChipVolume;
        Byte           m_bRegister;
        Byte           m_abRegisters[ 16 ];
        Sample*        m_pSample;
        int32          m_nUpdatePos;
        DWord          m_dwUpdateStep;
        int32          m_nPeriodA; 
        int32          m_nPeriodB; 
        int32          m_nPeriodC; 
        int32          m_nPeriodN; 
        int32          m_nPeriodE; 
        int32          m_nCountA;
        int32          m_nCountB;
        int32          m_nCountC;
        int32          m_nCountN;
        int32          m_nCountE;
        DWord          m_dwVolumeA;
        DWord          m_dwVolumeB;
        DWord          m_dwVolumeC;
        DWord          m_dwVolumeE;
        Byte           m_bEnvelopeA;
        Byte           m_bEnvelopeB;
        Byte           m_bEnvelopeC;
        Byte           m_bOutputA;
        Byte           m_bOutputB;
        Byte           m_bOutputC;
        Byte           m_bOutputN;
        int8           m_nCountEnv;
        Byte           m_bHold;
        Byte           m_bAlternate;
        Byte           m_bAttack;
        Byte           m_bHolding;
        int32          m_nRNG; 
        DWord          m_adwVolumeTable[ 32 ];
        ReadHandler*   m_pReadHandlerA;
        ReadHandler*   m_pReadHandlerB;
        WriteHandler*  m_pWriteHandlerA;
        WriteHandler*  m_pWriteHandlerB;

    private:

        //  Static Member Data.
        static const DWord sm_dwScale;
};


///////////////////////////////////////////////////////////////////////////////
//  AY8910 Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
AY8910::controlRegister(
    const Byte bRegister
)
{
    m_bRegister = bRegister;
}

inline
void
AY8910::writeRegister(
    const Byte bValue
)
{
     updateBuffer( FALSE ); 
     doWriteRegister( bValue );
}

inline
void
AY8910::setClock(
    const DWord dwClockRate,
    const DWord dwSampleRate
)
{
    //  The step for the generators is 1/8th of the chip clock.  The step
    //  for the envelope generator is 1/16th of the chip clock, but we 
    //  use 1/8th to be compatible with possible derived classes.
    //  The number of steps during one sample at the given sample rate
    //  is given here.  This will typically result in a decimal so we 
    //  include a scale to keep it integral.  This value will have to 
    //  be scaled back whenever it is used.
    m_dwUpdateStep = ( DWord )( 
        ( ( double ) sm_dwScale * dwSampleRate * 8 ) / dwClockRate
    );
}
    
inline
void 
AY8910::setPortAReadHandler( 
    ReadHandler::ReadFunc pfnHandler, 
    void*                 pData 
)
{
    m_pReadHandlerA = new ReadHandler( "ReadA", 0, 0, pfnHandler, pData );
}

inline
void 
AY8910::setPortBReadHandler( 
    ReadHandler::ReadFunc pfnHandler, 
    void*                 pData 
)
{
    m_pReadHandlerB = new ReadHandler( "ReadB", 0, 0, pfnHandler, pData );
}

inline
void 
AY8910::setPortAWriteHandler( 
    WriteHandler::WriteFunc pfnHandler, 
    void*                   pData 
)
{
    m_pWriteHandlerA = new WriteHandler( "WriteA", 0, 0, pfnHandler, pData );
}

inline
void 
AY8910::setPortBWriteHandler( 
    WriteHandler::WriteFunc pfnHandler, 
    void*                   pData 
)
{
    m_pWriteHandlerB = new WriteHandler( "WriteB", 0, 0, pfnHandler, pData );
}



#endif
