///////////////////////////////////////////////////////////////////////////////
//
//  File:       waveform.h
//
//  Class:      WaveForm - Concrete
//  
//  Hierarchy:  WaveForm - SoundDevice - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class is responsible for emulating waveform based sound found
//      in some of the Bally/Midway games such as the pacman series and
//      the dual 6809 games.
//
//      Thanks go to the Mame team for code on which this object is based.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _waveform_h_
#define _waveform_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "repbase.h"
#include "sdev.h"


//  Forward class declarations.
class Sample;


///////////////////////////////////////////////////////////////////////////////
//  WaveForm Class Definition
///////////////////////////////////////////////////////////////////////////////
class WaveForm : public SoundDevice
{
    public:
        
        //  Creation/Deletion.
        WaveForm ( 
            const KString& iName,
            const DWord    dwSampleRate,
            const DWord    dwNumVoices,
            const DWord    dwGain,
            const DWord    dwVolume = 0xff
        );
        virtual ~WaveForm ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Persistence.
        virtual Byte save ( AppFile* pSaveFile );
        virtual Byte load ( AppFile* pLoadFile );

        //  Interface.
        virtual void update ( );

        //  Get/Set.
        void setFrequency ( const DWord dwVoice, const DWord dwFrequency );
        void setVolume    ( const DWord dwVoice, const DWord dwVolume );
        void setWave      ( const DWord dwVoice, Byte* pbWave );
        void enable       ( const Byte bEnabled );

    protected:

    private:

        //  The following class is used to hold the information concerning
        //  a voice.
        class Voice
        {
            public:
                DWord m_dwFrequency;
                DWord m_dwVolume;
                Byte* m_pbWave;
                DWord m_dwCounter;
        };

        //  Member Functions.
        void initializeMixer ( );
        void updateBuffer    ( const Byte bToEnd );

        //  Member Data.
        Sample*        m_pSample;
        DWord          m_dwNumVoices;
        Voice          m_aVoiceList[ 8 ];
        DWord          m_dwGain;
        int16*         m_pnMixerBuffer;
        int8*          m_pnMixerLookup;
        int8*          m_pnMixerTable;
        int32          m_nUpdatePos;
        Byte           m_bEnabled;
};


///////////////////////////////////////////////////////////////////////////////
//  WaveForm Inline Functions
///////////////////////////////////////////////////////////////////////////////
inline
void 
WaveForm::setFrequency( 
    const DWord dwVoice, 
    const DWord dwFrequency 
)
{
    ASSERT( dwVoice < 8 );

    updateBuffer( FALSE );

    m_aVoiceList[ dwVoice ].m_dwFrequency = dwFrequency;
}

inline 
void 
WaveForm::setVolume( 
    const DWord dwVoice, 
    const DWord dwVolume 
)
{
    ASSERT( dwVoice < 8 );

    updateBuffer( FALSE );

    m_aVoiceList[ dwVoice ].m_dwVolume = dwVolume;
}

inline
void 
WaveForm::setWave( 
    const DWord dwVoice, 
    Byte*       pbWave 
)
{
    ASSERT( dwVoice < 8 );

    updateBuffer( FALSE );

    m_aVoiceList[ dwVoice ].m_pbWave = pbWave;
}

inline
void 
WaveForm::enable( 
    const Byte bEnabled
)
{
    m_bEnabled = bEnabled;
}

#endif
