///////////////////////////////////////////////////////////////////////////////
//
//  File:    kstring.cpp
//
//  Class:   KString
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is a lightweight string class.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <string.h>

//  Application Headers.
#include "kstring.h"


//  Static initializations.
const KString KString::sm_null( "" );


///////////////////////////////////////////////////////////////////////////////
//
//  Function: KString
//
//  Description:
//
//      This constructor creates a null KString.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString::KString(
)
:
    m_pstrData ( NULL ),
    m_nLength  ( 0 )
{
    m_pstrData = new char[ m_nLength + 1 ];
    strcpy( m_pstrData, "" );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: KString
//
//  Description:
//
//      This constructor creates a copy of the specified KString.
//
//  Parameters:
//
//      ksRef (input)
//          The string to copy.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString::KString(
    const KString& ksRef,
    const int      nLength /* = -1 */
)
:
    m_pstrData ( NULL ),
    m_nLength  ( 0 )
{
    //  Calculate the length of the string.  If the a length is specified
    //  than use it otherwise use the length of the string.
    if( ( nLength > 0 ) && ( nLength < ksRef.length( ) ) )
    {
        m_nLength = nLength;
    }
    else
    {
        m_nLength = ksRef.length( );
    }

    //  Create the string.
    m_pstrData = new char[ m_nLength + 1 ];
    strncpy( m_pstrData, ksRef.data( ), m_nLength ); 
    m_pstrData[ m_nLength ] = '\0';
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: KString
//
//  Description:
//
//      This constructor creates a copy of the specified string.
//
//  Parameters:
//
//      pstrInitial (input)
//          The string to copy.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString::KString(
    const char* pstrInitial,
    const int   nLength /* = -1 */
)
:
    m_pstrData ( NULL ),
    m_nLength  ( 0 )
{
    //  Calculate the length of the string.  If the a length is specified
    //  than use it otherwise use the length of the string.
    if( ( nLength > 0 ) && ( nLength < ( int )strlen( pstrInitial ) ) )
    {
        m_nLength = nLength;
    }
    else
    {
        m_nLength = strlen( pstrInitial );
    }

    //  Create the string.
    m_pstrData = new char[ m_nLength + 1 ];
    strncpy( m_pstrData, pstrInitial, m_nLength );
    m_pstrData[ m_nLength ] = '\0';
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~KString
//
//  Description:
//
//      This is the destructor for the KString object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString::~KString(
)
{
    //  Delete the space used by the string.
    delete [] m_pstrData;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: operator=
//
//  Description:
//
//      The assignment operator from another KString.
//
//  Parameters:
//
//      ksAssign (input)
//          The string to assign from.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString&
KString::operator=(
    const KString& ksAssign
)
{
    //  Remove the old space.
    delete [] m_pstrData;

    //  Assign the new string.
    m_nLength = ksAssign.length( );
    m_pstrData = new char[ m_nLength + 1 ];
    strcpy( m_pstrData, ksAssign.data( ) );

    return( *this );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: operator=
//
//  Description:
//
//      The assignment operator from a string.
//
//  Parameters:
//
//      pstrAssign (input)
//          The string to assign from.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString&
KString::operator=(
    const char* pstrAssign
)
{
    //  Remove the old space.
    delete [] m_pstrData;

    //  Assign the new string.
    m_nLength = strlen( pstrAssign );
    m_pstrData = new char[ m_nLength + 1 ];
    strcpy( m_pstrData, pstrAssign );

    return( *this );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: operator+=
//
//  Description:
//
//      The concatenation operator from another KString.
//
//  Parameters:
//
//      ksAppend (input)
//          The string to append.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString&
KString::operator+=(
    const KString& ksAppend
)
{
    //  A pointer to the previous data.
    char* m_pstrPrev;

    //  Hold on to the current contents of the string.
    m_pstrPrev = m_pstrData;
    
    //  The length is the sum of the lengths.
    m_nLength += ksAppend.length( );
    m_pstrData = new char[ m_nLength + 1 ];
    strcpy( m_pstrData, m_pstrPrev );
    strcat( m_pstrData, ksAppend.data( ) );
  
    //  Blow away the old string.
    delete [] m_pstrPrev;

    return( *this );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: operator+=
//
//  Description:
//
//      The concatenation operator from a string.
//
//  Parameters:
//
//      pstrAppend (input)
//          The string to append.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString&
KString::operator+=(
    const char* pstrAppend
)
{
    //  A pointer to the previous data.
    char* m_pstrPrev;

    //  Hold on to the current contents of the string.
    m_pstrPrev = m_pstrData;
    
    //  The length is the sum of the lengths.
    m_nLength += strlen( pstrAppend );
    m_pstrData = new char[ m_nLength + 1 ];
    strcpy( m_pstrData, m_pstrPrev );
    strcat( m_pstrData, pstrAppend );
  
    //  Blow away the old string.
    delete [] m_pstrPrev;

    return( *this );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: operator[]
//
//  Description:
//
//      Returns the specified character in the string.
//
//  Parameters:
//
//      nIdx (input)
//          The index of the character to return.
//
//  Returns:
//
//      The specified character.
//
///////////////////////////////////////////////////////////////////////////////
char&
KString::operator[](
    const int nIdx
) const
{
    //  If the index is valid then return the character, otherwise return
    //  the last character.
    if( ( nIdx < 0 ) || ( nIdx >= m_nLength ) )
    {
        return( *( m_pstrData + m_nLength ) );
    }
    else
    {
        return( *( m_pstrData + nIdx ) );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: pad
//
//  Description:
//
//      This function pads the string with the specified number of padding
//      characters.
//
//  Parameters:
//
//      cPadding (input)
//          The character to pad with.
//
//      nPadLength (input)
//          The number of cPadding characters to pad with.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KString&
KString::pad( 
    const char cPadding, 
    const int  nPadLength 
)
{
    //  A pointer to the previous data.
    char* m_pstrPrev;

    //  Hold on to the current contents of the string.
    m_pstrPrev = m_pstrData;
    
    //  The length is the sum of the old length and the padding.
    m_nLength += nPadLength;
    m_pstrData = new char[ m_nLength + 1 ];
    strcpy( m_pstrData, m_pstrPrev );
  
    //  Pad away.
    m_pstrData[ m_nLength ] = '\0';
    for( int nI = 0 ; nI < nPadLength ; nI += 1 )
    {
        m_pstrData[ m_nLength - nI - 1 ] = cPadding;
    }

    //  Blow away the old string.
    delete [] m_pstrPrev;

    return( *this );
}
