///////////////////////////////////////////////////////////////////////////////
//
//  File:    canvasx.cpp
//
//  Class:   CanvasUnixX
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents the Display of the Replay application on
//      the Unix/X platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "canvasx.h"
#include "pagex.h"
#include "bitmapx.h"
#include "colour.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a Unix/X Canvas object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
CanvasUnixX*
CanvasUnixX::s_build(
    const KString& iName
)
{
    //  Create the new object.
    CanvasUnixX* pThis = new CanvasUnixX( iName );
 
    //  Initialize the new object.
    pThis->init( );

    //  Send back the new canvas.
    return( pThis );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: CanvasUnixX
//
//  Description:
//
//      This is the main constructor for the Unix/X Canvas object.  This is
//      a protected member.  Clients wishing to create a canvas object should
//      do so through the factory method.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CanvasUnixX::CanvasUnixX(
    const KString& iName
)
:
    Canvas         ( iName ),
    m_pDisplay     ( NULL ),
    m_pScreen      ( NULL ),
    m_colourMap    ( 0 ),
    m_gc           ( NULL ),
    m_bTrueColour  ( FALSE ),
    m_bPrivateCMap ( FALSE ),
    m_bImageScale  ( 0 ),
    m_pdwPixelList ( NULL )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the canvas object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasUnixX::init(
)
{
    //  The default screen.
    int nScreen;



    //  Open the X Display.
    m_pDisplay = XOpenDisplay( NULL );
    CONFIRM( m_pDisplay != ( Display* )NULL, "Unable to open X-Display" );

    //  Grab the default screen.
    nScreen   = DefaultScreen( m_pDisplay );
    m_pScreen = DefaultScreenOfDisplay( m_pDisplay );
    CONFIRM( m_pScreen != ( Screen* )NULL, "Unable to get default screen" );


    //  If 8 bit psuedo-colour isn't being used find a true-colour mode.
    if( !XMatchVisualInfo( m_pDisplay, nScreen, 8, PseudoColor, &m_visual ) )
    {
        //  We're using a true colour mode.
        m_bTrueColour = TRUE;
         
        //  Which True Colour Mode?
        if( XMatchVisualInfo( m_pDisplay, nScreen, 16, TrueColor, &m_visual ) )
        {
            //  We need two bytes per pixel.
            m_bImageScale = 2; 
        }
        else
        if( XMatchVisualInfo( m_pDisplay, nScreen, 24, TrueColor, &m_visual ) )
        {
            //  We need four bytes per pixel.
            m_bImageScale = 4; 
        }
        else
        if( XMatchVisualInfo( m_pDisplay, nScreen, 32, TrueColor, &m_visual ) )
        {
            //  We need four bytes per pixel.
            m_bImageScale = 4; 
        }
        else
        {
            fatalError( "Could not determine visual used." );
        }
    }

    //  Now that we've determined the visual characteristics we can call
    //  the base init( ).  The base init( ) uses some of this information
    //  (such as # of available colours) so this init( ) shouldn't be done
    //  any sooner.
    Canvas::init( );


    //  Grab the default Graphics Context.
    m_gc = DefaultGCOfScreen( m_pScreen );


    //  We only use a private colour map for non-true colour displays.
    if( m_bTrueColour )
    {
        m_bPrivateCMap = FALSE;
    }
    else
    {
        m_bPrivateCMap = TRUE;
    }

    //  Initialize the colour map.
    initCMap( );
}
     

///////////////////////////////////////////////////////////////////////////////
//
//  Function: initCMap
//
//  Description:
//
//      This is called to initialize the colour map for the canvas object.
//      Replay supports both private and shared colour maps.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasUnixX::initCMap(
)
{
    //  If we using a private colour map then we must create one, otherwise
    //  we attempt to allocate as many shared colours as we can.
    if( m_bPrivateCMap )
    {
        //  Create the private colour map and allocate all colours for write.
        m_colourMap = XCreateColormap(
            m_pDisplay,
            DefaultRootWindow( m_pDisplay ),
            m_visual.visual,
            AllocAll
        ); 
    }
    else
    {
        //  If this isn't True-Colour then we will need an image buffer that
        //  is the same size as the bitmap buffer.
        if( !m_bTrueColour )
        {
            m_bImageScale = 1;
        }

        //  Grab the default shared colour map.    
        m_colourMap = DefaultColormapOfScreen( m_pScreen );

        //  Since we are using a shared colour map, we may not always be
        //  able to grab the colour cell we're interested in.  What this
        //  means is that if we attempt to allocate cell 5 and it is already
        //  taken then we may get cell 10.  To handle this, we need a mapping
        //  between the game's values and the X pixel values.  So we allocate
        //  it here.
        m_pdwPixelList = new DWord[ getNumColours( ) ];

        //  Attempt to allocate cells for each colour.
        for( DWord dwI = 0 ; dwI < getNumColours( ) ; dwI += 1 )
        {
            //  An X colour structure used to initialize the colour map.
            XColor colour;
            
            //  Initialize the colour structure.
            colour.red   = 0x0000;
            colour.green = 0x0000;
            colour.blue  = 0x0000;
            colour.pixel = dwI;
    
            //  Allocate the colour.
            if( XAllocColor( m_pDisplay, m_colourMap, &colour ) == 0 )
            {
                fatalError( "Could not initialize colour (%d).", dwI );
            } 
           
            //  Record the pixel value for the colour.
            m_pdwPixelList[ dwI ] = colour.pixel; 
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CanvasUnixX
//
//  Description:
//
//      This is the main destructor for the Unix/X Canvas object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CanvasUnixX::~CanvasUnixX(
)
{
    //  If we are using a private colour map then free the map, otherwise
    //  free the individual colours.
    if( m_bPrivateCMap )
    {
        XFreeColormap( m_pDisplay, m_colourMap );
    }
    else
    {
        for( DWord dwI = 0 ; dwI < getNumColours( ) ; dwI += 1 )
        {
            unsigned long lnPixel = m_pdwPixelList[ dwI ];
            XFreeColors( m_pDisplay, m_colourMap, &lnPixel, 1, 0 );
        }
    }

    //  Close the display.
    ASSERT( m_pDisplay != ( Display* )NULL );
    XCloseDisplay( m_pDisplay );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the canvas object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CanvasUnixX::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "CanvasUnixX" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addPage
//
//  Description:
//
//      This member is used to add a page to the canvas.
//
//  Parameters:
//
//      pageName (input)
//          The name of the page.
//
//      ePageNumber (input)
//          The page number.
//
//      dwWidth (input)
//          The width of the page.
//
//      dwHeight (input)
//          The height of the page.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasUnixX::addPage(
    const KString& pageName,
    PageNumber     ePageNumber,
    DWord          dwWidth,
    DWord          dwHeight
)
{
    //  Make sure the page doesn't exist already.
    ASSERT( m_apPageList[ ePageNumber ] == NULL );

    //  OK, create the new page.
    m_apPageList[ ePageNumber ] = new PageUnixX( 
        pageName, 
        dwWidth, 
        dwHeight, 
        getTranspose( ), 
        m_pDisplay, 
        m_pScreen, 
        m_colourMap
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: turnPage
//
//  Description:
//
//      This member is called to set the current page on the canvas.
//
//  Parameters:
//
//      ePageNumber (input)
//          The page to turn to.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasUnixX::turnPage(
    PageNumber ePageNumber
)
{
    //  Call the base class.
    Canvas::turnPage( ePageNumber );

    //  Raise the current page to the top of the stack.
    XMapRaised( m_pDisplay, ( ( PageUnixX* )m_pPage )->getWindow( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: createBitmap
//
//  Description:
//
//      This member is used to create a blank bitmap.
//
//  Parameters:
//
//      iName (input)
//          instance name of the object.
//
//      dwWidth (input)
//          The width of the page.
//
//      dwHeight (input)
//          The height of the page.
//
//      bForScreen (input)
//          Is the bitmap a screen bitmap.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Bitmap*
CanvasUnixX::createBitmap(
    const KString& iName,
    DWord          dwWidth,
    DWord          dwHeight,
    Byte           bForScreen /* = FALSE */

)
{
    //  Return a new bitmap. 
    return( 
        BitmapUnixX::s_build( 
            iName, 
            dwWidth, 
            dwHeight, 
            bForScreen, 
            m_bImageScale,
            m_pDisplay, 
            m_pScreen,
            &m_visual
        ) 
    );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is used to draw a bitmap on the current page.
//
//  Parameters:
//
//      pBitmap (input)
//          The bitmap to draw.
//
//      nX (input)
//          The X position to draw the bitmap at.
//
//      nY (input)
//          The Y position to draw the bitmap at.
//
//      dwWidth (input)
//          The width to draw.
//
//      dwHeight (input)
//          The height to draw.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasUnixX::draw(
    Bitmap* pBitmap,
    int32   nX,
    int32   nY,
    DWord   dwWidth,
    DWord   dwHeight
)
{
    //  There must be a page selected.
    ASSERT( m_pPage != NULL );

    //  The bitmap must be a screen bitmap.
    ASSERT( pBitmap->isScreen( ) );

    //  The width and height must be no greater than the bitmap.
    ASSERT( dwWidth  <= pBitmap->getWidth( ) );
    ASSERT( dwHeight <= pBitmap->getHeight( ) );
    
    //  Used to swap variables.
    int32 nSwap;
    DWord dwSwap;


    //  If the canvas is transposed then we need to swap the X/Y related 
    //  parameters.
    if( getTranspose( ) )
    {
        //  Swap the X/Y locations.
        nSwap = nX;
        nX    = nY;
        nY    = nSwap;

        //  Now swap the width/height of the area to draw.
        dwSwap   = dwWidth;
        dwWidth  = dwHeight;
        dwHeight = dwSwap;
    }
    
    //  If the canvas is X Flipped then we need to adjust the X coordinate.
    if( getFlipX( ) )
    {
        nX = 
            ( getTranspose( ) ? m_pPage->getHeight( ) : m_pPage->getWidth( ) ) -
            dwWidth -
            nX;
    }
    
    //  If the canvas is Y Flipped then we need to adjust the Y coordinate.
    if( getFlipY( ) )
    {
        nY = 
            ( getTranspose( ) ? m_pPage->getWidth( ) : m_pPage->getHeight( ) ) -
            dwHeight -
            nY;
    }
    
    //  If the image buffer is scaled then we need to transfer the bitmap
    //  buffer to it.
    if( m_bImageScale )
    {
        ( ( BitmapUnixX* )pBitmap )->transferBuffer( m_pdwPixelList );
    }


    //  Draw the bitmap.
#ifdef XSHM
    XShmPutImage(
        m_pDisplay,
        ( ( PageUnixX* )m_pPage )->getWindow( ),
        m_gc,
        ( ( BitmapUnixX* )pBitmap )->getImage( ),
        0,
        0,
        nX,
        nY,
        dwWidth,
        dwHeight,
        False
    );
#else
    XPutImage(
        m_pDisplay,
        ( ( PageUnixX* )m_pPage )->getWindow( ),
        m_gc,
        ( ( BitmapUnixX* )pBitmap )->getImage( ),
        0,
        0,
        nX,
        nY,
        dwWidth,
        dwHeight
    );
#endif
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getNumColours
//
//  Description:
//
//      This member returns the maximum number of colours supported by
//      the canvas.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The number of colours supported.
//
///////////////////////////////////////////////////////////////////////////////
DWord
CanvasUnixX::getNumColours( 
) const
{
    //  Currently we're always returning 256 colours.  In the future, this
    //  should be changed to return the real number of colours supported.
    return( 256 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: changeColour
//
//  Description:
//
//      This member is called to change a colour in the palette.
//
//  Parameters:
//
//      dwIndex (input)
//          The colour to change.
//
//      dwRed (input)
//          The red component to set.
//
//      dwGreen (input)
//          The green component to set.
//
//      dwBlue (input)
//          The blue component to set.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
CanvasUnixX::changeColour( 
    DWord dwIndex, 
    DWord dwRed, 
    DWord dwGreen, 
    DWord dwBlue
)
{
    ASSERT( dwIndex < getNumColours( ) );
    
    //  A colour structure and a pointer to the colour object we're interested
    //  in.
    XColor  colour;
    Colour* pColour = &( m_pPaletteColours[ dwIndex ] );

    //  Save the RGB values.
    pColour->set( dwRed, dwGreen, dwBlue );

    //  Set the R/G/B values.
    colour.flags = DoRed | DoGreen | DoBlue;
    colour.red   = ( dwRed   * 65535 ) / 255;
    colour.green = ( dwGreen * 65535 ) / 255;
    colour.blue  = ( dwBlue  * 65535 ) / 255;

    //  Set the pixel number (colour cell number).
    colour.pixel = dwIndex;


    //  If we're using a private colour map then just store the colour, 
    //  otherwise we have to free the colour and reallocate it.
    if( m_bPrivateCMap )
    {
        XStoreColor( m_pDisplay, m_colourMap, &colour );
    }
    else
    {
        //  Free the colour so we can reallocate it.
        unsigned long lnPixel = m_pdwPixelList[ dwIndex ];
        XFreeColors( m_pDisplay, m_colourMap, &lnPixel, 1, 0 );

        //  Attempt to allocate the colour and if we succeed then reassign
        //  the pixel value for this entry.
        if( XAllocColor( m_pDisplay, m_colourMap, &colour ) )
        {
            m_pdwPixelList[ dwIndex ] = colour.pixel;
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called to allow the canvas to update it's display.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if an expose event has been received.
//      FALSE if no expose event has been received.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CanvasUnixX::update(
)
{
    //  Indicates whether or not an expose event has occurred.
    Byte bExpose;

    //  The next X-event.
    XEvent event;


    //  If there is no page then we don't really care about any events.
    if( m_pPage == NULL )
    {
        return( FALSE );
    }


    //  If there is an update needed then say so.
    if( m_bUpdateNeeded )
    {
        m_bUpdateNeeded = FALSE;
        return( TRUE );
    }


    //  Look for expose events on the current page.
    for(
        bExpose = FALSE ;
        XCheckWindowEvent(
            m_pDisplay,
            ( ( PageUnixX* )m_pPage )->getWindow( ),
            ExposureMask,
            &event
        ) == True ;
    )
    {
        //  We have an exposure.
        bExpose = TRUE;
    }

    return( bExpose );
}
