///////////////////////////////////////////////////////////////////////////////
//
//  File:    gamfactx.cpp
//
//  Class:   GameFactoryUnixX
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is responsible for creating game objects for the games
//      that Replay knows about.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <dlfcn.h>
#include <dirent.h>
#include <string.h>

//  Application Headers.
#include "gamfactx.h"
#include "replay.h"
#include "disk.h"
#include "gameinfo.h"
#include "game.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a Unix/X Game Factory object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object.
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
GameFactoryUnixX*
GameFactoryUnixX::s_build(
    const KString& iName
)
{
    //  Create the new object.
    GameFactoryUnixX* pThis = new GameFactoryUnixX( iName );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new object.
    return( pThis );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameFactoryUnixX
//
//  Description:
//
//      This is the main constructor for a game factory object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameFactoryUnixX::GameFactoryUnixX(
    const KString& iName
)
:
    GameFactory    ( iName ),
    m_pGameFinder  ( NULL ),
    m_pLibrary     ( NULL )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the game factory object.  By using an 
//      init member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameFactoryUnixX::init(
)
{
    //  A directory to check for shared libraries.
    KString libDir;

    //  A pointer to the disk object.
    Disk* pDisk = Replay::s_instance( ).getDisk( );


    //  We need to search for all shared libraries and open them.  They 
    //  should be stored in <path0>/games but for development purposes,
    //  ./lib/<arch> is also allowed.
    libDir  = pDisk->getPath( );
    libDir += pDisk->getDirectorySeparator( );
    libDir += "games";
    addGames( libDir );
    libDir  = ".";
    libDir += pDisk->getDirectorySeparator( );
    libDir += "lib.";
    libDir += ARCH;
    addGames( libDir );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameFactoryUnixX
//
//  Description:
//
//      This is the destructor for a Game Factory object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameFactoryUnixX::~GameFactoryUnixX(
)
{
    //  Free the handles in the game finder objects.
    for( DWord dwI = 0 ; dwI < m_gameFinderList.entries( ) ; dwI += 1 )
    {
        delete m_gameFinderList[ dwI ]->getHandle( );
    }

    //  If there is a shared library open, then close it.
    if( m_pLibrary )
    {
        dlclose( m_pLibrary );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Game Factory class.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
GameFactoryUnixX::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "GameFactoryUnixX" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addGames
//
//  Description:
//
//      This is called to search the specified directory for shared game
//      libraries.  It will then add each game found to the list.
//
//  Parameters:
//
//      libDir (input)
//          The directory to search in.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameFactoryUnixX::addGames(
    const KString& libDir
)
{
    //  Has a specific game been specified?
    KString runGame;
    Configuration::s_instance( ).getParam( "-game", &runGame );

    //  A pointer to the disk object.
    Disk* pDisk = Replay::s_instance( ).getDisk( );

    //  A pointer to the directory specified.
    DIR* pDir;
    
    //  A directory entry.
    struct dirent* pDirEntry;

    //  A list of shared libraries.
    KPtrList<KString> libList;

    //  The full file name of a shared library.
    KString fullFileName;

    //  The id of a game and of a function in the shared library.
    char* pstrId;
    char  strFnName[ 16 ];

    //  The length of the game identifier.
    DWord dwGameIdLength;

    //  Function pointer to the "info function of the library.
    const char**(* pfnInfo )( );


    //  Open the directory.  If it can't be opened, then return.
    pDir = opendir( libDir.data( ) );
    if( pDir == NULL )
    {
        return;
    }

    //  Loop through each directory entry and add any .so files to a list.
    for(
        pDirEntry = readdir( pDir ) ;
        pDirEntry != NULL ;
        pDirEntry = readdir( pDir )
    )
    {
        //  If the extension is '.so' then we have found a shared library so
        //  get the information from it.
        if( 
            ( strrchr( pDirEntry->d_name, '.' ) != NULL ) && 
            ( strcmp( strrchr( pDirEntry->d_name, '.' ), ".so" ) == 0 )
        )
        {
            //  Build the full file name of the shared library.
            fullFileName  = libDir;
            fullFileName += pDisk->getDirectorySeparator( );
            fullFileName += pDirEntry->d_name;

            //  Open the shared library.
            m_pLibrary = dlopen( fullFileName.data( ), RTLD_LAZY );
            if( m_pLibrary == NULL )
            {
                CHECK1( 
                    FALSE,
                    "Shared library %s could not be opened.",
                    fullFileName.data( )
                );
                continue;
            }
            
            //  The name of the file (less the .so) should be the identifier
            //  of the game.  Retrieve the length of the identifier.
            dwGameIdLength = 
                strrchr( pDirEntry->d_name, '.' ) - pDirEntry->d_name;
            CONFIRM( 
                dwGameIdLength <= 8, "Invalid file name %s.", pDirEntry->d_name
            );

            //  Get the game id.
            pstrId = new char[ dwGameIdLength + 1 ];
            strncpy( pstrId, pDirEntry->d_name, dwGameIdLength );
            pstrId[ dwGameIdLength ] = '\0';

            //  Now retrieve the info function.
            sprintf( strFnName, "%s_info", pstrId );
            pfnInfo = ( const char** ( * )( ) )dlsym( m_pLibrary, strFnName );
            CONFIRM( 
                pfnInfo != NULL, "%s has no info function.", pDirEntry->d_name
            );

            //  Create the new game finder object for this shared library.
            if(
                ( runGame == KStringNULL ) || 
                ( strcmp( pstrId, runGame.data( ) ) == 0 )
            )
            {
                m_gameFinderList.add(
                    new GameFinder(
                        pstrId, 
                        new GameInfo( "info", ( *pfnInfo )( ) ),  
                        fullFileName,
                        pstrId
                    )
                );
            }

            //  All done, close the library.
            dlclose( m_pLibrary );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: newGame
//
//  Description:
//
//      This member creates a new game based on the specified index.
//
//  Parameters:
//
//      dwGameId (input)
//          The index of the game to create.
//
//  Returns:
//
//      A pointer to the newly created game object.
//
///////////////////////////////////////////////////////////////////////////////
Game*
GameFactoryUnixX::newGame(
    const DWord dwGameId
)
{
    //  The name of a function in the opened library.
    char strFnName[ 16 ];

    //  Function pointer to the "create" function of the library.
    Game*(* pfnCreate )( );


    ASSERT( m_pGame == NULL );
    ASSERT( dwGameId < m_gameFinderList.entries( ) );


    //  Assign the game finder for this game.
    m_pGameFinder = m_gameFinderList[ dwGameId ];

    //  Open the library.
    m_pLibrary = dlopen( m_pGameFinder->getFileName( ).data( ), RTLD_LAZY );

    //  Now retrieve the create function.
    sprintf( strFnName, "%s_create", ( char* )m_pGameFinder->getHandle( ) );
    pfnCreate = ( Game* ( * )( ) )dlsym( m_pLibrary, strFnName );
    CONFIRM( 
        pfnCreate != NULL, 
        "%s has no create function.",
        m_pGameFinder->getFileName( ).data( )
    );

    //  Build the game.
    m_pGame =  ( *pfnCreate )( );

    //  Pass the game information to the game.
    m_pGame->setGameInfo( m_pGameFinder->getGameInfo( ) );


    return( m_pGame );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: deleteGame
//
//  Description:
//
//      This member deletes the specified game.
//
//  Parameters:
//
//      pGame (input)
//          The game object to delete.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameFactoryUnixX::deleteGame(
    Game* pGame
)
{
    //  Let the base class do it's thing.
    GameFactory::deleteGame( pGame );

    //  Close the shared library.
    ASSERT( m_pLibrary != NULL );
    dlclose( m_pLibrary );
    m_pLibrary = NULL;
    m_pGameFinder = NULL;
}
