///////////////////////////////////////////////////////////////////////////////
//
//  File:    keybx.cpp
//
//  Class:   KeyboardUnixX
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The KeyboardUnixX class encapsulates keyboard access for the Unix/X
//      platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "keybx.h"
#include "canvasx.h"
#include "replay.h"
#include "switch.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data.
///////////////////////////////////////////////////////////////////////////////
const Keyboard::KeyboardKey KeyboardUnixX::sm_aKeyCodes[ ] =
{
    KEY__NONE,              //  N/A                    0x000
    KEY__NONE,              //  N/A                    0x001
    KEY__NONE,              //  N/A                    0x002
    KEY__NONE,              //  N/A                    0x003
    KEY__NONE,              //  N/A                    0x004
    KEY__NONE,              //  N/A                    0x005
    KEY__NONE,              //  N/A                    0x006
    KEY__NONE,              //  N/A                    0x007
    KEY__NONE,              //  N/A                    0x008
    KEY__NONE,              //  N/A                    0x009
    KEY__NONE,              //  N/A                    0x00a
    KEY__NONE,              //  N/A                    0x00b
    KEY__NONE,              //  N/A                    0x00c
    KEY__NONE,              //  N/A                    0x00d
    KEY__NONE,              //  N/A                    0x00e
    KEY__NONE,              //  N/A                    0x00f
    KEY__NONE,              //  N/A                    0x010
    KEY__NONE,              //  N/A                    0x011
    KEY__NONE,              //  N/A                    0x012
    KEY__NONE,              //  N/A                    0x013
    KEY__NONE,              //  N/A                    0x014
    KEY__NONE,              //  N/A                    0x015
    KEY__NONE,              //  N/A                    0x016
    KEY__NONE,              //  N/A                    0x017
    KEY__NONE,              //  N/A                    0x018
    KEY__NONE,              //  N/A                    0x019
    KEY__NONE,              //  N/A                    0x01a
    KEY__NONE,              //  N/A                    0x01b
    KEY__NONE,              //  N/A                    0x01c
    KEY__NONE,              //  N/A                    0x01d
    KEY__NONE,              //  N/A                    0x01e
    KEY__NONE,              //  N/A                    0x01f
    KEY__SPACE,             //  XK_space               0x020
    KEY__1,                 //  XK_exclam              0x021
    KEY__QUOTE,             //  XK_quotedbl            0x022
    KEY__3,                 //  XK_numbersign          0x023
    KEY__4,                 //  XK_dollar              0x024
    KEY__5,                 //  XK_percent             0x025
    KEY__7,                 //  XK_ampersand           0x026
    KEY__QUOTE,             //  XK_quoteright          0x027
    KEY__9,                 //  XK_parenleft           0x028
    KEY__0,                 //  XK_parenright          0x029
    KEY__8,                 //  XK_asterisk            0x02a
    KEY__EQUAL,             //  XK_plus                0x02b
    KEY__COMMA,             //  XK_comma               0x02c
    KEY__MINUS,             //  XK_minus               0x02d
    KEY__PERIOD,            //  XK_period              0x02e
    KEY__SLASH,             //  XK_slash               0x02f
    KEY__0,                 //  XK_0                   0x030
    KEY__1,                 //  XK_1                   0x031
    KEY__2,                 //  XK_2                   0x032
    KEY__3,                 //  XK_3                   0x033
    KEY__4,                 //  XK_4                   0x034
    KEY__5,                 //  XK_5                   0x035
    KEY__6,                 //  XK_6                   0x036
    KEY__7,                 //  XK_7                   0x037
    KEY__8,                 //  XK_8                   0x038
    KEY__9,                 //  XK_9                   0x039
    KEY__COLON,             //  XK_colon               0x03a
    KEY__COLON,             //  XK_semicolon           0x03b
    KEY__COMMA,             //  XK_less                0x03c
    KEY__EQUAL,             //  XK_equal               0x03d
    KEY__PERIOD,            //  XK_greater             0x03e
    KEY__SLASH,             //  XK_question            0x03f
    KEY__2,                 //  XK_at                  0x040
    KEY__A,                 //  XK_A                   0x041
    KEY__B,                 //  XK_B                   0x042
    KEY__C,                 //  XK_C                   0x043
    KEY__D,                 //  XK_D                   0x044
    KEY__E,                 //  XK_E                   0x045
    KEY__F,                 //  XK_F                   0x046
    KEY__G,                 //  XK_G                   0x047
    KEY__H,                 //  XK_H                   0x048
    KEY__I,                 //  XK_I                   0x049
    KEY__J,                 //  XK_J                   0x04a
    KEY__K,                 //  XK_K                   0x04b
    KEY__L,                 //  XK_L                   0x04c
    KEY__M,                 //  XK_M                   0x04d
    KEY__N,                 //  XK_N                   0x04e
    KEY__O,                 //  XK_O                   0x04f
    KEY__P,                 //  XK_P                   0x050
    KEY__Q,                 //  XK_Q                   0x051
    KEY__R,                 //  XK_R                   0x052
    KEY__S,                 //  XK_S                   0x053
    KEY__T,                 //  XK_T                   0x054
    KEY__U,                 //  XK_U                   0x055
    KEY__V,                 //  XK_V                   0x056
    KEY__W,                 //  XK_W                   0x057
    KEY__X,                 //  XK_X                   0x058
    KEY__Y,                 //  XK_Y                   0x059
    KEY__Z,                 //  XK_Z                   0x05a
    KEY__LBRACE,            //  XK_bracketleft         0x05b
    KEY__BSLASH,            //  XK_backslash           0x05c
    KEY__RBRACE,            //  XK_bracketright        0x05d
    KEY__6,                 //  XK_asciicircum         0x05e
    KEY__MINUS,             //  XK_underscore          0x05f
    KEY__BQUOTE,            //  XK_quoteleft           0x060
    KEY__A,                 //  XK_a                   0x061
    KEY__B,                 //  XK_b                   0x062
    KEY__C,                 //  XK_c                   0x063
    KEY__D,                 //  XK_d                   0x064
    KEY__E,                 //  XK_e                   0x065
    KEY__F,                 //  XK_f                   0x066
    KEY__G,                 //  XK_g                   0x067
    KEY__H,                 //  XK_h                   0x068
    KEY__I,                 //  XK_i                   0x069
    KEY__J,                 //  XK_j                   0x06a
    KEY__K,                 //  XK_k                   0x06b
    KEY__L,                 //  XK_l                   0x06c
    KEY__M,                 //  XK_m                   0x06d
    KEY__N,                 //  XK_n                   0x06e
    KEY__O,                 //  XK_o                   0x06f
    KEY__P,                 //  XK_p                   0x070
    KEY__Q,                 //  XK_q                   0x071
    KEY__R,                 //  XK_r                   0x072
    KEY__S,                 //  XK_s                   0x073
    KEY__T,                 //  XK_t                   0x074
    KEY__U,                 //  XK_u                   0x075
    KEY__V,                 //  XK_v                   0x076
    KEY__W,                 //  XK_w                   0x077
    KEY__X,                 //  XK_x                   0x078
    KEY__Y,                 //  XK_y                   0x079
    KEY__Z,                 //  XK_z                   0x07a
    KEY__LBRACE,            //  XK_braceleft           0x07b
    KEY__BSLASH,            //  XK_bar                 0x07c
    KEY__RBRACE,            //  XK_braceright          0x07d
    KEY__BQUOTE,            //  XK_asciitilde          0x07e
    KEY__NONE,              //  N/A                    0x07f
    KEY__NONE,              //  N/A                    0x080
    KEY__NONE,              //  N/A                    0x081
    KEY__NONE,              //  N/A                    0x082
    KEY__NONE,              //  N/A                    0x083
    KEY__NONE,              //  N/A                    0x084
    KEY__NONE,              //  N/A                    0x085
    KEY__NONE,              //  N/A                    0x086
    KEY__NONE,              //  N/A                    0x087
    KEY__NONE,              //  N/A                    0x088
    KEY__NONE,              //  N/A                    0x089
    KEY__NONE,              //  N/A                    0x08a
    KEY__NONE,              //  N/A                    0x08b
    KEY__NONE,              //  N/A                    0x08c
    KEY__NONE,              //  N/A                    0x08d
    KEY__NONE,              //  N/A                    0x08e
    KEY__NONE,              //  N/A                    0x08f
    KEY__NONE,              //  N/A                    0x090
    KEY__NONE,              //  N/A                    0x091
    KEY__NONE,              //  N/A                    0x092
    KEY__NONE,              //  N/A                    0x093
    KEY__NONE,              //  N/A                    0x094
    KEY__NONE,              //  N/A                    0x095
    KEY__NONE,              //  N/A                    0x096
    KEY__NONE,              //  N/A                    0x097
    KEY__NONE,              //  N/A                    0x098
    KEY__NONE,              //  N/A                    0x099
    KEY__NONE,              //  N/A                    0x09a
    KEY__NONE,              //  N/A                    0x09b
    KEY__NONE,              //  N/A                    0x09c
    KEY__NONE,              //  N/A                    0x09d
    KEY__NONE,              //  N/A                    0x09e
    KEY__NONE,              //  N/A                    0x09f
    KEY__NONE,              //  XK_nobreakspace        0x0a0
    KEY__NONE,              //  XK_exclamdown          0x0a1
    KEY__NONE,              //  XK_cent                0x0a2
    KEY__NONE,              //  XK_sterling            0x0a3
    KEY__NONE,              //  XK_currency            0x0a4
    KEY__NONE,              //  XK_yen                 0x0a5
    KEY__NONE,              //  XK_brokenbar           0x0a6
    KEY__NONE,              //  XK_section             0x0a7
    KEY__NONE,              //  XK_diaeresis           0x0a8
    KEY__NONE,              //  XK_copyright           0x0a9
    KEY__NONE,              //  XK_ordfeminine         0x0aa
    KEY__NONE,              //  XK_guillemotleft       0x0ab
    KEY__NONE,              //  XK_notsign             0x0ac
    KEY__NONE,              //  XK_hyphen              0x0ad
    KEY__NONE,              //  XK_registered          0x0ae
    KEY__NONE,              //  XK_macron              0x0af
    KEY__NONE,              //  XK_degree              0x0b0
    KEY__NONE,              //  XK_plusminus           0x0b1
    KEY__NONE,              //  XK_twosuperior         0x0b2
    KEY__NONE,              //  XK_threesuperior       0x0b3
    KEY__NONE,              //  XK_acute               0x0b4
    KEY__NONE,              //  XK_mu                  0x0b5
    KEY__NONE,              //  XK_paragraph           0x0b6
    KEY__NONE,              //  XK_periodcentered      0x0b7
    KEY__NONE,              //  XK_cedilla             0x0b8
    KEY__NONE,              //  XK_onesuperior         0x0b9
    KEY__NONE,              //  XK_masculine           0x0ba
    KEY__NONE,              //  XK_guillemotright      0x0bb
    KEY__NONE,              //  XK_onequarter          0x0bc
    KEY__NONE,              //  XK_onehalf             0x0bd
    KEY__NONE,              //  XK_threequarters       0x0be
    KEY__NONE,              //  XK_questiondown        0x0bf
    KEY__NONE,              //  XK_Agrave              0x0c0
    KEY__NONE,              //  XK_Aacute              0x0c1
    KEY__NONE,              //  XK_Acircumflex         0x0c2
    KEY__NONE,              //  XK_Atilde              0x0c3
    KEY__NONE,              //  XK_Adiaeresis          0x0c4
    KEY__NONE,              //  XK_Aring               0x0c5
    KEY__NONE,              //  XK_AE                  0x0c6
    KEY__NONE,              //  XK_Ccedilla            0x0c7
    KEY__NONE,              //  XK_Egrave              0x0c8
    KEY__NONE,              //  XK_Eacute              0x0c9
    KEY__NONE,              //  XK_Ecircumflex         0x0ca
    KEY__NONE,              //  XK_Ediaeresis          0x0cb
    KEY__NONE,              //  XK_Igrave              0x0cc
    KEY__NONE,              //  XK_Iacute              0x0cd
    KEY__NONE,              //  XK_Icircumflex         0x0ce
    KEY__NONE,              //  XK_Idiaeresis          0x0cf
    KEY__NONE,              //  XK_Eth                 0x0d0
    KEY__NONE,              //  XK_Ntilde              0x0d1
    KEY__NONE,              //  XK_Ograve              0x0d2
    KEY__NONE,              //  XK_Oacute              0x0d3
    KEY__NONE,              //  XK_Ocircumflex         0x0d4
    KEY__NONE,              //  XK_Otilde              0x0d5
    KEY__NONE,              //  XK_Odiaeresis          0x0d6
    KEY__NONE,              //  XK_multiply            0x0d7
    KEY__NONE,              //  XK_Ooblique            0x0d8
    KEY__NONE,              //  XK_Ugrave              0x0d9
    KEY__NONE,              //  XK_Uacute              0x0da
    KEY__NONE,              //  XK_Ucircumflex         0x0db
    KEY__NONE,              //  XK_Udiaeresis          0x0dc
    KEY__NONE,              //  XK_Yacute              0x0dd
    KEY__NONE,              //  XK_Thorn               0x0de
    KEY__NONE,              //  XK_ssharp              0x0df
    KEY__NONE,              //  XK_agrave              0x0e0
    KEY__NONE,              //  XK_aacute              0x0e1
    KEY__NONE,              //  XK_acircumflex         0x0e2
    KEY__NONE,              //  XK_atilde              0x0e3
    KEY__NONE,              //  XK_adiaeresis          0x0e4
    KEY__NONE,              //  XK_aring               0x0e5
    KEY__NONE,              //  XK_ae                  0x0e6
    KEY__NONE,              //  XK_ccedilla            0x0e7
    KEY__NONE,              //  XK_egrave              0x0e8
    KEY__NONE,              //  XK_eacute              0x0e9
    KEY__NONE,              //  XK_ecircumflex         0x0ea
    KEY__NONE,              //  XK_ediaeresis          0x0eb
    KEY__NONE,              //  XK_igrave              0x0ec
    KEY__NONE,              //  XK_iacute              0x0ed
    KEY__NONE,              //  XK_icircumflex         0x0ee
    KEY__NONE,              //  XK_idiaeresis          0x0ef
    KEY__NONE,              //  XK_eth                 0x0f0
    KEY__NONE,              //  XK_ntilde              0x0f1
    KEY__NONE,              //  XK_ograve              0x0f2
    KEY__NONE,              //  XK_oacute              0x0f3
    KEY__NONE,              //  XK_ocircumflex         0x0f4
    KEY__NONE,              //  XK_otilde              0x0f5
    KEY__NONE,              //  XK_odiaeresis          0x0f6
    KEY__NONE,              //  XK_division            0x0f7
    KEY__NONE,              //  XK_oslash              0x0f8
    KEY__NONE,              //  XK_ugrave              0x0f9
    KEY__NONE,              //  XK_uacute              0x0fa
    KEY__NONE,              //  XK_ucircumflex         0x0fb
    KEY__NONE,              //  XK_udiaeresis          0x0fc
    KEY__NONE,              //  XK_yacute              0x0fd
    KEY__NONE,              //  XK_thorn               0x0fe
    KEY__NONE,              //  XK_ydiaeresis          0x0ff
};
const Keyboard::KeyboardKey KeyboardUnixX::sm_aExKeyCodes[ ] =
{
    KEY__NONE,              //  N/A                    0xff00
    KEY__NONE,              //  N/A                    0xff01
    KEY__NONE,              //  N/A                    0xff02
    KEY__NONE,              //  N/A                    0xff03
    KEY__NONE,              //  N/A                    0xff04
    KEY__NONE,              //  N/A                    0xff05
    KEY__NONE,              //  N/A                    0xff06
    KEY__NONE,              //  N/A                    0xff07
    KEY__BS,                //  XK_BackSpace           0xff08
    KEY__TAB,               //  XK_Tab                 0xff09
    KEY__ENTER,             //  XK_Linefeed            0xff0a
    KEY__DEL,               //  XK_Clear               0xff0b
    KEY__NONE,              //  N/A                    0xff0c
    KEY__ENTER,             //  XK_Return              0xff0d
    KEY__NONE,              //  N/A                    0xff0e
    KEY__NONE,              //  N/A                    0xff0f
    KEY__NONE,              //  N/A                    0xff10
    KEY__F11,               //  XK_F11                 0xff11
    KEY__F12,               //  XK_F12                 0xff12
    KEY__BREAK,             //  XK_Pause               0xff13
    KEY__SLOCK,             //  XK_Scroll_Lock         0xff14
    KEY__SYSREQ,            //  XK_Sys_Req             0xff15
    KEY__NONE,              //  N/A                    0xff16
    KEY__NONE,              //  N/A                    0xff17
    KEY__NONE,              //  N/A                    0xff18
    KEY__NONE,              //  N/A                    0xff19
    KEY__NONE,              //  N/A                    0xff1a
    KEY__ESC,               //  XK_Escape              0xff1b
    KEY__NONE,              //  N/A                    0xff1c
    KEY__NONE,              //  N/A                    0xff1d
    KEY__NONE,              //  N/A                    0xff1e
    KEY__NONE,              //  N/A                    0xff1f
    KEY__NONE,              //  XK_Multi_key           0xff20
    KEY__NONE,              //  N/A                    0xff21
    KEY__NONE,              //  N/A                    0xff22
    KEY__NONE,              //  N/A                    0xff23
    KEY__NONE,              //  N/A                    0xff24
    KEY__NONE,              //  N/A                    0xff25
    KEY__NONE,              //  N/A                    0xff26
    KEY__NONE,              //  N/A                    0xff27
    KEY__NONE,              //  N/A                    0xff28
    KEY__NONE,              //  N/A                    0xff29
    KEY__NONE,              //  N/A                    0xff2a
    KEY__NONE,              //  N/A                    0xff2b
    KEY__NONE,              //  N/A                    0xff2c
    KEY__NONE,              //  N/A                    0xff2d
    KEY__NONE,              //  N/A                    0xff2e
    KEY__NONE,              //  N/A                    0xff2f
    KEY__NONE,              //  N/A                    0xff30
    KEY__NONE,              //  N/A                    0xff31
    KEY__NONE,              //  N/A                    0xff32
    KEY__NONE,              //  N/A                    0xff33
    KEY__NONE,              //  N/A                    0xff34
    KEY__NONE,              //  N/A                    0xff35
    KEY__NONE,              //  N/A                    0xff36
    KEY__NONE,              //  N/A                    0xff37
    KEY__NONE,              //  N/A                    0xff38
    KEY__NONE,              //  N/A                    0xff39
    KEY__NONE,              //  N/A                    0xff3a
    KEY__NONE,              //  N/A                    0xff3b
    KEY__NONE,              //  N/A                    0xff3c
    KEY__NONE,              //  N/A                    0xff3d
    KEY__NONE,              //  N/A                    0xff3e
    KEY__NONE,              //  N/A                    0xff3f
    KEY__NONE,              //  N/A                    0xff40
    KEY__NONE,              //  N/A                    0xff41
    KEY__NONE,              //  N/A                    0xff42
    KEY__NONE,              //  N/A                    0xff43
    KEY__NONE,              //  N/A                    0xff44
    KEY__NONE,              //  N/A                    0xff45
    KEY__NONE,              //  N/A                    0xff46
    KEY__NONE,              //  N/A                    0xff47
    KEY__NONE,              //  N/A                    0xff48
    KEY__NONE,              //  N/A                    0xff49
    KEY__NONE,              //  N/A                    0xff4a
    KEY__NONE,              //  N/A                    0xff4b
    KEY__NONE,              //  N/A                    0xff4c
    KEY__NONE,              //  N/A                    0xff4d
    KEY__NONE,              //  N/A                    0xff4e
    KEY__NONE,              //  N/A                    0xff4f
    KEY__HOME,              //  XK_Home                0xff50
    KEY__LEFT,              //  XK_Left                0xff51
    KEY__UP,                //  XK_Up                  0xff52
    KEY__RIGHT,             //  XK_Right               0xff53
    KEY__DOWN,              //  XK_Down                0xff54
    KEY__PGUP,              //  XK_Page_Up             0xff55
    KEY__PGDN,              //  XK_Page_Down           0xff56
    KEY__END,               //  XK_End                 0xff57
    KEY__HOME,              //  XK_Begin               0xff58
    KEY__NONE,              //  N/A                    0xff59
    KEY__NONE,              //  N/A                    0xff5a
    KEY__NONE,              //  N/A                    0xff5b
    KEY__NONE,              //  N/A                    0xff5c
    KEY__NONE,              //  N/A                    0xff5d
    KEY__NONE,              //  N/A                    0xff5e
    KEY__NONE,              //  N/A                    0xff5f
    KEY__NONE,              //  XK_Select              0xff60
    KEY__NONE,              //  XK_Print               0xff61
    KEY__ENTER,             //  XK_Execute             0xff62
    KEY__INS,               //  XK_Insert              0xff63
    KEY__NONE,              //  N/A                    0xff64
    KEY__NONE,              //  XK_Undo                0xff65
    KEY__NONE,              //  XK_Redo                0xff66
    KEY__NONE,              //  XK_Menu                0xff67
    KEY__NONE,              //  XK_Find                0xff68
    KEY__NONE,              //  XK_Cancel              0xff69
    KEY__NONE,              //  XK_Help                0xff6a
    KEY__BREAK,             //  XK_Break               0xff6b
    KEY__NONE,              //  N/A                    0xff6c
    KEY__NONE,              //  N/A                    0xff6d
    KEY__NONE,              //  N/A                    0xff6e
    KEY__NONE,              //  N/A                    0xff6f
    KEY__NONE,              //  N/A                    0xff70
    KEY__NONE,              //  N/A                    0xff71
    KEY__NONE,              //  N/A                    0xff72
    KEY__NONE,              //  N/A                    0xff73
    KEY__NONE,              //  N/A                    0xff74
    KEY__NONE,              //  N/A                    0xff75
    KEY__NONE,              //  N/A                    0xff76
    KEY__NONE,              //  N/A                    0xff77
    KEY__NONE,              //  N/A                    0xff78
    KEY__NONE,              //  N/A                    0xff79
    KEY__NONE,              //  N/A                    0xff7a
    KEY__NONE,              //  N/A                    0xff7b
    KEY__NONE,              //  N/A                    0xff7c
    KEY__NONE,              //  N/A                    0xff7d
    KEY__NONE,              //  XK_Mode_switch         0xff7e
    KEY__NLOCK,             //  XK_Num_Lock            0xff7f
    KEY__NONE,              //  XK_KP_Space            0xff80
    KEY__NONE,              //  N/A                    0xff81
    KEY__NONE,              //  N/A                    0xff82
    KEY__NONE,              //  N/A                    0xff83
    KEY__NONE,              //  N/A                    0xff84
    KEY__NONE,              //  N/A                    0xff85
    KEY__NONE,              //  N/A                    0xff86
    KEY__NONE,              //  N/A                    0xff87
    KEY__NONE,              //  N/A                    0xff88
    KEY__TAB,               //  XK_KP_Tab              0xff89
    KEY__NONE,              //  N/A                    0xff8a
    KEY__NONE,              //  N/A                    0xff8b
    KEY__NONE,              //  N/A                    0xff8c
    KEY__ENTER,             //  XK_KP_Enter            0xff8d
    KEY__NONE,              //  N/A                    0xff8e
    KEY__NONE,              //  N/A                    0xff8f
    KEY__NONE,              //  N/A                    0xff90
    KEY__F1,                //  XK_KP_F1               0xff91
    KEY__F2,                //  XK_KP_F2               0xff92
    KEY__F3,                //  XK_KP_F3               0xff93
    KEY__F4,                //  XK_KP_F4               0xff94
    KEY__HOME,              //  XK_KP_Home             0xff95
    KEY__LEFT,              //  XK_KP_Left             0xff96
    KEY__UP,                //  XK_KP_Up               0xff97
    KEY__RIGHT,             //  XK_KP_Right            0xff98
    KEY__DOWN,              //  XK_KP_Down             0xff99
    KEY__PGUP,              //  XK_KP_Page_Up          0xff9a
    KEY__PGDN,              //  XK_KP_Page_Down        0xff9b
    KEY__END,               //  XK_KP_End              0xff9c
    KEY__HOME,              //  XK_KP_Begin            0xff9d
    KEY__INS,               //  XK_KP_Insert           0xff9e
    KEY__DEL,               //  XK_KP_Delete           0xff9f
    KEY__NONE,              //  N/A                    0xffa0
    KEY__NONE,              //  N/A                    0xffa1
    KEY__NONE,              //  N/A                    0xffa2
    KEY__NONE,              //  N/A                    0xffa3
    KEY__NONE,              //  N/A                    0xffa4
    KEY__NONE,              //  N/A                    0xffa5
    KEY__NONE,              //  N/A                    0xffa6
    KEY__NONE,              //  N/A                    0xffa7
    KEY__NONE,              //  N/A                    0xffa8
    KEY__NONE,              //  N/A                    0xffa9
    KEY__MULT,              //  XK_KP_Multiply         0xffaa
    KEY__ADD,               //  XK_KP_Add              0xffab
    KEY__NONE,              //  XK_KP_Separator        0xffac
    KEY__SUBT,              //  XK_KP_Subtract         0xffad
    KEY__DEL,               //  XK_KP_Decimal          0xffae
    KEY__DIV,               //  XK_KP_Divide           0xffaf
    KEY__0,                 //  XK_KP_0                0xffb0
    KEY__1,                 //  XK_KP_1                0xffb1
    KEY__2,                 //  XK_KP_2                0xffb2
    KEY__3,                 //  XK_KP_3                0xffb3
    KEY__4,                 //  XK_KP_4                0xffb4
    KEY__5,                 //  XK_KP_5                0xffb5
    KEY__6,                 //  XK_KP_6                0xffb6
    KEY__7,                 //  XK_KP_7                0xffb7
    KEY__8,                 //  XK_KP_8                0xffb8
    KEY__9,                 //  XK_KP_9                0xffb9
    KEY__NONE,              //  N/A                    0xffba
    KEY__NONE,              //  N/A                    0xffbb
    KEY__NONE,              //  N/A                    0xffbc
    KEY__EQUAL,             //  XK_KP_Equal            0xffbd
    KEY__F1,                //  XK_F1                  0xffbe
    KEY__F2,                //  XK_F2                  0xffbf
    KEY__F3,                //  XK_F3                  0xffc0
    KEY__F4,                //  XK_F4                  0xffc1
    KEY__F5,                //  XK_F5                  0xffc2
    KEY__F6,                //  XK_F6                  0xffc3
    KEY__F7,                //  XK_F7                  0xffc4
    KEY__F8,                //  XK_F8                  0xffc5
    KEY__F9,                //  XK_F9                  0xffc6
    KEY__F10,               //  XK_F10                 0xffc7
    KEY__F11,               //  XK_F11                 0xffc8
    KEY__F12,               //  XK_F12                 0xffc9
    KEY__NONE,              //  XK_F13                 0xffca
    KEY__NONE,              //  XK_F14                 0xffcb
    KEY__NONE,              //  XK_F15                 0xffcc
    KEY__NONE,              //  XK_F16                 0xffcd
    KEY__NONE,              //  XK_F17                 0xffce
    KEY__NONE,              //  XK_F18                 0xffcf
    KEY__NONE,              //  XK_F19                 0xffd0
    KEY__NONE,              //  XK_F20                 0xffd1
    KEY__NONE,              //  XK_F21                 0xffd2
    KEY__NONE,              //  XK_F22                 0xffd3
    KEY__NONE,              //  XK_F23                 0xffd4
    KEY__NONE,              //  XK_F24                 0xffd5
    KEY__NONE,              //  XK_F25                 0xffd6
    KEY__NONE,              //  XK_F26                 0xffd7
    KEY__NONE,              //  XK_F27                 0xffd8
    KEY__NONE,              //  XK_F28                 0xffd9
    KEY__NONE,              //  XK_F29                 0xffda
    KEY__NONE,              //  XK_F30                 0xffdb
    KEY__NONE,              //  XK_F31                 0xffdc
    KEY__NONE,              //  XK_F32                 0xffdd
    KEY__NONE,              //  XK_F33                 0xffde
    KEY__NONE,              //  XK_F34                 0xffdf
    KEY__NONE,              //  XK_F35                 0xffe0
    KEY__LSHIFT,            //  XK_Shift_L             0xffe1
    KEY__RSHIFT,            //  XK_Shift_R             0xffe2
    KEY__LCTRL,             //  XK_Control_L           0xffe3
    KEY__RCTRL,             //  XK_Control_R           0xffe4
    KEY__CLOCK,             //  XK_Caps_Lock           0xffe5
    KEY__CLOCK,             //  XK_Shift_Lock          0xffe6
    KEY__LALT,              //  XK_Meta_L              0xffe7
    KEY__RALT,              //  XK_Meta_R              0xffe8
    KEY__LALT,              //  XK_Alt_L               0xffe9
    KEY__RALT,              //  XK_Alt_R               0xffea
    KEY__NONE,              //  XK_Super_L             0xffeb
    KEY__NONE,              //  XK_Super_R             0xffec
    KEY__NONE,              //  XK_Hyper_L             0xffed
    KEY__NONE,              //  XK_Hyper_R             0xffee
    KEY__NONE,              //  N/A                    0xffef
    KEY__NONE,              //  N/A                    0xfff0
    KEY__NONE,              //  N/A                    0xfff1
    KEY__NONE,              //  N/A                    0xfff2
    KEY__NONE,              //  N/A                    0xfff3
    KEY__NONE,              //  N/A                    0xfff4
    KEY__NONE,              //  N/A                    0xfff5
    KEY__NONE,              //  N/A                    0xfff6
    KEY__NONE,              //  N/A                    0xfff7
    KEY__NONE,              //  N/A                    0xfff8
    KEY__NONE,              //  N/A                    0xfff9
    KEY__NONE,              //  N/A                    0xfffa
    KEY__NONE,              //  N/A                    0xfffb
    KEY__NONE,              //  N/A                    0xfffc
    KEY__NONE,              //  N/A                    0xfffd
    KEY__NONE,              //  N/A                    0xfffe
    KEY__DEL,               //  XK_Delete              0xffff
};



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a Unix/X Keyboard object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
KeyboardUnixX*
KeyboardUnixX::s_build(
    const KString& iName
)
{
    //  Create the new object.
    KeyboardUnixX* pThis = new KeyboardUnixX( iName );
 
    //  Initialize the new object.
    pThis->init( );

    //  Send back the new keyboard.
    return( pThis );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: KeyboardUnixX
//
//  Description:
//
//      This is the main constructor for the Unix/X Keyboard object.  This is
//      a protected member.  Clients wishing to create a keyboard object should
//      do so through the factory method.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KeyboardUnixX::KeyboardUnixX(
    const KString&  iName
)
:
    Keyboard   ( iName ),
    m_pCanvas  ( NULL )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the keyboard object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
KeyboardUnixX::init(
)
{
    //  Call the base class.
    Keyboard::init( );


    //  Get the canvas of the application.
    Canvas* pCanvas = Replay::s_instance( ).getCanvas( );

    //  Make sure that there is a canvas.
    ASSERT( pCanvas != NULL );

    //  Save the canvas for later.
    m_pCanvas = ( CanvasUnixX* )pCanvas;

    //  We don't want auto-repeat while the game is running.
    XAutoRepeatOff( m_pCanvas->getDisplay( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~KeyboardUnixX
//
//  Description:
//
//      This is the destructor for a Unix/X keyboard object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KeyboardUnixX::~KeyboardUnixX(
)
{
    //  Turn auto-repeat back on.
    XAutoRepeatOn( m_pCanvas->getDisplay( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the keyboard object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
KeyboardUnixX::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "KeyboardUnixX" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member should be called periodically to allow the
//      keyboard object to update itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if a key state has changed.
//      FALSE if no key state has changed.
//
///////////////////////////////////////////////////////////////////////////////
Byte
KeyboardUnixX::update(
)
{
    //  Get the display and the window.
    Display* pDisplay = m_pCanvas->getDisplay( );
    Window   window   = m_pCanvas->getCurWindow( );

    //  Indicates whether or not a key state has changed.
    Byte bKeyChanged;

    //  The next X-event.
    XEvent event;

    //  The key involved in the event.
    KeySym key;

    //  The keyboard key to modify.
    KeyboardKey eKey;

    //  Loop while there are more key events.
    for(
        bKeyChanged = FALSE ;
        XCheckWindowEvent(
            pDisplay,
            window,
            KeyPressMask | KeyReleaseMask,
            &event
        ) == True ;
    )
    {
        //  Look up the key involved.
        key = XLookupKeysym( ( XKeyEvent* )&event, 0 );

        //  Is the key extended?
        if( ( key <= 0xff ) )
        {
            //  The key is not extended therefore use the regular table.
            eKey = sm_aKeyCodes[ key ];
        }
        else
        {
            //  The key is extended therefore use the extended table and mask
            //  out the upper 8 bits.
            eKey = sm_aExKeyCodes[ key & 0xff ];
        }

        //  Set the state of the key and note the last key pressed.
        if( event.type == KeyPress )
        {
            m_switchList[ eKey ]->physicalOn( TRUE );
        }
        else
        {
            m_switchList[ eKey ]->physicalOn( FALSE );
            m_dwLastSwitchOn = eKey;
        }
    }
    
    //  Return the status.
    return( bKeyChanged );
}
