///////////////////////////////////////////////////////////////////////////////
//
//  File:    repfactx.cpp
//
//  Class:   ReplayFactoryUnixX
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is a concrete factory that serves as a default factory
//      for those Unix/X platforms that do not have any special processing.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "repfactx.h"
#include "canvasx.h"
#include "keybx.h"
#include "joyx.h"
#include "mousex.h"
#include "soundx.h"
#include "diskx.h"
#include "networkx.h"
#include "clockx.h"
#include "6502mfc.h"
#include "6809kbc.h"
#include "z80mdc.h"
#include "68000bsc.h"
#ifdef GAMES_shared
#include "gamfactx.h"
#else
#include "gamfact.h"
#endif


///////////////////////////////////////////////////////////////////////////////
//  File Static Variables.
///////////////////////////////////////////////////////////////////////////////

//  The singleton instance.
ReplayFactoryUnixX ReplayFactoryUnixX::sm_replayFactoryUnixX( "ReplayFactory" );


///////////////////////////////////////////////////////////////////////////////
//
//  Function: ReplayFactoryUnixX
//
//  Description:
//
//      This is the main constructor for the Unix/X factory object.  It
//      is protected because it is a singleton and therefore cannot be
//      instantiated by anyone but itself or a derived class.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ReplayFactoryUnixX::ReplayFactoryUnixX(
    const KString& iName
)
:
    ReplayFactory  ( iName )
{
    //  Nothing to do.
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the replay factory object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
ReplayFactoryUnixX::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "ReplayFactoryUnixX" );

    return( className );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~ReplayFactoryUnixX
//
//  Description:
//
//      This is the main destructor for the Unix/X Replay factory object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
ReplayFactoryUnixX::~ReplayFactoryUnixX(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: createCanvas
//
//  Description:
//
//      This creates the default canvas object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the canvas.
//
///////////////////////////////////////////////////////////////////////////////
Canvas*
ReplayFactoryUnixX::createCanvas(
    const KString& iName
)
{
    return( CanvasUnixX::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createKeyboard
//
//  Description:
//
//      This creates the default keyboard object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the keyboard.
//
///////////////////////////////////////////////////////////////////////////////
Keyboard*
ReplayFactoryUnixX::createKeyboard(
    const KString& iName
)
{
    return( KeyboardUnixX::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createJoystick
//
//  Description:
//
//      This creates the default joystick object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the joystick.
//
///////////////////////////////////////////////////////////////////////////////
Joystick*
ReplayFactoryUnixX::createJoystick(
    const KString& iName
)
{
    return( JoystickUnixX::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createMouse
//
//  Description:
//
//      This creates the default mouse object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the mouse.
//
///////////////////////////////////////////////////////////////////////////////
Mouse*
ReplayFactoryUnixX::createMouse(
    const KString& iName
)
{
    return( MouseUnixX::s_build( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createSound
//
//  Description:
//
//      This creates the default sound object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the sound.
//
///////////////////////////////////////////////////////////////////////////////
Sound*
ReplayFactoryUnixX::createSound(
    const KString& iName
)
{
    return( new SoundUnixX( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createDisk
//
//  Description:
//
//      This creates the default disk object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the disk.
//
///////////////////////////////////////////////////////////////////////////////
Disk*
ReplayFactoryUnixX::createDisk(
    const KString& iName
)
{
    return( new DiskUnixX( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createNetwork
//
//  Description:
//
//      This creates the default network object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the network.
//
///////////////////////////////////////////////////////////////////////////////
Network*
ReplayFactoryUnixX::createNetwork(
    const KString& iName
)
{
    return( new NetworkUnixX( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createClock
//
//  Description:
//
//      This creates the default clock object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the clock.
//
///////////////////////////////////////////////////////////////////////////////
Clock*
ReplayFactoryUnixX::createClock(
    const KString& iName
)
{
    return( new ClockUnixX( iName ) );
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createGameFactory
//
//  Description:
//
//      This creates the default game factory object for Unix/X.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the game factory.
//
///////////////////////////////////////////////////////////////////////////////
GameFactory*
ReplayFactoryUnixX::createGameFactory(
    const KString& iName
)
{
#ifdef GAMES_shared
    return( GameFactoryUnixX::s_build( iName ) );
#else
    return( GameFactory::s_build( iName ) );
#endif
}

///////////////////////////////////////////////////////////////////////////////
//
//  Function: createCPU
//
//  Description:
//
//      This creates CPUs for use with the Unix/X platform.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      nCPUType (input)
//          The type of CPU to create.
//
//      pGame (input)
//          The game that the CPU belongs to.
//
//      pSpace (input)
//          The address space to provide to the new CPU.
//
//      ePurpose (input)
//          The purpose of CPU to create.
//
//      bCONSERVATIVE (input)
//          A TEMPORARY FLAG USED TO ALLOW US TO PICK THE 'C' BASED Z80
//          CORE DUE TO THE INSTABILITY OF THE ASM.
//
//  Returns:
//
//      A pointer to the CPU.
//
///////////////////////////////////////////////////////////////////////////////
CPU*
ReplayFactoryUnixX::createCPU(
    const KString&        iName,
    const CPU::CPUType    eCPUType,
    Game*                 pGame,
    AddrSpace*            pSpace,
    const CPU::CPUPurpose ePurpose      /* = CPU::GAME */,
    const Byte            /* bConservative is unused */ /* = FALSE */
)
{
    ASSERT( eCPUType < CPU::NUM_CPU_TYPES );
 
    //  Create the appropriate CPU.  Since this factory is used for generic
    //  Unix/X platforms we have to stick with the generic 'C' versions of the
    //  CPU cores.
    switch( eCPUType )
    {
        case CPU::CPU_6502:
        {
            return( CPU6502MFC::s_build( iName, pGame, pSpace, ePurpose ) );
        }
        
        case CPU::CPU_6809:
        {
            return( CPU6809KBC::s_build( iName, pGame, pSpace, ePurpose ) );
        } 

        case CPU::CPU_68000:
        {
            return( CPU68000BSC::s_build( iName, pGame, pSpace, ePurpose ) );
        }

        case CPU::CPU_Z80:
        {
            return( CPUZ80MDC::s_build( iName, pGame, pSpace, ePurpose ) );
        }

        default:
        {
            fatalError( 
                "CPUFactory::createCPU - unknown CPU Type (%d).", eCPUType
            );
        }
    }

    //  Should never arrive here.
    return( NULL );
}


