///////////////////////////////////////////////////////////////////////////////
//
//  File:    dcontrol.cpp
//
//  Class:   DigitalController
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The DigitalController class is an abstract base class representing a
//      digital controller object such as a digital joystick, keyboard, etc.
//      The characteristics of a digital controller is that it consists of
//      a number of switches (i.e. keys, buttons, joystick directions, etc.)
//      which can take on one of two states (on or off).
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "dcontrol.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: DigitalController
//
//  Description:
//
//      This is the main constructor for a digital controller object.  
//      It is protected so that only derived classes have access to it.  
//      Clients should use a build method to create a digital controller 
//      object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DigitalController::DigitalController(
    const KString&  iName
)
:
    Controller           ( iName ),
    m_bMappingsEnabled   ( TRUE ),
    m_dwLastSwitchOn     ( SWITCH_NONE ),
    m_switchList         ( 10 )
{
    //  Initialization is to be done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called from the build method to initialize the object.
//      It allows virtual methods to be called during construction.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DigitalController::init(
)
{
    //  Let the base class initialize.
    Controller::init( );

    //  Create the list of switches.
    for( DWord dwI = 0 ; dwI < getNumSwitches( ) ; dwI += 1 )
    {
        m_switchList.add( new Switch( switchName( dwI ), this ) );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DigitalController
//
//  Description:
//
//      This is the destructor for a digital controller object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DigitalController::~DigitalController(
)
{
    //  Clean up the switches.
    for( DWord dwI = 0 ; dwI < m_switchList.entries( ) ; dwI += 1 )
    {
        delete m_switchList[ dwI ];
    }
    m_switchList.clear( );

}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: enableMappings
//
//  Description:
//
//      This is used as a convenience to enable/disable the mappings for
//      all switches belonging to this controller.
//
//  Parameters:
//
//      bEnable (input)
//          Should the mappings be enabled or disabled?
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DigitalController::enableMappings(
    Byte bEnabled
)
{
    //  Set our local flag.
    m_bMappingsEnabled = bEnabled;

    //  Do the same for each switch.
    for( DWord dwI = 0 ; dwI < getNumSwitches( ) ; dwI += 1 )
    {
        m_switchList[ dwI ]->enableMappings( bEnabled );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clearMappings
//
//  Description:
//
//      This is used as a convenience to clear all of the mappings for
//      all switches belonging to this controller.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DigitalController::clearMappings(
)
{
    //  Free the mappings for each switch.
    for( DWord dwI = 0 ; dwI < getNumSwitches( ) ; dwI += 1 )
    {
        m_switchList[ dwI ]->clearMappings( );
    }
}
