///////////////////////////////////////////////////////////////////////////////
//
//  File:       dcontrol.h
//
//  Class:      DigitalController - Abstract
//  
//  Hierarchy:  DigitalController - Controller - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      The DigitalController class is an abstract base class representing a
//      digital controller object such as a digital joystick, keyboard, etc.
//      The characteristics of a digital controller is that it consists of
//      a number of switches (i.e. keys, buttons, joystick directions, etc.)
//      which can take on one of two states (on or off).
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _dcontrol_h_
#define _dcontrol_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "control.h"
#include "kstring.h"
#include "kplist.h"
#include "switch.h"


///////////////////////////////////////////////////////////////////////////////
//  DigitalController Class Definition
///////////////////////////////////////////////////////////////////////////////
class DigitalController : public Controller
{
    public:

        typedef enum
        {
            SWITCH_NONE = 0
        }
        DigitalControllerSwitch;

        //  Deletion.
        virtual ~DigitalController ( );

        //  Access.
        virtual DWord         getNumSwitches    ( ) const = 0;
        virtual const char*   switchName        ( DWord dwSwitch ) const = 0;
        Switch*               getSwitch         ( DWord dwSwitch ) const;
        Byte                  switchOn          ( DWord dwSwitch ) const;
        Byte                  switchOnWait      ( DWord dwSwitch ) const;
        void                  waitUntilOff      ( DWord dwSwitch ) const;
        DWord                 lastSwitchOn      ( ) const;
        void                  clearLastSwitchOn ( );
        void                  clearVirtual      ( );

        //  Control Mapping Interface.
        void enableMappings  ( Byte bEnable );
        Byte mappingsEnabled ( ) const;
        void clearMappings   ( );
        void addMapping      ( DWord dwSwitch, Byte* pbLocation, Byte bMask );

    protected:

        //  Creation is protected.  Clients should use a build method.
        DigitalController ( const KString& iName );

        //  This function should be called from the build method.
        virtual void init ( );


        //  Data Members.
        Byte             m_bMappingsEnabled;
        DWord            m_dwLastSwitchOn;
        KPtrList<Switch> m_switchList;

    private:

};

///////////////////////////////////////////////////////////////////////////////
//  DigitalController Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Switch*
DigitalController::getSwitch(
    DWord dwSwitch
) const
{
    ASSERT( dwSwitch < getNumSwitches( ) );

    return( m_switchList[ dwSwitch ] );
}

inline
Byte
DigitalController::switchOn(
    DWord dwSwitch
) const
{
    ASSERT( dwSwitch < getNumSwitches( ) );

    return( m_switchList[ dwSwitch ]->isOn( ) );
}

inline
Byte
DigitalController::switchOnWait(
    DWord dwSwitch
) const
{
    ASSERT( dwSwitch < getNumSwitches( ) );

    return( m_switchList[ dwSwitch ]->isOnWait( ) );
}

inline
void
DigitalController::waitUntilOff(
    DWord dwSwitch
) const
{
    ASSERT( dwSwitch < getNumSwitches( ) );

    m_switchList[ dwSwitch ]->waitUntilOff( );
}

inline
DWord
DigitalController::lastSwitchOn(
) const
{
    return( m_dwLastSwitchOn );
}

inline
void
DigitalController::clearLastSwitchOn(
)
{
    m_dwLastSwitchOn = SWITCH_NONE;
}

inline
void
DigitalController::clearVirtual(
)
{
    for( DWord dwI = 0 ; dwI < m_switchList.entries( ) ; dwI += 1 )
    {
        m_switchList[ dwI ]->virtualOn( FALSE );
    }
}

inline
Byte
DigitalController::mappingsEnabled(
) const
{
    return( m_bMappingsEnabled );
}

inline
void
DigitalController::addMapping(
    DWord dwSwitch,
    Byte* pbLocation,
    Byte  bMask
)
{
    ASSERT( dwSwitch < getNumSwitches( ) );

    m_switchList[ dwSwitch ]->addMapping( pbLocation, bMask );
}

#endif
