///////////////////////////////////////////////////////////////////////////////
//
//  File:    keyb.cpp
//
//  Class:   Keyboard
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Keyboard class encapsulates keyboard access.  This is an abstract
//      class that serves as a base for derived keyboard classes for each
//      platform supported.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "keyb.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data Initialization.
///////////////////////////////////////////////////////////////////////////////
//  The names of the keys.  This is declared as an an array of indeterminate
//  size instead of size KEY__COUNT so that we can check to make sure that
//  all keys have names.  If we had specified the size of KEY__COUNT we could
//  have left entries out of this list without the compiler complaining.  
//  Instead, an ASSERT in the constructor will catch any size 
//  mismatches.
const char* Keyboard::sm_pstrKeyNames[ ] =
{
    "No Key",
    
    "0",         "1",      "2",       "3",       "4",
    "5",         "6",      "7",       "8",       "9",

    "A",         "B",      "C",       "D",       "E",
    "F",         "G",      "H",       "I",       "J",
    "K",         "L",      "M",       "N",       "O",
    "P",         "Q",      "R",       "S",       "T",
    "U",         "V",      "W",       "X",       "Y",
    "Z",

    "Space",     "\"",     "=",       ",",       "-",
    ".",         "/",      ":",       "[",       "]",
    "\\",        "`",

    "BSpace",    "Tab",    "Enter",   "Pause",   "ScrLock",
    "SysReq",    "Ins",    "Del",     "Home",    "End",
    "PgUp",      "PgDn",   "Left",    "Right",   "Up",
    "Down",      "Esc",    "Break",   "NumLock",
    "LShift",    "RShift", "LCtrl",   "RCtrl",   "CapsLock",
    "LAlt",      "RAlt",

    "+",         "-",      "*",       "/",

    "F1",        "F2",     "F3",      "F4",      "F5",
    "F6",        "F7",     "F8",      "F9",      "F10",
    "F11",       "F12"
};



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Keyboard
//
//  Description:
//
//      This is the main constructor for a keyboard object.  
//      It is protected so that only derived classes have access to it.  
//      Clients should use a build method to create a keyboard object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Keyboard::Keyboard(
    const KString&  iName
)
:
    DigitalController ( iName )
{
    //  Check to make sure we have descriptions for each key.
    ASSERT( 
        sizeof( sm_pstrKeyNames ) / sizeof( sm_pstrKeyNames[ 0 ] ) == KEY__COUNT
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Keyboard
//
//  Description:
//
//      This is the destructor for a keyboard object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Keyboard::~Keyboard(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getNumSwitches
//
//  Description:
//
//      This returns the number of switches(keys) on the keyboard.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The number of switches(keys).
//
///////////////////////////////////////////////////////////////////////////////
DWord
Keyboard::getNumSwitches(
) const
{
    return( KEY__COUNT );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: switchName
//
//  Description:
//
//      This returns the name of the specified switch(key).
//
//  Parameters:
//
//      dwSwitch (input)
//          The switch(key) of interest.
//
//  Returns:
//
//      The name of the switch(key).
//
///////////////////////////////////////////////////////////////////////////////
const
char*
Keyboard::switchName(
    DWord dwSwitch
) const
{
    ASSERT( dwSwitch < getNumSwitches( ) );

    //  Return the switch(key) name.
    return( sm_pstrKeyNames[ dwSwitch ] );
}
