///////////////////////////////////////////////////////////////////////////////
//
//  File:    switch.cpp
//
//  Class:   Switch
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The Switch class encapsulates a single digital control on a digital
//      controller (e.g. key, button, joystick direction, etc.).  A switch
//      can take on one of two possible states (on, off).
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "switch.h"
#include "replay.h"
#include "clock.h"
#include "bytemod.h"
#include "dcontrol.h"



///////////////////////////////////////////////////////////////////////////////
//  Static Data Initialization.
///////////////////////////////////////////////////////////////////////////////
const DWord Switch::sm_dwWaitPeriod = 500000;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Switch
//
//  Description:
//
//      This is the main constructor for a switch object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Switch::Switch(
    const KString&        iName,
    DigitalController*    pOwner
)
:
    RepBase              ( iName ),
    m_pOwner             ( pOwner ),
    m_bOnPhysically      ( FALSE ),
    m_bOnVirtually       ( FALSE ),
    m_bWaiting           ( FALSE ),
    m_dwWaitStart        ( 0 ),
    m_bMappingsEnabled   ( TRUE ),
    m_byteModifierList   ( 5 ),
    m_bHardwareEnabled   ( TRUE )
{
    ASSERT( m_pOwner != NULL );

    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Switch
//
//  Description:
//
//      This is the destructor for a switch object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Switch::~Switch(
)
{
    //  Clean up the byte modifiers for the switch.
    clearMappings( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the switch object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Switch::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Switch" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: isOnWait
//
//  Description:
//
//      This member indicates whether or not the switch is on taking into
//      account a waiting period to simulate repeat (like key repeat).
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if the switch is on.
//      FALSE if the switch is not on.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Switch::isOnWait(
)
{
    //  Make sure the controller is up to date.
    m_pOwner->update( );

    //  Is the switch on?
    if( isOn( ) )
    {
        //  A convenience pointer to the clock.
        Clock* pClock = Replay::s_instance( ).getClock( );

        //  Has the waiting period expired.
        Byte bWaitingExpired = 
            ( pClock->getTime( ) - m_dwWaitStart ) > sm_dwWaitPeriod;

        
        //  If we're waiting for the auto-repeat period then we 
        //  check whether or not the period has  expired and return
        //  the appropriate value.
        if( m_bWaiting )
        {
            if( bWaitingExpired )
            {
                return( TRUE );
            }
            else
            {
                return( FALSE );
            }
        }
        //  If we're not waiting for the auto-repeat period then this is the
        //  first time the switch has been checked since it was turned on so 
        //  we indicate that the switch is on but set up a waiting period
        //  during which it will not indicate as on.
        else
        {
            m_dwWaitStart = pClock->getTime( );
            m_bWaiting = TRUE; 
            return( TRUE );
        }
    }
    else
    {
        //  If it ain't on then it ain't on.
        return( FALSE );
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: waitUntilOff
//
//  Description:
//
//      This member is called to pause until the switch turns off.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Switch::waitUntilOff(
) const
{
    //  While the switch is on update the owner.
    while( isOn( ) )
    {
        m_pOwner->update( );
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: set
//
//  Description:
//
//      This member is called to update the mapping values based on the
//      state of the switch.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Switch::set(
)
{
    //  Always reset the waiting state when the state of the switch changes.
    m_bWaiting = FALSE;

    //  Perform the mappings for this switch if mappings are enabled.
    if( m_bMappingsEnabled )
    {
        //  Loop through each byte modifying any where appropriate.
        for( DWord dwI = 0 ; dwI < m_byteModifierList.entries( ) ; dwI += 1 )
        {
            //  Modify the byte based on whether the switch is on or not.
            m_byteModifierList[ dwI ]->modify( isOn( ) );
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: clearMappings
//
//  Description:
//
//      This function is used to clear the mappings for the switch (these
//      are kept in the list of bytes that are modified when the switch is
//      toggled).
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Switch::clearMappings(
)
{
    //  Clean up the byte modifiers for the switch.
    for( DWord dwI = 0 ; dwI < m_byteModifierList.entries( ) ; dwI += 1 )
    {
        delete m_byteModifierList[ dwI ];
    }
    m_byteModifierList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setMapping
//
//  Description:
//
//      This function is used to add a mapping to the switch.  A mapping
//      is simply an indication of a byte to modify when the switch 
//      is toggled.
//
//  Parameters:
//
//      pbLocation (input)
//          The location of the byte that is to be modified.
//
//      bMask (input)
//          The bits to modify.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
Switch::addMapping( 
    Byte* pbLocation, 
    Byte  bMask
)
{
    m_byteModifierList.add( new ByteModifier( pbLocation, bMask ) );
}
