///////////////////////////////////////////////////////////////////////////////
//
//  File:    cpu6502.cpp
//
//  Class:   CPU6502
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class serves as a base class for any 6502 CPU emulation cores.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#ifdef DEBUGGER
#include <stdio.h>
#endif

//  Application Headers.
#include "reptypes.h"
#include "cpu6502.h"
#ifdef DEBUGGER
#include "space.h"
#endif


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
#ifdef DEBUGGER
//  The following table is a listing of all the possible 6502 mnemonics.
char* CPU6502::sm_astrMnemonics[ 56 ] =
{
    "adc", "and", "asl", "bcc", "bcs", "beq", "bit", "bmi",
    "bne", "bpl", "brk", "bvc", "bvs", "clc", "cld", "cli",
    "clv", "cmp", "cpx", "cpy", "dec", "dex", "dey", "inx",
    "iny", "eor", "inc", "jmp", "jsr", "lda", "nop", "ldx",
    "ldy", "lsr", "ora", "pha", "php", "pla", "plp", "rol",
    "ror", "rti", "rts", "sbc", "sta", "stx", "sty", "sec",
    "sed", "sei", "tax", "tay", "txa", "tya", "tsx", "txs"
};

//  The following table is used to retrieve the format string for disassembling
//  an instruction based on the AddressMode of the instruction.
char* CPU6502::sm_astrFormats[ 14 ] =
{
    /* AC */ "%s a",
    /* IL */ "%s",
    /* IM */ "%s #$%02x",
    /* AB */ "%s $%04x",
    /* ZP */ "%s $%02x",
    /* ZX */ "%s $%02x,x",
    /* ZY */ "%s $%02x,y",
    /* AX */ "%s $%04x,x",
    /* AY */ "%s $%04x,y",
    /* RL */ "%s $%04x",
    /* IX */ "%s ($%02x,x)",
    /* IY */ "%s ($%02x),y",
    /* IN */ "%s ($%04x)",
    /* NO */ "---"
};
 
//  This table is used to hold the indices into the mnemonic name array and the
//  address mode for each possible opcode.  The two pieces of information are
//  stored consecutively in the table.
Byte CPU6502::sm_abMnAm[ 512 ] =
{
    10, CPU6502::IL, 34, CPU6502::IX,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 34, CPU6502::ZP,  2, CPU6502::ZP,  0, CPU6502::NO,
    36, CPU6502::IL, 34, CPU6502::IM,  2, CPU6502::AC,  0, CPU6502::NO, 
     0, CPU6502::NO, 34, CPU6502::AB,  2, CPU6502::AB,  0, CPU6502::NO,
     9, CPU6502::RL, 34, CPU6502::IY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 34, CPU6502::ZX,  2, CPU6502::ZX,  0, CPU6502::NO,
    13, CPU6502::IL, 34, CPU6502::AY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 34, CPU6502::AX,  2, CPU6502::AX,  0, CPU6502::NO,
    28, CPU6502::AB,  1, CPU6502::IX,  0, CPU6502::NO,  0, CPU6502::NO,  
     6, CPU6502::ZP,  1, CPU6502::ZP, 39, CPU6502::ZP,  0, CPU6502::NO,
    38, CPU6502::IL,  1, CPU6502::IM, 39, CPU6502::AC,  0, CPU6502::NO,  
     6, CPU6502::AB,  1, CPU6502::AB, 39, CPU6502::AB,  0, CPU6502::NO,
     7, CPU6502::RL,  1, CPU6502::IY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO,  1, CPU6502::ZX, 39, CPU6502::ZX,  0, CPU6502::NO,
    47, CPU6502::IL,  1, CPU6502::AY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO,  1, CPU6502::AX, 39, CPU6502::AX,  0, CPU6502::NO,
    41, CPU6502::IL, 25, CPU6502::IX,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 25, CPU6502::ZP, 33, CPU6502::ZP,  0, CPU6502::NO,
    35, CPU6502::IL, 25, CPU6502::IM, 33, CPU6502::AC,  0, CPU6502::NO, 
    27, CPU6502::AB, 25, CPU6502::AB, 33, CPU6502::AB,  0, CPU6502::NO,
    11, CPU6502::RL, 25, CPU6502::IY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 25, CPU6502::ZX, 33, CPU6502::ZX,  0, CPU6502::NO,
    15, CPU6502::IL, 25, CPU6502::AY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 25, CPU6502::AX, 33, CPU6502::AX,  0, CPU6502::NO,
    42, CPU6502::IL,  0, CPU6502::IX,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO,  0, CPU6502::ZP, 40, CPU6502::ZP,  0, CPU6502::NO,
    37, CPU6502::IL,  0, CPU6502::IM, 40, CPU6502::AC,  0, CPU6502::NO, 
    27, CPU6502::IN,  0, CPU6502::AB, 40, CPU6502::AB,  0, CPU6502::NO,
    12, CPU6502::RL,  0, CPU6502::IY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO,  0, CPU6502::ZX, 40, CPU6502::ZX,  0, CPU6502::NO,
    49, CPU6502::IL,  0, CPU6502::AY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO,  0, CPU6502::AX, 40, CPU6502::AX,  0, CPU6502::NO,
     0, CPU6502::NO, 44, CPU6502::IX,  0, CPU6502::NO,  0, CPU6502::NO, 
    46, CPU6502::ZP, 44, CPU6502::ZP, 45, CPU6502::ZP,  0, CPU6502::NO,
    22, CPU6502::IL,  0, CPU6502::NO, 52, CPU6502::IL,  0, CPU6502::NO, 
    46, CPU6502::AB, 44, CPU6502::AB, 45, CPU6502::AB,  0, CPU6502::NO,
     3, CPU6502::RL, 44, CPU6502::IY,  0, CPU6502::NO,  0, CPU6502::NO, 
    46, CPU6502::ZX, 44, CPU6502::ZX, 45, CPU6502::ZY,  0, CPU6502::NO,
    53, CPU6502::IL, 44, CPU6502::AY, 55, CPU6502::IL,  0, CPU6502::NO, 
     0, CPU6502::NO, 44, CPU6502::AX,  0, CPU6502::NO,  0, CPU6502::NO,
    32, CPU6502::IM, 29, CPU6502::IX, 31, CPU6502::IM,  0, CPU6502::NO, 
    32, CPU6502::ZP, 29, CPU6502::ZP, 31, CPU6502::ZP,  0, CPU6502::NO,
    51, CPU6502::IL, 29, CPU6502::IM, 50, CPU6502::IL,  0, CPU6502::NO, 
    32, CPU6502::AB, 29, CPU6502::AB, 31, CPU6502::AB,  0, CPU6502::NO,
     4, CPU6502::RL, 29, CPU6502::IY,  0, CPU6502::NO,  0, CPU6502::NO, 
    32, CPU6502::ZX, 29, CPU6502::ZX, 31, CPU6502::ZY,  0, CPU6502::NO,
    16, CPU6502::IL, 29, CPU6502::AY, 54, CPU6502::IL,  0, CPU6502::NO, 
    32, CPU6502::AX, 29, CPU6502::AX, 31, CPU6502::AY,  0, CPU6502::NO,
    19, CPU6502::IM, 17, CPU6502::IX,  0, CPU6502::NO,  0, CPU6502::NO, 
    19, CPU6502::ZP, 17, CPU6502::ZP, 20, CPU6502::ZP,  0, CPU6502::NO,
    24, CPU6502::IL, 17, CPU6502::IM, 21, CPU6502::IL,  0, CPU6502::NO, 
    19, CPU6502::AB, 17, CPU6502::AB, 20, CPU6502::AB,  0, CPU6502::NO,
     8, CPU6502::RL, 17, CPU6502::IY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 17, CPU6502::ZX, 20, CPU6502::ZX,  0, CPU6502::NO,
    14, CPU6502::IL, 17, CPU6502::AY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 17, CPU6502::AX, 20, CPU6502::AX,  0, CPU6502::NO,
    18, CPU6502::IM, 43, CPU6502::IX,  0, CPU6502::NO,  0, CPU6502::NO, 
    18, CPU6502::ZP, 43, CPU6502::ZP, 26, CPU6502::ZP,  0, CPU6502::NO,
    23, CPU6502::IL, 43, CPU6502::IM, 30, CPU6502::IL,  0, CPU6502::NO, 
    18, CPU6502::AB, 43, CPU6502::AB, 26, CPU6502::AB,  0, CPU6502::NO,
     5, CPU6502::RL, 43, CPU6502::IY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 43, CPU6502::ZX, 26, CPU6502::ZX,  0, CPU6502::NO,
    48, CPU6502::IL, 43, CPU6502::AY,  0, CPU6502::NO,  0, CPU6502::NO, 
     0, CPU6502::NO, 43, CPU6502::AX, 26, CPU6502::AX,  0, CPU6502::NO
};
#endif



///////////////////////////////////////////////////////////////////////////////
//
//  Function: CPU6502
//
//  Description:
//
//      This is the main constructor for a 6502 CPU object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use the build method to create a CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU6502::CPU6502(
    const KString&   iName,
    Game*            pGame,
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
:
    CPU  ( iName, pGame, pSpace, ePurpose )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CPU6502
//
//  Description:
//
//      This is the destructor for a 6502 CPU object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU6502::~CPU6502(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getType
//
//  Description:
//
//      This member returns the type of CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CPU6502::getType(
) const
{
    //  The name of the class.
    static const KString cpuType( "6502" );

    return( cpuType );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getAddrSize
//
//  Description:
//
//      This member indicates the size of address space of the CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      the size of the CPU's address space.
//
///////////////////////////////////////////////////////////////////////////////
DWord
CPU6502::getAddrSize(
) const
{
    return( 0x10000 );
}





#ifdef DEBUGGER
///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgDisassemble
//
//  Description:
//
//      This member is called to disassemble the source at the specified
//      location and return it to the client.  The contents in the returned
//      buffer are valid until the next invocation of the function.
//
//  Parameters:
//
//      rdwAddress (input/output)
//          The address to disassemble from.  It will contain the address of
//          the next instruction when the function completes.
//
//  Returns:
//
//      A pointer to the string containing the disassembled code.
//
///////////////////////////////////////////////////////////////////////////////
const
char*
CPU6502::dbgDisassemble(
    DWord& rdwAddress
)
{
    //  The following static buffer will contain the disassembled code line.
    static char strLine[ 80 ];

    //  Work buffers.
    char strAddr[ 20 ];
    char strVals[ 20 ];
    char strDasm[ 20 ];

    //  An opcode.
    Byte bOp;

    //  Temporary bits and bytes.
    Byte  bTemp;
    Byte  bTemp2;
    DWord dwTemp;
    

    //  If the address is out of range then return nothing.
    if( rdwAddress > 0xffff )
    {
        return( "" );
    }
    
    //  Store the address in the address buffer.
    sprintf( strAddr, "$%06lx", rdwAddress );

    //  Read the opcode.  This should be read from the decrypt buffer to
    //  make sure we're not spewing out crap for opcodes.
    bOp = m_pSpace->getDecryptBuffer( )[ rdwAddress ];
    rdwAddress += 1;

    
    //  Determine the addressing mode used for the instruction.  Since the
    //  last bytes in memory contain vectors we treat them specially by
    //  always marking them as data (i.e. by using an invalid address mode).
    Byte        bMn = sm_abMnAm[ bOp * 2 ];
    AddressMode eAM =  ( ( rdwAddress - 1 ) < 0xfffa ) ? 
        ( AddressMode )sm_abMnAm[ bOp * 2 + 1 ] : NO;

    //  Add the command to the buffer based on the address mode.
    switch( eAM )
    {
        //  Accumulator mode.
        case AC:
        //  Implicit mode.
        case IL:
        {
            sprintf( strVals, "%02x", bOp );
            sprintf( strDasm, sm_astrFormats[ eAM ], sm_astrMnemonics[ bMn ] );

            break;
        }

        //  Relative mode.
        case RL:
        {
            //  Get the operand from the regular buffer.
            bTemp = m_pSpace->getBuffer( )[ rdwAddress ];
            rdwAddress += 1;


            //  Determine the address that the instruction is relative to.
            if( bTemp < 0x80 )
            {
                dwTemp = rdwAddress + bTemp;
            }
            else
            {
                dwTemp = rdwAddress + bTemp - 256;
            }

            sprintf( strVals, "%02x %02x", bOp, bTemp );
            sprintf(
                strDasm, sm_astrFormats[ eAM ], sm_astrMnemonics[ bMn ], dwTemp
            );

            break;
        }

        //  Immediate mode.
        case IM:
        //  Zero-Page mode.
        case ZP:
        //  Zero-Page X indirect mode.
        case ZX:
        //  Zero-Page Y indirect mode.
        case ZY:
        //  Indirect X mode.
        case IX:
        //  Indirect Y mode.
        case IY:
        {
            //  Get the operand from the regular buffer.
            bTemp = m_pSpace->getBuffer( )[ rdwAddress ];
            rdwAddress += 1;

            sprintf( strVals, "%02x %02x", bOp, bTemp );
            sprintf(
                strDasm, sm_astrFormats[ eAM ], sm_astrMnemonics[ bMn ], bTemp
            );

            break;
        }
        
        //  Absolute mode.
        case AB:
        //  Absolute X mode.
        case AX:
        //  Absolute Y mode.
        case AY:
        //  Indirect mode.
        case IN:
        {
            //  Get the two operands from the regular buffer.
            bTemp = m_pSpace->getBuffer( )[ rdwAddress ];
            rdwAddress += 1;
            bTemp2 = m_pSpace->getBuffer( )[ rdwAddress ];
            rdwAddress += 1;

            sprintf( strVals, "%02x %02x %02x", bOp, bTemp, bTemp2 );
            sprintf(
                strDasm, 
                sm_astrFormats[ eAM ], 
                sm_astrMnemonics[ bMn ],
                bTemp2 * 256 + bTemp
            );
            break;
        }

        //  Unknown mode.  Must be data.
        default:
        { 
            sprintf( strVals, "%02x", bOp );
            sprintf( strDasm, ".db  $%02x", bOp );
            break;
        }
    }

    //  OK, build the disassembled line and return it.
    sprintf( strLine, "%-7s  %-10s  %s", strAddr, strVals, strDasm );
    return( strLine );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgRegister
//
//  Description:
//
//      This member is called to return a string describing the value of
//      a register to the client.
//
//  Parameters:
//
//      dwNum (input)
//          Which register to retrieve.
//
//  Returns:
//
//      A pointer to the string containing the register description.
//      NULL if the register is invalid.
//
///////////////////////////////////////////////////////////////////////////////
const
char*
CPU6502::dbgRegister(
    const DWord dwNum
)
{
    //  The following static buffer will contain the register.
    static char strLine[ 80 ];

    //  Which register?
    switch( dwNum )
    {
        case REG_ICOUNT: 
            sprintf( strLine, "IC: $%04lx", getReg( REG_ICOUNT ) ); break;
        case REG_PC: 
            sprintf( strLine, "PC: $%04lx", getReg( REG_PC ) );     break;
        case REG_A:  
            sprintf( strLine, "A:  $%02lx", getReg( REG_A ) );      break;
        case REG_X:  
            sprintf( strLine, "X:  $%02lx", getReg( REG_X ) );      break;
        case REG_Y:  
            sprintf( strLine, "Y:  $%02lx", getReg( REG_Y ) );      break;
        case REG_SP: 
            sprintf( strLine, "SP: $%02lx", getReg( REG_SP ) );     break;
        case REG_SR: 
            sprintf( 
                strLine, 
                "SR: %c%c%c%c%c%c%c%c", 
                ( getReg( REG_SR ) & 0x80 ) ? 'N' : '-',
                ( getReg( REG_SR ) & 0x40 ) ? 'V' : '-',
                ( getReg( REG_SR ) & 0x20 ) ? 'U' : '-',
                ( getReg( REG_SR ) & 0x10 ) ? 'B' : '-',
                ( getReg( REG_SR ) & 0x08 ) ? 'D' : '-',
                ( getReg( REG_SR ) & 0x04 ) ? 'I' : '-',
                ( getReg( REG_SR ) & 0x02 ) ? 'Z' : '-',
                ( getReg( REG_SR ) & 0x01 ) ? 'C' : '-'
            ); 
            break;
        default:     
            return( NULL );
    }

    return( strLine );
}

#endif
