///////////////////////////////////////////////////////////////////////////////
//
//  File:    68000bsc.cpp
//
//  Class:   CPU68000BSC
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class wraps Bernd's 68000 emulation core.  This is a general
//      68000 core that can be used for any platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "68000bsc.h"
#include "appfile.h"
#include "space.h"
#ifdef DEBUGGER
#include "debugger.h"
#endif




///////////////////////////////////////////////////////////////////////////////
//  The following are required by the CPU core.
///////////////////////////////////////////////////////////////////////////////
int         movem_index1[ 256 ];
int         movem_index2[ 256 ];
int         movem_next[ 256 ];
M68000      regs;
M68000      lastint_regs;
union flagu intel_flag_lookup[ 256 ];
union flagu regflags;

///////////////////////////////////////////////////////////////////////////////
//  The following are externals found in the CPU core.
///////////////////////////////////////////////////////////////////////////////
extern void BuildCPU ( void );


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a 68000 CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
CPU68000BSC*
CPU68000BSC::s_build(
    const KString&   iName,
    Game*            pGame, 
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
{
    //  Create the new object.
    CPU68000BSC* pThis = new CPU68000BSC( iName, pGame, pSpace, ePurpose );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new CPU.
    return( pThis );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: CPU68000BSC
//
//  Description:
//
//      This is the main constructor for a 68000 CPU object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use the build method to create a CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU68000BSC::CPU68000BSC(
    const KString&   iName,
    Game*            pGame, 
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
:
    CPU68000       ( iName, pGame, pSpace, ePurpose ),
    m_nICount      ( 0 ),
    m_nPendingInts ( 0 )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the cpu object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::init(
)
{
    //  Call the base class.
    CPU68000::init( );

    //  Reset the CPU to initialize it.
    reset( ); 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CPU68000BSC
//
//  Description:
//
//      This is the destructor for the 68000 cpu object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU68000BSC::~CPU68000BSC(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the 'C' based 68000 emulation object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CPU68000BSC::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "CPU68000BSC" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the CPU to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the CPU to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU68000BSC::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    CPU68000::save( pSaveFile );

    //  Save the member data that keeps the state.
    pSaveFile->write( ( Byte* )&m_context, sizeof( m_context ) );

    //  Was it saved successfully?
    return( pSaveFile->count( ) != sizeof( m_context ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the CPU from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the CPU from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU68000BSC::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    CPU68000::load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->read( ( Byte* )&m_context, sizeof( m_context ) );

    //  Was it loaded successfully?
    return( pLoadFile->count( ) != sizeof( m_context ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: activate
//
//  Description:
//
//      This member is called to make this CPU the active CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::activate(
)
{
    //  If we're already active the just return.
    if( sm_pCPU == this )
    {
        return;
    }

    //  Allow the base class to process.
    CPU68000::activate( );

    //  Set the context in the core.
    setContext( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: deactivate
//
//  Description:
//
//      This member is called when this CPU is no longer the active CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::deactivate(
)
{
    //  Allow the base class to process.
    CPU68000::deactivate( );

    //  Get the context from the core.
    getContext( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: reset
//
//  Description:
//
//      This member is called to have the CPU reset itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::reset(
)
{
    //  Indicates whether the CPU core has been initialized.
    static Byte bCoreInit = FALSE;

    //  Indices used for iterations.
    int nI;
    int nJ;


    //  Let the base clase reset.
    CPU68000::reset( );

    //  The CPU core requires a one shot initialization so do it now if it
    //  hasn't been done yet.
    if( !bCoreInit )
    {
        bCoreInit = TRUE;

        //  Call the core function to build the CPU.
        BuildCPU( );

        //  Initialize the CPU.
        for( nI = 0 ; nI < 256 ; nI += 1 )
        {
            for( nJ = 0 ; nJ < 8 ; nJ += 1 )
            {
                if( nI & ( 1 << nJ ) )
                {
                    break; 
                }
            }
            movem_index1[ nI ] = nJ;
            movem_index2[ nI ] = 7 - nJ;
            movem_next[ nI ]   = nI & ( ~( 1 << nJ ) );
        }
        for( nI = 0 ; nI < 256 ; nI += 1 )
        {
            intel_flag_lookup[ nI ].flags.c = !!( nI & 1 );
            intel_flag_lookup[ nI ].flags.z = !!( nI & 64 );
            intel_flag_lookup[ nI ].flags.n = !!( nI & 128 );
            intel_flag_lookup[ nI ].flags.v = 0;
        }
    }

    //  Do the reset.
    activate( );
    memset( ( void* )&regs, 0x00, sizeof( regs ) );
    m68k_setpc( get_long( 0x000004 ) );
    regs.a[ 7 ]      = get_long( 0x000000 );
    regs.s           = 1;
    ZFLG             = 0;
    CFLG             = 0;
    NFLG             = 0;
    VFLG             = 0;
    regs.intmask     = 7;
    m_nPendingInts   = 0;
    deactivate( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: run
//
//  Description:
//
//      This member is called to have the CPU execute for a time slice.
//
//  Parameters:
//
//      nCycles (input)
//          The number of cycles to execute for.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::run(
    const int32 nCycles
)
{
    //  We should be active.
    ASSERT( sm_pCPU == this );
 
    //  The opcode.
    Word wOp;
    
    //  Calculate the cycles to execute this slice.
    m_nCyclesThisRun = nCycles + m_nCyclesCarry;

    //  Let's rock.
    for( m_nICount = m_nCyclesThisRun ; m_nICount > 0 ; m_nICount -= 15 )
    {
        //  Perform an interrupt if one is pending.
        if( m_nPendingInts )
        {
            static Byte abIntTable[ 128 ] =
            {
                0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4,
                5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
                6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
                6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
                7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
                7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7
            };

            int nIPL   = ( regs.sr & 0xf00 ) >> 8;
            int nLevel = abIntTable[ m_nPendingInts ];
            if( nLevel >= nIPL )
            {
                Exception( 24 + nLevel, 0 );
                m_nPendingInts &= ~( 1 << ( nLevel - 1 ) );
            }
        }

#ifdef DEBUGGER
        Debugger::s_instance( ).check( );
#endif

//printf( "0x%06x\n", m68k_getpc( ) );fflush(stdout);

        //  Get the opcode and execute the corresponding instruction.
        wOp = nextiword( );
        cpufunctbl[ wOp ]( wOp );
    }

    //  Carry over the remainder.
    m_nCyclesCarry = m_nICount;

    //  Update the total cycles executed.
    m_nCyclesTotal += ( m_nCyclesThisRun - m_nCyclesCarry );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: interrupt
//
//  Description:
//
//      This member is called to interrupt the cpu.
//
//  Parameters:
//
//      nType (input)
//          The type of interrupt.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::interrupt(
    const int32 nType
)
{
    if( ( nType >= 1 ) && ( nType <= 7 ) )
    {
        m_nPendingInts |= ( 1 << ( nType - 1 ) );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getReg
//
//  Description:
//
//      This member is called to retrieve the value of a CPU register.
//
//  Parameters:
//
//      nReg (input)
//          The register to retrieve.
//
//  Returns:
//
//      The value of the register.
//
///////////////////////////////////////////////////////////////////////////////
int32
CPU68000BSC::getReg(
    int32 nReg
)
{
    //  The result.
    int32 nResult = 0;

    //  Make sure the context is up to date.
    if( sm_pCPU == this )
    {
        getContext( );
    }

    //  Return the requested register.
    switch( nReg )
    {
        case REG_ICOUNT: nResult = m_nICount;                           break;
        case REG_PC:     nResult = m_context.pc;                        break;
        case REG_A0:     nResult = m_context.a[ 0 ];                    break;
        case REG_A1:     nResult = m_context.a[ 1 ];                    break;
        case REG_A2:     nResult = m_context.a[ 2 ];                    break;
        case REG_A3:     nResult = m_context.a[ 3 ];                    break;
        case REG_A4:     nResult = m_context.a[ 4 ];                    break;
        case REG_A5:     nResult = m_context.a[ 5 ];                    break;
        case REG_A6:     nResult = m_context.a[ 6 ];                    break;
        case REG_A7:     nResult = m_context.a[ 7 ];                    break;
        case REG_D0:     nResult = m_context.d[ 0 ];                    break;
        case REG_D1:     nResult = m_context.d[ 1 ];                    break;
        case REG_D2:     nResult = m_context.d[ 2 ];                    break;
        case REG_D3:     nResult = m_context.d[ 3 ];                    break;
        case REG_D4:     nResult = m_context.d[ 4 ];                    break;
        case REG_D5:     nResult = m_context.d[ 5 ];                    break;
        case REG_D6:     nResult = m_context.d[ 6 ];                    break;
        case REG_D7:     nResult = m_context.d[ 7 ];                    break;
        case REG_ASP:    nResult = m_context.usp;                       break;
        case REG_SR:     nResult = m_context.sr;                        break;
        default:     fatalError( "Unknown 68000 register %d", nReg );   break;
    }

    //  Return the result.
    return( nResult );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setReg
//
//  Description:
//
//      This member is called to set the value of a CPU register.
//
//  Parameters:
//
//      nReg (input)
//          The register to set.
//
//      nValue (input)
//          The value to set the register to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::setReg(
    int32 nReg,
    int32 nValue
)
{
    //  If we're executing we need to get the context.
    if( sm_pCPU == this )
    {
        getContext( );
    }

    //  Set the requested register.
    switch( nReg )
    {
        case REG_ICOUNT: m_nICount        = nValue;                    break;
        case REG_PC:     m_context.pc     = nValue;                    break;
        case REG_A0:     m_context.a[ 0 ] = nValue;                    break;
        case REG_A1:     m_context.a[ 1 ] = nValue;                    break;
        case REG_A2:     m_context.a[ 2 ] = nValue;                    break;
        case REG_A3:     m_context.a[ 3 ] = nValue;                    break;
        case REG_A4:     m_context.a[ 4 ] = nValue;                    break;
        case REG_A5:     m_context.a[ 5 ] = nValue;                    break;
        case REG_A6:     m_context.a[ 6 ] = nValue;                    break;
        case REG_A7:     m_context.a[ 7 ] = nValue;                    break;
        case REG_D0:     m_context.d[ 0 ] = nValue;                    break;
        case REG_D1:     m_context.d[ 1 ] = nValue;                    break;
        case REG_D2:     m_context.d[ 2 ] = nValue;                    break;
        case REG_D3:     m_context.d[ 3 ] = nValue;                    break;
        case REG_D4:     m_context.d[ 4 ] = nValue;                    break;
        case REG_D5:     m_context.d[ 5 ] = nValue;                    break;
        case REG_D6:     m_context.d[ 6 ] = nValue;                    break;
        case REG_D7:     m_context.d[ 7 ] = nValue;                    break;
        case REG_ASP:    m_context.usp    = nValue;                    break;
        case REG_SR:     m_context.sr     = nValue;                    break;
        default:     fatalError( "Unknown 6502 register %d", nReg );   break;
    }

    //  If we're executing, put the context back.
    if( sm_pCPU == this )
    {
        setContext( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getContext
//
//  Description:
//
//      This member is called to retrieve the context from the CPU core.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::getContext(
)
{
    regs.sr = 
        ( regs.sr & 0xfff0 ) | 
        ( NFLG << 3 ) | 
        ( ZFLG << 2 ) | 
        ( VFLG << 1 ) | 
        CFLG;
    m_context = regs;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setContext
//
//  Description:
//
//      This member is called to set the context in the CPU core.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU68000BSC::setContext(
)
{
    regs = m_context;
    NFLG = ( regs.sr >> 3 ) & 0x01;
    ZFLG = ( regs.sr >> 2 ) & 0x01;
    VFLG = ( regs.sr >> 1 ) & 0x01;
    CFLG = regs.sr & 0x01;
}




///////////////////////////////////////////////////////////////////////////////
//
//  The following function is required by the CPU core.
//
///////////////////////////////////////////////////////////////////////////////
void
Exception(
    int  nr,
    CPTR /* oldpc is unused */
)
{
    MakeSR( );

    if( !regs.s )
    {
        regs.a[ 7 ] = regs.isp;
        regs.s      = 1;
    }

    regs.a[ 7 ] -= 4;
    put_long( regs.a[ 7 ], m68k_getpc( ) );
    regs.a[ 7 ] -= 2;
    put_word( regs.a[ 7 ], regs.sr );
    m68k_setpc( get_long( regs.vbr + 4 * nr ) );

    regs.t1 = 0;
    regs.t0 = 0;
    regs.m  = 0;
}
