///////////////////////////////////////////////////////////////////////////////
//
//  File:    6809kbc.cpp
//
//  Class:   CPU6809KBC
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class wraps my old 6809 emulation core.  This is a general 6809
//      core that can be used for any platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "6809kbc.h"
#include "appfile.h"
#include "space.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a 6809 CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
CPU6809KBC*
CPU6809KBC::s_build(
    const KString&   iName,
    Game*            pGame, 
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
{
    //  Create the new object.
    CPU6809KBC* pThis = new CPU6809KBC( iName, pGame, pSpace, ePurpose );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new CPU.
    return( pThis );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: CPU6809KBC
//
//  Description:
//
//      This is the main constructor for a 6809 CPU object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use the build method to create a CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU6809KBC::CPU6809KBC(
    const KString&   iName,
    Game*            pGame, 
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
:
    CPU6809  ( iName, pGame, pSpace, ePurpose )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the cpu object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6809KBC::init(
)
{
    //  Call the base class.
    CPU6809::init( );

    //  Reset the CPU to initialize it.
    reset( ); 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CPU6809KBC
//
//  Description:
//
//      This is the destructor for the 6809 cpu object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU6809KBC::~CPU6809KBC(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the 'C' based 6809 emulation object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CPU6809KBC::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "CPU6809KBC" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the CPU to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the CPU to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU6809KBC::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    CPU6809::save( pSaveFile );

    //  Save the member data that keeps the state.
    pSaveFile->write( ( Byte* )&m_context, sizeof( m_context ) );

    //  Was it saved successfully?
    return( pSaveFile->count( ) != sizeof( m_context ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the CPU from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the CPU from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPU6809KBC::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    CPU6809::load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->read( ( Byte* )&m_context, sizeof( m_context ) );

    //  Was it loaded successfully?
    return( pLoadFile->count( ) != sizeof( m_context ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: reset
//
//  Description:
//
//      This member is called to have the CPU reset itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6809KBC::reset(
)
{
    //  Let the base clase reset.
    CPU6809::reset( );

    //  Do the reset.
    activate( );
    MC6809Reset( &m_context );
    deactivate( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: run
//
//  Description:
//
//      This member is called to have the CPU execute for a time slice.
//
//  Parameters:
//
//      nCycles (input)
//          The number of cycles to execute for.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6809KBC::run(
    const int32 nCycles
)
{
    //  We should be active.
    ASSERT( sm_pCPU == this );
 
    //  Calculate the cycles to execute this slice.
    m_nCyclesThisRun = nCycles + m_nCyclesCarry;

    //  Let's rock.
    m_nCyclesCarry = MC6809Execute( &m_context, m_nCyclesThisRun );

    //  Update the total cycles executed.
    m_nCyclesTotal += ( m_nCyclesThisRun - m_nCyclesCarry );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: interrupt
//
//  Description:
//
//      This member is called to interrupt the cpu.
//
//  Parameters:
//
//      nType (input)
//          The type of interrupt.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6809KBC::interrupt(
    const int32 nType
)
{
    //  Interrupt.
    MC6809Interrupt( &m_context, nType );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getReg
//
//  Description:
//
//      This member is called to retrieve the value of a CPU register.
//
//  Parameters:
//
//      nReg (input)
//          The register to retrieve.
//
//  Returns:
//
//      The value of the register.
//
///////////////////////////////////////////////////////////////////////////////
int32
CPU6809KBC::getReg(
    int32 nReg
)
{
    //  The result.
    int32 nResult = 0;

    //  Return the requested register.
    switch( nReg )
    {
        case REG_ICOUNT: nResult = m_context.ICount;                    break;
        case REG_PC:     nResult = m_context.pc;                        break;
        case REG_DP:     nResult = m_context.dp;                        break;
        case REG_U:      nResult = m_context.u;                         break;
        case REG_S:      nResult = m_context.s;                         break;
        case REG_X:      nResult = m_context.x;                         break;
        case REG_Y:      nResult = m_context.y;                         break;
        case REG_A:      nResult = m_context.acc.B.h;                   break;
        case REG_B:      nResult = m_context.acc.B.l;                   break;
        case REG_CC:     nResult = m_context.cc.all;                    break;
        default:     fatalError( "Unknown 6809 register %d", nReg );    break;
    }

    //  Return the result.
    return( nResult );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setReg
//
//  Description:
//
//      This member is called to set the value of a CPU register.
//
//  Parameters:
//
//      nReg (input)
//          The register to set.
//
//      nValue (input)
//          The value to set the register to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPU6809KBC::setReg(
    int32 nReg,
    int32 nValue
)
{
    //  Set the requested register.
    switch( nReg )
    {
        case REG_ICOUNT: m_context.ICount   = nValue;                  break;
        case REG_PC:     m_context.pc       = nValue;                  break;
        case REG_DP:     m_context.dp       = nValue;                  break;
        case REG_U:      m_context.u        = nValue;                  break;
        case REG_S:      m_context.s        = nValue;                  break;
        case REG_X:      m_context.x        = nValue;                  break;
        case REG_Y:      m_context.y        = nValue;                  break;
        case REG_A:      m_context.acc.B.h  = nValue;                  break;
        case REG_B:      m_context.acc.B.l  = nValue;                  break;
        case REG_CC:     m_context.cc.all   = nValue;                  break;
        default:     fatalError( "Unknown 6809 register %d", nReg );   break;
    }
}
