///////////////////////////////////////////////////////////////////////////////
//
//  File:    cpu6809.cpp
//
//  Class:   CPU6809
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class serves as a base class for any 6809 CPU emulation cores.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#ifdef DEBUGGER
#include <stdio.h>
#endif

//  Application Headers.
#include "reptypes.h"
#include "cpu6809.h"
#include "space.h"
#ifdef DEBUGGER
#include "debugger.h"
#endif


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
#ifdef DEBUGGER
//  The following table is a listing of the information for regular opcodes.
CPU6809::OpcodeInfo CPU6809::sm_aOpInfoTable[ 256 ] =
{
    // 0x00
    { "NEG  ",  2,  DIRECT     }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "COM  ",  2,  DIRECT     },
    { "LSR  ",  2,  DIRECT     }, { "?????",  1,  ILLEGAL    },
    { "ROR  ",  2,  DIRECT     }, { "ASR  ",  2,  DIRECT     },
    { "LSL  ",  2,  DIRECT     }, { "ROR  ",  2,  DIRECT     },
    { "DEC  ",  2,  DIRECT     }, { "?????",  1,  ILLEGAL    },
    { "INC  ",  2,  DIRECT     }, { "TST  ",  2,  DIRECT     },
    { "JMP  ",  2,  DIRECT     }, { "CLR  ",  2,  DIRECT     },
    // 0x10
    { "",       1,  INHERENT   }, { "",       1,  INHERENT   },
    { "NOP  ",  1,  INHERENT   }, { "SYNC ",  1,  INHERENT   },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "LBRA ",  3,  RELATIVEL  }, { "LBSR ",  3,  RELATIVEL  },
    { "?????",  1,  ILLEGAL    }, { "DAA  ",  1,  INHERENT   },
    { "ORCC ",  2,  IMMEDIATE  }, { "?????",  1,  ILLEGAL    },
    { "ANDCC",  2,  IMMEDIATE  }, { "SEX  ",  1,  INHERENT   },
    { "EXG  ",  2,  REGISTER0  }, { "TFR  ",  2,  REGISTER0  },
    // 0x20
    { "BRA  ",  2,  RELATIVE   }, { "BRN  ",  2,  RELATIVE   },
    { "BHI  ",  2,  RELATIVE   }, { "BLS  ",  2,  RELATIVE   },
    { "BCC  ",  2,  RELATIVE   }, { "BCS  ",  2,  RELATIVE   },
    { "BNE  ",  2,  RELATIVE   }, { "BEQ  ",  2,  RELATIVE   },
    { "BVC  ",  2,  RELATIVE   }, { "BVS  ",  2,  RELATIVE   },
    { "BPL  ",  2,  RELATIVE   }, { "BMI  ",  2,  RELATIVE   },
    { "BGE  ",  2,  RELATIVE   }, { "BLT  ",  2,  RELATIVE   },
    { "BGT  ",  2,  RELATIVE   }, { "BLE  ",  2,  RELATIVE   },
    //  0x30
    { "LEAX ",  2,  INDEXED    }, { "LEAY ",  2,  INDEXED    },
    { "LEAS ",  2,  INDEXED    }, { "LEAU ",  2,  INDEXED    },
    { "PSHS ",  2,  REGISTER1  }, { "PULS ",  2,  REGISTER1  },
    { "PSHU ",  2,  REGISTER2  }, { "PULU ",  2,  REGISTER2  },
    { "?????",  1,  ILLEGAL    }, { "RTS  ",  1,  INHERENT   },
    { "ABX  ",  1,  INHERENT   }, { "RTI  ",  1,  INHERENT   },
    { "CWAI ",  2,  INHERENT   }, { "MUL  ",  1,  INHERENT   },
    { "?????",  1,  ILLEGAL    }, { "SWI  ",  1,  INHERENT   },
    //  0x40
    { "NEGA ",  1,  INHERENT   }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "COMA ",  1,  INHERENT   },
    { "LSRA ",  1,  INHERENT   }, { "?????",  1,  ILLEGAL    },
    { "RORA ",  1,  INHERENT   }, { "ASRA ",  1,  INHERENT   },
    { "LSLA ",  1,  INHERENT   }, { "ROLA ",  1,  INHERENT   },
    { "DECA ",  1,  INHERENT   }, { "?????",  1,  ILLEGAL    },
    { "INCA ",  1,  INHERENT   }, { "TSTA ",  1,  INHERENT   },
    { "?????",  1,  ILLEGAL    }, { "CLRA ",  1,  INHERENT   },
    //  0x50
    { "NEGB ",  1,  INHERENT   }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "COMB ",  1,  INHERENT   },
    { "LSRB ",  1,  INHERENT   }, { "?????",  1,  ILLEGAL    },
    { "RORB ",  1,  INHERENT   }, { "ASRB ",  1,  INHERENT   },
    { "LSLB ",  1,  INHERENT   }, { "ROLB ",  1,  INHERENT   },
    { "DECB ",  1,  INHERENT   }, { "?????",  1,  ILLEGAL    },
    { "INCB ",  1,  INHERENT   }, { "TSTB ",  1,  INHERENT   },
    { "?????",  1,  ILLEGAL    }, { "CLRB ",  1,  INHERENT   },
    //  0x60
    { "NEG  ",  2,  INDEXED    }, { "?????",  2,  ILLEGAL    },
    { "?????",  2,  ILLEGAL    }, { "COM  ",  2,  INDEXED    },
    { "LSR  ",  2,  INDEXED    }, { "?????",  2,  INDEXED    },
    { "ROR  ",  2,  INDEXED    }, { "ASR  ",  2,  INDEXED    },
    { "LSL  ",  2,  INDEXED    }, { "ROL  ",  2,  INDEXED    },
    { "DEC  ",  2,  INDEXED    }, { "?????",  2,  ILLEGAL    },
    { "INC  ",  2,  INDEXED    }, { "TST  ",  2,  INDEXED    },
    { "JMP  ",  2,  INDEXED    }, { "CLR  ",  2,  INDEXED    },
    //  0x70
    { "NEG  ",  3,  EXTENDED   }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "COM  ",  3,  EXTENDED   },
    { "LSR  ",  3,  EXTENDED   }, { "?????",  1,  ILLEGAL    },
    { "ROR  ",  3,  EXTENDED   }, { "ASR  ",  3,  EXTENDED   },
    { "LSL  ",  3,  EXTENDED   }, { "ROL  ",  3,  EXTENDED   },
    { "DEC  ",  3,  EXTENDED   }, { "?????",  1,  ILLEGAL    },
    { "INC  ",  3,  EXTENDED   }, { "TST  ",  3,  EXTENDED   },
    { "JMP  ",  3,  EXTENDED   }, { "CLR  ",  3,  EXTENDED   },
    //  0x80
    { "SUBA ",  2,  IMMEDIATE  }, { "CMPA ",  2,  IMMEDIATE  },
    { "SBCA ",  2,  IMMEDIATE  }, { "SUBD ",  3,  IMMEDIATEL },
    { "ANDA ",  2,  IMMEDIATE  }, { "BITA ",  2,  IMMEDIATE  },
    { "LDA  ",  2,  IMMEDIATE  }, { "?????",  2,  ILLEGAL    },
    { "EORA ",  2,  IMMEDIATE  }, { "ADCA ",  2,  IMMEDIATE  },
    { "ORA  ",  2,  IMMEDIATE  }, { "ADDA ",  2,  IMMEDIATE  },
    { "CMPX ",  3,  IMMEDIATEL }, { "BSR  ",  2,  RELATIVE   },
    { "LDX  ",  3,  IMMEDIATEL }, { "?????",  2,  ILLEGAL    },
    //  0x90
    { "SUBA ",  2,  DIRECT     }, { "CMPA ",  2,  DIRECT     },
    { "SBCA ",  2,  DIRECT     }, { "SUBD ",  2,  DIRECT     },
    { "ANDA ",  2,  DIRECT     }, { "BITA ",  2,  DIRECT     },
    { "LDA  ",  2,  DIRECT     }, { "STA  ",  2,  DIRECT     },
    { "EORA ",  2,  DIRECT     }, { "ADCA ",  2,  DIRECT     },
    { "ORA  ",  2,  DIRECT     }, { "ADDA ",  2,  DIRECT     },
    { "CMPX ",  2,  DIRECT     }, { "JSR  ",  2,  DIRECT     },
    { "LDX  ",  2,  DIRECT     }, { "STX  ",  2,  DIRECT     },
    //  0xa0
    { "SUBA ",  2,  INDEXED    }, { "CMPA ",  2,  INDEXED    },
    { "SBCA ",  2,  INDEXED    }, { "SUBD ",  2,  INDEXED    },
    { "ANDA ",  2,  INDEXED    }, { "BITA ",  2,  INDEXED    },
    { "LDA  ",  2,  INDEXED    }, { "STA  ",  2,  INDEXED    },
    { "EORA ",  2,  INDEXED    }, { "ADCA ",  2,  INDEXED    },
    { "ORA  ",  2,  INDEXED    }, { "ADDA ",  2,  INDEXED    },
    { "CMPX ",  2,  INDEXED    }, { "JSR  ",  2,  INDEXED    },
    { "LDX  ",  2,  INDEXED    }, { "STX  ",  2,  INDEXED    },
    //  0xb0
    { "SUBA ",  3,  EXTENDED   }, { "CMPA ",  3,  EXTENDED   },
    { "SBCA ",  3,  EXTENDED   }, { "SUBD ",  3,  EXTENDED   },
    { "ANDA ",  3,  EXTENDED   }, { "BITA ",  3,  EXTENDED   },
    { "LDA  ",  3,  EXTENDED   }, { "STA  ",  3,  EXTENDED   },
    { "EORA ",  3,  EXTENDED   }, { "ADCA ",  3,  EXTENDED   },
    { "ORA  ",  3,  EXTENDED   }, { "ADDA ",  3,  EXTENDED   },
    { "CMPX ",  3,  EXTENDED   }, { "JSR  ",  3,  EXTENDED   },
    { "LDX  ",  3,  EXTENDED   }, { "STX  ",  3,  EXTENDED   },
    //  0xc0
    { "SUBB ",  2,  IMMEDIATE  }, { "CMPB ",  2,  IMMEDIATE  },
    { "SBCB ",  2,  IMMEDIATE  }, { "ADDD ",  3,  IMMEDIATEL },
    { "ANDB ",  2,  IMMEDIATE  }, { "BITB ",  2,  IMMEDIATE  },
    { "LDB  ",  2,  IMMEDIATE  }, { "?????",  1,  ILLEGAL    },
    { "EORB ",  2,  IMMEDIATE  }, { "ADCB ",  2,  IMMEDIATE  },
    { "ORB  ",  2,  IMMEDIATE  }, { "ADDB ",  2,  IMMEDIATE  },
    { "LDD  ",  3,  IMMEDIATEL }, { "?????",  1,  ILLEGAL    },
    { "LDU  ",  3,  IMMEDIATEL }, { "?????",  1,  ILLEGAL    },
    //  0xd0
    { "SUBB ",  2,  DIRECT     }, { "CMPB ",  2,  DIRECT     },
    { "SBCB ",  2,  DIRECT     }, { "ADDD ",  2,  DIRECT     },
    { "ANDB ",  2,  DIRECT     }, { "BITB ",  2,  DIRECT     },
    { "LDB  ",  2,  DIRECT     }, { "STB  ",  2,  DIRECT     },
    { "EORB ",  2,  DIRECT     }, { "ADCB ",  2,  DIRECT     },
    { "ORB  ",  2,  DIRECT     }, { "ADDB ",  2,  DIRECT     },
    { "LDD  ",  2,  DIRECT     }, { "STD  ",  2,  DIRECT     },
    { "LDU  ",  2,  DIRECT     }, { "STU  ",  2,  DIRECT     },
    //  0xe0
    { "SUBB ",  2,  INDEXED    }, { "CMPB ",  2,  INDEXED    },
    { "SBCB ",  2,  INDEXED    }, { "ADDD ",  2,  INDEXED    },
    { "ANDB ",  2,  INDEXED    }, { "BITB ",  2,  INDEXED    },
    { "LDB  ",  2,  INDEXED    }, { "STB  ",  2,  INDEXED    },
    { "EORB ",  2,  INDEXED    }, { "ADCB ",  2,  INDEXED    },
    { "ORB  ",  2,  INDEXED    }, { "ADDB ",  2,  INDEXED    },
    { "LDD  ",  2,  INDEXED    }, { "STD  ",  2,  INDEXED    },
    { "LDU  ",  2,  INDEXED    }, { "STU  ",  2,  INDEXED    },
    //  0xf0
    { "SUBB ",  3,  EXTENDED   }, { "CMPB ",  3,  EXTENDED   },
    { "SBCB ",  3,  EXTENDED   }, { "ADDD ",  3,  EXTENDED   },
    { "ANDB ",  3,  EXTENDED   }, { "BITB ",  3,  EXTENDED   },
    { "LDB  ",  3,  EXTENDED   }, { "STB  ",  3,  EXTENDED   },
    { "EORB ",  3,  EXTENDED   }, { "ADCB ",  3,  EXTENDED   },
    { "ORB  ",  3,  EXTENDED   }, { "ADDB ",  3,  EXTENDED   },
    { "LDD  ",  3,  EXTENDED   }, { "STD  ",  3,  EXTENDED   },
    { "LDU  ",  3,  EXTENDED   }, { "STU  ",  3,  EXTENDED   }
};

//  The following table is a listing of the information for page 10 opcodes.
CPU6809::OpcodeInfo CPU6809::sm_aOpInfo10Table[ 256 ] =
{
    //  0x00
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0x10
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0x20
    { "?????",  1,  ILLEGAL    }, { "LBRN ",  4,  RELATIVEL  },
    { "LBHI ",  4,  RELATIVEL  }, { "LBLS ",  4,  RELATIVEL  },
    { "LBCC ",  4,  RELATIVEL  }, { "LBCS ",  4,  RELATIVEL  },
    { "LBNE ",  4,  RELATIVEL  }, { "LBEQ ",  4,  RELATIVEL  },
    { "LBVC ",  4,  RELATIVEL  }, { "LBVS ",  4,  RELATIVEL  },
    { "LBPL ",  4,  RELATIVEL  }, { "LBMI ",  4,  RELATIVEL  },
    { "LBGE ",  4,  RELATIVEL  }, { "LBLT ",  4,  RELATIVEL  },
    { "LBGT ",  4,  RELATIVEL  }, { "LBLE ",  4,  RELATIVEL  },
    //  0x30
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "SWI2 ",  3,  INHERENT   },
    //  0x40
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0x50
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0x60
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0x70
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0x80
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "CMPD ",  4,  IMMEDIATEL },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "CMPY ",  4,  IMMEDIATEL }, { "?????",  1,  ILLEGAL    },
    { "LDY  ",  4,  IMMEDIATEL }, { "?????",  1,  ILLEGAL    },
    //  0x90
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "CMPD ",  3,  DIRECT     },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "CMPY ",  3,  DIRECT     }, { "?????",  1,  ILLEGAL    },
    { "LDY  ",  3,  DIRECT     }, { "STY  ",  3,  DIRECT     },
    //  0xa0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "CMPD ",  3,  INDEXED    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "CMPY ",  3,  INDEXED    }, { "?????",  1,  ILLEGAL    },
    { "LDY  ",  3,  INDEXED    }, { "STY  ",  3,  INDEXED    },
    //  0xb0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "CMPD ",  4,  EXTENDED   },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "CMPY ",  4,  EXTENDED   }, { "?????",  1,  ILLEGAL    },
    { "LDY  ",  4,  EXTENDED   }, { "STY  ",  4,  EXTENDED   },
    //  0xc0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "LDS  ",  4,  IMMEDIATEL }, { "?????",  1,  ILLEGAL    },
    //  0xd0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "LDS  ",  3,  DIRECT     }, { "STS  ",  3,  DIRECT     },
    //  0xe0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "LDS  ",  3,  INDEXED    }, { "STS  ",  3,  INDEXED    },
    //  0xf0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "LDS  ",  4,  EXTENDED   }, { "STS  ",  4,  EXTENDED   }
};

//  The following table is a listing of the information for page 11 opcodes.
CPU6809::OpcodeInfo CPU6809::sm_aOpInfo11Table[ 256 ] =
{
    //  0X00
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0X10
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0X20
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0X30
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "SWI3 ",  2,  INHERENT   },
    //  0X40
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0X50
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0X60
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0X70
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0X80
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "CMPU ",  4,  IMMEDIATEL },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "CMPS ",  4,  IMMEDIATEL }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0X90
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "CMPU ",  3,  DIRECT     },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "CMPS ",  3,  DIRECT     }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0XA0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "CMPU ",  3,  INDEXED    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "CMPS ",  3,  INDEXED    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0XB0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "CMPU ",  4,  EXTENDED   },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "CMPS ",  4,  EXTENDED   }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0XC0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0XD0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0XE0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    //  0XF0
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    },
    { "?????",  1,  ILLEGAL    }, { "?????",  1,  ILLEGAL    }
};


//  Arrays for handling registers.
char* CPU6809::sm_astrInterRegister[ 16 ] =
{
    "D","X","Y","U","S","PC","??","??","A","B","CC","DP","??","??","??","??"
};
char* CPU6809::sm_astrIndexedRegister[ 4 ] =
{
    "X","Y","U","S"
};
char* CPU6809::sm_astrStackS[ 8 ] =
{
    "PC","U","Y","X","DP","B","A","CC"
};
char* CPU6809::sm_astrStackU[ 8 ] =
{
    "PC","S","Y","X","DP","B","A","CC"
};
int32 CPU6809::sm_anBits[ 8 ] =
{
    0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01
};

#endif



///////////////////////////////////////////////////////////////////////////////
//
//  Function: CPU6809
//
//  Description:
//
//      This is the main constructor for a 6809 CPU object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use the build method to create a CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The type of the CPU.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU6809::CPU6809(
    const KString&   iName,
    Game*            pGame,
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
:
    CPU  ( iName, pGame, pSpace, ePurpose )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CPU6809
//
//  Description:
//
//      This is the destructor for a 6809 CPU object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPU6809::~CPU6809(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getType
//
//  Description:
//
//      This member returns the type of CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CPU6809::getType(
) const
{
    //  The name of the class.
    static const KString cpuType( "6809" );

    return( cpuType );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getAddrSize
//
//  Description:
//
//      This member indicates the size of address space of the CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      the size of the CPU's address space.
//
///////////////////////////////////////////////////////////////////////////////
DWord
CPU6809::getAddrSize(
) const
{
    return( 0x10000 );
}





#ifdef DEBUGGER
///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgDisassemble
//
//  Description:
//
//      This member is called to disassemble the source at the specified
//      location and return it to the client.  The contents in the returned
//      buffer are valid until the next invocation of the function.
//
//  Parameters:
//
//      rdwAddress (input/output)
//          The address to disassemble from.  It will contain the address of
//          the next instruction when the function completes.
//
//  Returns:
//
//      A pointer to the string containing the disassembled code.
//
///////////////////////////////////////////////////////////////////////////////
const
char*
CPU6809::dbgDisassemble(
    DWord& rdwAddress
)
{
    //  The following static buffer will contain the disassembled code line.
    static char strLine[ 80 ];

    //  Work buffers.
    char strAddr[ 20 ];
    char strVals[ 20 ];
    char strDasm[ 20 ];

    //  Pointers to the opcode buffer and the argument buffer.
    Byte* pbOpBuf  = m_pSpace->getDecryptBuffer( );
    Byte* pbArgBuf = m_pSpace->getBuffer( );

    //  The extended opcode, opcode and arguments.
    Byte bExtOp; 
    Byte bOp;
    Byte bArg1;
    Byte bArg2;
    Byte bArg3;

    //  The opcode information and type.
    OpcodeInfo* pOpInfo;
    OpcodeType  eOpType;



    //  If the address is out of range then return nothing.
    if( rdwAddress > 0xffff )
    {
        return( "" );
    }
    

    //  Initialize the work buffers.
    sprintf( strAddr, "$%06lx", rdwAddress );
    strcpy( strVals, "" );
    strcpy( strDasm, "" );


    //  Read the opcode.  This should be read from the decrypt buffer to
    //  make sure we're not spewing out crap for opcodes.
    bExtOp = pbOpBuf[ rdwAddress ];
    rdwAddress += 1;

    //  If it is an extended opcode then get the extended information.
    if( bExtOp == 0x10 )
    {
        //  Get the extended opcode.
        bOp = pbOpBuf[ rdwAddress ];
        rdwAddress += 1;
    
        //  Get the information on the page 0x10 opcode.
        pOpInfo = &( sm_aOpInfo10Table[ bOp ] );
    }
    else
    if( bExtOp == 0x11 )
    {
        //  Get the extended opcode.
        bOp = pbOpBuf[ rdwAddress ];
        rdwAddress += 1;
    
        //  Get the information on the page 0x10 opcode.
        pOpInfo = &( sm_aOpInfo11Table[ bOp ] );
    }
    else
    {
        //  Move the extended opcode to the opcode.
        bOp    = bExtOp;
        bExtOp = 0x00;

        //  Get the information on the opcode.
        pOpInfo = &( sm_aOpInfoTable[ bOp ] );
    }

    //  Get the opcode type.  Since the last bytes contain vectors we
    //  treat them specially by always marking them as data (i.e. by making
    //  them an invalid type).
    if( 
        ( ( rdwAddress - 1 - ( bExtOp ? 1 : 0 ) ) >= 0xfff0 ) ||
        ( pOpInfo->eType == ILLEGAL )
    )
    {
        //  If this was extended then readjust the address.
        if( bExtOp )
        {
            rdwAddress -= 1;
            bOp         = bExtOp;
            bExtOp      = 0x00;
        }

        eOpType = ILLEGAL;
    } 
    else
    {
        eOpType = pOpInfo->eType;
    }

    //  Assign the initial values.
    if( bExtOp )
    {
        sprintf( strVals, "%02x %02x", bExtOp, bOp );
    }
    else
    {
        sprintf( strVals, "%02x", bOp );
    }


    //  Process the opcode based on the type.
    switch( eOpType )
    {
        case DIRECT:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            rdwAddress += 1;
            sprintf( strVals, "%s %02x", strVals, bArg1 );

            sprintf( strDasm, "%6s $%02x", pOpInfo->pstrName, bArg1 );
            break;
        }
            
        case IMMEDIATE:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            rdwAddress += 1;
            sprintf( strVals, "%s %02x", strVals, bArg1 );

            sprintf( strDasm, "%6s #$%02x", pOpInfo->pstrName, bArg1 );
            break;
        }

        case IMMEDIATEL:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            bArg2 = pbArgBuf[ rdwAddress + 1 ];
            rdwAddress += 2;
            sprintf( strVals, "%s %02x %02x", strVals, bArg1, bArg2 );

            sprintf( 
                strDasm, 
                "%6s #$%04x", 
                pOpInfo->pstrName,
                ( Word )( bArg1 * 256 + bArg2 ) 
            );
            break;
        }

        case INHERENT:
        {
            sprintf( strDasm, "%6s", pOpInfo->pstrName );
            break;
        }

        case INDEXED:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            rdwAddress += 1;
            sprintf( strVals, "%s %02x", strVals, bArg1 );

            if( ( bArg1 & 0x80 ) == 0 )
            {
                sprintf(
                    strDasm,
                    "%6s %s$%02x,%s",
                    pOpInfo->pstrName,
                    ( bArg1 & 0x10 ) ? "-" : "+",
                    ( bArg1 & 0x10 ) ? 
                        0x20 - ( bArg1 & 0x1f ) : ( bArg1 & 0x1f ),
                    sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                );
            }
            else
            {
                switch( bArg1 & 0x1f )
                {
                    case 0x00:
                    {
                        sprintf(
                            strDasm,
                            "%6s ,%s+",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x01:
                    {
                        sprintf(
                            strDasm,
                            "%6s ,%s++",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x02:
                    {
                        sprintf(
                            strDasm,
                            "%6s ,-%s",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x03:
                    {
                        sprintf(
                            strDasm,
                            "%6s ,--%s",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x04:
                    {
                        sprintf(
                            strDasm,
                            "%6s ,%s",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x05:
                    {
                        sprintf(
                            strDasm,
                            "%6s B,%s",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x06:
                    {
                        sprintf(
                            strDasm,
                            "%6s A,%s",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x07:
                    {
                        break;
                    }
                    case 0x08:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        rdwAddress += 1;
                        sprintf( strVals, "%s %02x", strVals, bArg2 );
                       
                        sprintf(
                            strDasm,
                            "%6s %s$%02x,%s",
                            pOpInfo->pstrName,
                            bArg2 & 0x80 ? "-" : "+",
                            bArg2 & 0x80 ? 0x0100 - bArg2 : bArg2,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x09:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        bArg3 = pbArgBuf[ rdwAddress + 1 ];
                        rdwAddress += 2;
                        sprintf( 
                            strVals, "%s %02x %02x", strVals, bArg2, bArg3 
                        );

                        sprintf(
                            strDasm,
                            "%6s %s$%04x,%s",
                            pOpInfo->pstrName,
                            bArg2 & 0x80 ? "-" : "+",
                            bArg2 & 0x80 ? 
                                0x10000 - ( bArg2 * 256 + bArg3 ) :
                                ( bArg2 * 256 + bArg3 ),
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x0a:
                    {
                        break;
                    }
                    case 0x0b:
                    {
                        sprintf(
                            strDasm,
                            "%6s D,%s",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x0c:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        rdwAddress += 1;
                        sprintf( strVals, "%s %02x", strVals, bArg2 );
     
                        sprintf(
                            strDasm,
                            "%6s <%04lx,PCR",
                            pOpInfo->pstrName,
                            ( bArg2 + rdwAddress ) & 0xffff
                        );
                        break;
                    }
                    case 0x0d:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        bArg3 = pbArgBuf[ rdwAddress + 1];
                        rdwAddress += 2;
                        sprintf( 
                            strVals, "%s %02x %02x", strVals, bArg2, bArg3 
                        );

                        sprintf(
                            strDasm,
                            "%6s >%04lx,PCR",
                            pOpInfo->pstrName,
                            ( bArg2 * 256 + bArg3 + rdwAddress ) & 0xffff
                        );
                        break;
                    }
                    case 0x0e:
                    {
                        break;
                    }
                    case 0x0f:
                    {
                        strcpy( strDasm, "????? " );
                        break;
                    }
                    case 0x10:
                    {
                        strcpy( strDasm, "????? " );
                        break;
                    }
                    case 0x11:
                    {
                        sprintf(
                            strDasm,
                            "%6s [,%s++]",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    } 
                    case 0x12:
                    {
                        strcpy( strDasm, "????? " );
                        break;
                    }
                    case 0x13:
                    {
                        sprintf(
                            strDasm,
                            "%6s [,--%s]",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    } 
                    case 0x14:
                    {
                        sprintf(
                            strDasm,
                            "%6s [,%s]",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    } 
                    case 0x15:
                    {
                        sprintf(
                            strDasm,
                            "%6s [B,%s]",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    } 
                    case 0x16:
                    {
                        sprintf(
                            strDasm,
                            "%6s [A,%s]",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    } 
                    case 0x17:
                    {
                        break;
                    }
                    case 0x18:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        rdwAddress += 1;
                        sprintf( strVals, "%s %02x", strVals, bArg2 );
                       
                        sprintf(
                            strDasm,
                            "%6s [%s$%02x,%s]",
                            pOpInfo->pstrName,
                            bArg2 & 0x80 ? "-" : "+",
                            bArg2 & 0x80 ? 0x0100 - bArg2 : bArg2,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x19:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        bArg3 = pbArgBuf[ rdwAddress + 1 ];
                        rdwAddress += 2;
                        sprintf( 
                            strVals, "%s %02x %02x", strVals, bArg2, bArg3 
                        );
                       
                        sprintf(
                            strDasm,
                            "%6s [%s$%04x,%s]",
                            pOpInfo->pstrName,
                            bArg2 & 0x80 ? "-" : "+",
                            bArg2 & 0x80 ? 
                                0x10000 - ( bArg2 * 256 + bArg3 ) :
                                ( bArg2 * 256 + bArg3 ),
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x1a:
                    {
                        break;
                    }
                    case 0x1b:
                    {
                        sprintf(
                            strDasm,
                            "%6s [D,%s]",
                            pOpInfo->pstrName,
                            sm_astrIndexedRegister[ ( bArg1 >> 5 ) & 0x03 ]
                        );
                        break;
                    }
                    case 0x1c:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        rdwAddress += 1;
                        sprintf( strVals, "%s %02x", strVals, bArg2 );

                        sprintf(
                            strDasm,
                            "%6s [<$%04lx,PCR]",
                            pOpInfo->pstrName,
                            ( bArg2 + rdwAddress ) & 0xffff
                        );
                        break;
                    }
                    case 0x1d:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        bArg3 = pbArgBuf[ rdwAddress + 1 ];
                        rdwAddress += 2;
                        sprintf( 
                            strVals, "%s %02x %02x", strVals, bArg2, bArg3 
                        );

                        sprintf(
                            strDasm,
                            "%6s [>$%04lx,PCR]",
                            pOpInfo->pstrName,
                            ( bArg2 * 256 + bArg3 + rdwAddress ) & 0xffff
                        );
                        break;
                    }
                    case 0x1e:
                    {
                        break;
                    }
                    case 0x1f:
                    {
                        bArg2 = pbArgBuf[ rdwAddress ];
                        bArg3 = pbArgBuf[ rdwAddress + 1 ];
                        rdwAddress += 2;
                        sprintf( 
                            strVals, "%s %02x %02x", strVals, bArg2, bArg3 
                        );

                        sprintf(
                            strDasm,
                            "%6s [$%4x]",
                            pOpInfo->pstrName,
                            ( bArg2 * 256 + bArg3 )
                        );
                        break;
                    }
                    default:
                    {
                        fatalError( "Invalid Post Byte." );
                        break;
                    } 
                }
            }
            break;
        }

        case EXTENDED:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            bArg2 = pbArgBuf[ rdwAddress + 1 ];
            rdwAddress += 2;
            sprintf( strVals, "%s %02x %02x", strVals, bArg1, bArg2 );

            sprintf( 
                strDasm, 
                "%6s $%04x", 
                pOpInfo->pstrName,
                ( Word )( bArg1 * 256 + bArg2 ) 
            );
            break;
        }

        case RELATIVE:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            rdwAddress += 1;
            sprintf( strVals, "%s %02x", strVals, bArg1 );

            sprintf(
                strDasm,
                "%6s $%04lx",
                pOpInfo->pstrName,
                bArg1 & 0x80 ?  rdwAddress - 256 + bArg1 : rdwAddress + bArg1
            );
            break;
        }

        case RELATIVEL:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            bArg2 = pbArgBuf[ rdwAddress + 1 ];
            rdwAddress += 2;
            sprintf( strVals, "%s %02x %02x", strVals, bArg1, bArg2 );

            sprintf( 
                strDasm, 
                "%6s $%04lx", 
                pOpInfo->pstrName,
                bArg1 & 0x80 ? 
                    rdwAddress - 65536 + ( bArg1 * 256 + bArg2 ) :
                    rdwAddress + ( bArg1 * 256 + bArg2 )
            );
            break;
        }
             
        case REGISTER0:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            rdwAddress += 1;
            sprintf( strVals, "%s %02x", strVals, bArg1 );

            sprintf(
                strDasm,
                "%6s %s,%s",
                pOpInfo->pstrName,
                sm_astrInterRegister[ bArg1 >> 4 ],
                sm_astrInterRegister[ bArg1 & 0x0f ]
            );
            break;
        }

        case REGISTER1:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            rdwAddress += 1; 
            sprintf( strVals, "%s %02x", strVals, bArg1 );

            sprintf( strDasm, "%6s ", pOpInfo->pstrName );

            int nFlag = 0;
            for( int nI = 0 ; nI < 8 ; nI += 1 )
            {
                if( ( bArg1 & sm_anBits[ nI ] ) != 0 )
                {
                    if( nFlag != 0 )
                    {
                        strcat( strDasm, "," );
                    }
                    else
                    {
                        nFlag = 1;
                    }
                    strcat( strDasm, sm_astrStackS[ nI ] );
                }
            }
            break;
        }

        case REGISTER2:
        {
            bArg1 = pbArgBuf[ rdwAddress ];
            rdwAddress += 1; 
            sprintf( strVals, "%s %02x", strVals, bArg1 );

            sprintf( strDasm, "%6s ", pOpInfo->pstrName );

            int nFlag = 0;
            for( int nI = 0 ; nI < 8 ; nI += 1 )
            {
                if( ( bArg1 & sm_anBits[ nI ] ) != 0 )
                {
                    if( nFlag != 0 )
                    {
                        strcat( strDasm, "," );
                    }
                    else
                    {
                        nFlag = 1;
                    }
                    strcat( strDasm, sm_astrStackU[ nI ] );
                }
            }
            break;
        }

        case ILLEGAL:
        {
            sprintf( strDasm, ".db    $%02x", bOp );
            break;
        }
    }
    
    //  OK, build the disassembled line and return it.
    sprintf( strLine, "%-7s  %-13s  %s", strAddr, strVals, strDasm );
    return( strLine );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: dbgRegister
//
//  Description:
//
//      This member is called to return a string describing the value of
//      a register to the client.
//
//  Parameters:
//
//      dwNum (input)
//          Which register to retrieve.
//
//  Returns:
//
//      A pointer to the string containing the register description.
//      NULL if the register is invalid.
//
///////////////////////////////////////////////////////////////////////////////
const
char*
CPU6809::dbgRegister(
    const DWord dwNum
)
{
    //  The following static buffer will contain the register.
    static char strLine[ 80 ];

    //  Which register?
    switch( dwNum )
    {
        case REG_ICOUNT: 
            sprintf( strLine, "IC: $%04lx", getReg( REG_ICOUNT ) ); break;
        case REG_PC: 
            sprintf( strLine, "PC: $%04lx", getReg( REG_PC ) );     break;
        case REG_DP: 
            sprintf( strLine, "DP: $%02lx", getReg( REG_DP ) );     break;
        case REG_U:  
            sprintf( strLine, "U:  $%04lx", getReg( REG_U ) );      break;
        case REG_S:  
            sprintf( strLine, "S:  $%04lx", getReg( REG_S ) );      break;
        case REG_X:  
            sprintf( strLine, "X:  $%04lx", getReg( REG_X ) );      break;
        case REG_Y:  
            sprintf( strLine, "Y:  $%04lx", getReg( REG_Y ) );      break;
        case REG_A: 
            sprintf( strLine, "A:  $%04lx", getReg( REG_A ) );      break;
        case REG_B: 
            sprintf( strLine, "B:  $%04lx", getReg( REG_B ) );      break;
        case REG_CC: 
            sprintf( 
                strLine, 
                "CC: %c%c%c%c%c%c%c%c", 
                ( getReg( REG_CC ) & 0x80 ) ? 'C' : '-',
                ( getReg( REG_CC ) & 0x40 ) ? 'V' : '-',
                ( getReg( REG_CC ) & 0x20 ) ? 'Z' : '-',
                ( getReg( REG_CC ) & 0x10 ) ? 'N' : '-',
                ( getReg( REG_CC ) & 0x08 ) ? 'I' : '-',
                ( getReg( REG_CC ) & 0x04 ) ? 'H' : '-',
                ( getReg( REG_CC ) & 0x02 ) ? 'F' : '-',
                ( getReg( REG_CC ) & 0x01 ) ? 'E' : '-'
            ); 
            break;
        default:     
            return( NULL );
    }
    return( strLine );
}

#endif
