///////////////////////////////////////////////////////////////////////////////
//
//  File:    z80mdc.cpp
//
//  Class:   CPUZ80MDC
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class wraps Marcel's Z80 emulation core.  This is a general Z80
//      core that can be used for any platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "z80mdc.h"
#include "appfile.h"
#include "space.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a Z80 CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
CPUZ80MDC*
CPUZ80MDC::s_build(
    const KString&   iName,
    Game*            pGame, 
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
{
    //  Create the new object.
    CPUZ80MDC* pThis = new CPUZ80MDC( iName, pGame, pSpace, ePurpose );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new CPU.
    return( pThis );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: CPUZ80MDC
//
//  Description:
//
//      This is the main constructor for a Z80 CPU object.  It is protected
//      so that only derived classes have access to it.  Clients should
//      use the build method to create a CPU object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game the CPU belongs to.
//
//      pSpace (input)
//          The address space that the CPU is primarily bound to.
//
//      ePurpose (input)
//          The purpose of the CPU.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPUZ80MDC::CPUZ80MDC(
    const KString&   iName,
    Game*            pGame, 
    AddrSpace*       pSpace,
    const CPUPurpose ePurpose /* = GAME */
)
:
    CPUZ80  ( iName, pGame, pSpace, ePurpose )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the cpu object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPUZ80MDC::init(
)
{
    //  Call the base class.
    CPUZ80::init( );

    //  Reset the CPU to initialize it.
    reset( ); 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CPUZ80MDC
//
//  Description:
//
//      This is the destructor for the Z80 cpu object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CPUZ80MDC::~CPUZ80MDC(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the 'C' based Z80 emulation object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CPUZ80MDC::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "CPUZ80MDC" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the CPU to a file for retrieval at a later
//      date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the CPU to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPUZ80MDC::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    CPUZ80::save( pSaveFile );

    //  Save the member data that keeps the state.
    pSaveFile->write( ( Byte* )&m_context, sizeof( m_context ) );

    //  Was it saved successfully?
    return( pSaveFile->count( ) != sizeof( m_context ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the CPU from a file that it was previously
//      saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the CPU from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CPUZ80MDC::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    CPUZ80::load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->read( ( Byte* )&m_context, sizeof( m_context ) );

    //  Was it loaded successfully?
    return( pLoadFile->count( ) != sizeof( m_context ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: activate
//
//  Description:
//
//      This member is called to make this CPU the active CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPUZ80MDC::activate(
)
{
    //  If we're already active the just return.
    if( sm_pCPU == this )
    {
        return;
    }

    //  Allow the base class to process.
    CPUZ80::activate( );

    //  Set the context in the core.
    Z80_SetRegs( &m_context );
}
    

///////////////////////////////////////////////////////////////////////////////
//
//  Function: deactivate
//
//  Description:
//
//      This member is called when this CPU is no longer the active CPU.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPUZ80MDC::deactivate(
)
{
    //  Allow the base class to process.
    CPUZ80::deactivate( );

    //  Get the context from the core.
    Z80_GetRegs( &m_context );
}
    

///////////////////////////////////////////////////////////////////////////////
//
//  Function: reset
//
//  Description:
//
//      This member is called to have the CPU reset itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPUZ80MDC::reset(
)
{
    //  Allow the base class to reset itself.
    CPUZ80::reset( );

    //  Do the reset.
    activate( );
    Z80_Reset( );
    deactivate( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: run
//
//  Description:
//
//      This member is called to have the CPU execute for a time slice.
//
//  Parameters:
//
//      nCycles (input)
//          The number of cycles to execute for.
//
//  Returns:
//
//      The number of cycles executed.
//
///////////////////////////////////////////////////////////////////////////////
void
CPUZ80MDC::run(
    const int32 nCycles
)
{
    //  We should be active.
    ASSERT( sm_pCPU == this );
 
    //  Calculate the cycles to execute this slice.
    m_nCyclesThisRun = nCycles + m_nCyclesCarry;

    //  Let's rock.
    m_nCyclesCarry = Z80_Execute( m_nCyclesThisRun );

    //  Update the total cycles executed.
    m_nCyclesTotal += ( m_nCyclesThisRun - m_nCyclesCarry );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: interrupt
//
//  Description:
//
//      This member is called to interrupt the cpu.
//
//  Parameters:
//
//      nType (input)
//          The type of interrupt.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPUZ80MDC::interrupt(
    const int32 nType
)
{
    //  The currently executing CPU.
    CPU* pOldCPU = sm_pCPU;

    //  If there is no interrupt then return.
    if( nType == CPU::INT_NONE )
    {
        return;
    }

    //  Activate this CPU.
    activate( );

    Z80_Interrupt( nType );

    //  Reactivate the original CPU.
    if( pOldCPU != NULL )
    {
        pOldCPU->activate( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getReg
//
//  Description:
//
//      This member is called to retrieve the value of a CPU register.
//
//  Parameters:
//
//      nReg (input)
//          The register to retrieve.
//
//  Returns:
//
//      The value of the register.
//
///////////////////////////////////////////////////////////////////////////////
int32
CPUZ80MDC::getReg(
    int32 nReg
)
{
    //  The result.
    int32 nResult = 0;

    //  Make sure the context is up to date.
    if( sm_pCPU == this )
    {
        Z80_GetRegs( &m_context );
    }

    //  Return the requested register.
    switch( nReg )
    {
        case REG_ICOUNT:   nResult = m_context.ICount;               break;
        case REG_PC:       nResult = m_context.PC.W.l;               break;
        case REG_AF:       nResult = m_context.AF.W.l;               break;
        case REG_BC:       nResult = m_context.BC.W.l;               break;
        case REG_DE:       nResult = m_context.DE.W.l;               break;
        case REG_HL:       nResult = m_context.HL.W.l;               break;
        case REG_AF_PRIME: nResult = m_context.AF2.W.l;              break;
        case REG_BC_PRIME: nResult = m_context.BC2.W.l;              break;
        case REG_DE_PRIME: nResult = m_context.DE2.W.l;              break;
        case REG_HL_PRIME: nResult = m_context.HL2.W.l;              break;
        case REG_IX:       nResult = m_context.IX.W.l;               break;
        case REG_IY:       nResult = m_context.IY.W.l;               break;
        case REG_SP:       nResult = m_context.SP.W.l;               break;
        case REG_I:        nResult = m_context.I;                    break;
        case REG_R:        nResult = m_context.R;                    break;
        default:     fatalError( "Unknown Z80 register %d", nReg );  break;
    }

    //  Return the result.
    return( nResult );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setReg
//
//  Description:
//
//      This member is called to set the value of a CPU register.
//
//  Parameters:
//
//      nReg (input)
//          The register to set.
//
//      nValue (input)
//          The value to set the register to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CPUZ80MDC::setReg(
    int32 nReg,
    int32 nValue
)
{
    //  If we're executing we need to get the context.
    if( sm_pCPU == this )
    {
        Z80_GetRegs( &m_context );
    }

    //  Set the register.
    switch( nReg )
    {
        case REG_ICOUNT:   m_context.ICount  = nValue;              break;
        case REG_PC:       m_context.PC.W.l  = nValue;              break;
        case REG_AF:       m_context.AF.W.l  = nValue;              break;
        case REG_BC:       m_context.BC.W.l  = nValue;              break;
        case REG_DE:       m_context.DE.W.l  = nValue;              break;
        case REG_HL:       m_context.HL.W.l  = nValue;              break;
        case REG_AF_PRIME: m_context.AF2.W.l = nValue;              break;
        case REG_BC_PRIME: m_context.BC2.W.l = nValue;              break;
        case REG_DE_PRIME: m_context.DE2.W.l = nValue;              break;
        case REG_HL_PRIME: m_context.HL2.W.l = nValue;              break;
        case REG_IX:       m_context.IX.W.l  = nValue;              break;
        case REG_IY:       m_context.IY.W.l  = nValue;              break;
        case REG_SP:       m_context.SP.W.l  = nValue;              break;
        case REG_I:        m_context.I       = nValue;              break;
        case REG_R:        m_context.R       = nValue;              break;
        default:     fatalError( "Unknown Z80 register %d", nReg ); break;
    }

    //  If we're executing, put the context back.
    if( sm_pCPU == this )
    {
        Z80_SetRegs( &m_context );
    }
}
