///////////////////////////////////////////////////////////////////////////////
//
//  File:    dbginfo.cpp
//
//  Class:   DebugInfo
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents an information window in the Replay debugger.
//      The information window displays miscellaneous information such
//      as the type and name of the current CPU being debugged.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>

//  Application Headers.
#include "dbginfo.h"
#include "cpu.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: DebugInfo
//
//  Description:
//
//      This is the main constructor for the debug info window object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pDebugger (input)
//          The debugger that the window belongs to.
//
//      dwX (input)
//          The X position of the window.
//
//      dwY (input)
//          The Y position of the window.
//
//      dwWidth (input)
//          The width of the window.
//
//      dwHeight (input)
//          The height of the window.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugInfo::DebugInfo(
    const KString& iName,
    Debugger*      pDebugger,
    const DWord    dwX,
    const DWord    dwY,
    const DWord    dwWidth,
    const DWord    dwHeight
)
:
    DebugWindow ( iName, pDebugger, dwX, dwY, dwWidth, dwHeight )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DebugInfo
//
//  Description:
//
//      This is the destructor for the debugger info window object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugInfo::~DebugInfo(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the debugger info window.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
DebugInfo::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "DebugInfo" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is called when the window is to draw itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugInfo::draw(
)
{
    //  A temporary buffer.
    char strTemp[ 80 ];

    //  The current CPU.
    CPU* pCPU = m_pDebugger->m_pCPU;


    DebugWindow::draw( );

    ASSERT( pCPU != NULL );

    //  Draw the CPU information.
    sprintf( strTemp, "CPU Type:     %-60s", pCPU->getType( ).data( ) );
    writeLine( strTemp );
    sprintf( strTemp, "CPU Class:    %-60s", pCPU->getClassName( ).data( ) );
    writeLine( strTemp );
    sprintf( strTemp, "CPU Instance: %-60s", pCPU->getInstanceName( ).data( ) );
    writeLine( strTemp );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This member is called when the window is to execute.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugInfo::execute(
)
{
    return( Debugger::LEAVE_NO );
}






#endif
