///////////////////////////////////////////////////////////////////////////////
//
//  File:    dbgmem.cpp
//
//  Class:   DebugMemory
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a window in the Replay+ debugger that
//      is used to browse the contents of memory for the current CPU.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <string.h>

//  Application Headers.
#include "dbgmem.h"
#include "clip.h"
#include "cpu.h"
#include "keyb.h"
#include "bitmap.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: DebugMemory
//
//  Description:
//
//      This is the main constructor for the debug memory window object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pDebugger (input)
//          The debugger that the window belongs to.
//
//      dwX (input)
//          The X position of the window.
//
//      dwY (input)
//          The Y position of the window.
//
//      dwWidth (input)
//          The width of the window.
//
//      dwHeight (input)
//          The height of the window.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugMemory::DebugMemory(
    const KString& iName,
    Debugger*      pDebugger,
    const DWord    dwX,
    const DWord    dwY,
    const DWord    dwWidth,
    const DWord    dwHeight
)
:
    DebugWindow ( iName, pDebugger, dwX, dwY, dwWidth, dwHeight )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DebugMemory
//
//  Description:
//
//      This is the destructor for the debugger memory window object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugMemory::~DebugMemory(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the debugger info window.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
DebugMemory::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "DebugMemory" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is called when the window is to draw itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugMemory::draw(
)
{
    //  Temporary buffers.
    char strLine[ 80 ];
    char strTemp[ 16 ];

    //  An address of interest.
    DWord dwAddr;


    DebugWindow::draw( );

    //  The starting address should start on a 16 byte boundary.
    dwAddr = m_pDebugger->m_dwMemAddr & ~0xf;


    //  Draw until we run out of lines or space.
    while( 
        ( m_pCurClipping->m_nMinY < m_pCurClipping->m_nMaxY ) &&
        ( dwAddr < m_pDebugger->m_pCPU->getAddrSize( ) )
    )
    {
        sprintf( strLine, "   $%06lx: ", dwAddr );

        //  We display 16 bytes per line.
        for( DWord dwI = 0 ; dwI < 16 ; dwI += 1 )
        {
            sprintf( 
                strTemp, "%02x ", m_pDebugger->m_pCPU->readMem( dwAddr + dwI ) 
            );
            strcat( strLine, strTemp );
        }

        //  Now draw them as characters.
        for( DWord dwI = 0 ; dwI < 16 ; dwI += 1 )
        {
            Byte bValue = m_pDebugger->m_pCPU->readMem( dwAddr + dwI );
            sprintf( strTemp, "%c", bValue == 0 ? 46 : bValue % 128 );
            strcat( strLine, strTemp );
        }
       
        dwAddr += 16;
        writeLine( strLine );
    }

    //  Clear out the remainder of the window to get rid of any junk.
    m_pBitmap->clearArea( m_pCurClipping, m_pDebugger->sm_bColourBG );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This member is called when the window is to execute.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugMemory::execute(
)
{
    //  Allow basic movements.
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__UP ) )
    {
        if( m_pDebugger->m_dwMemAddr > 15 )
        {
            m_pDebugger->m_dwMemAddr -= 16;
            m_pDebugger->m_bRedrawScreen = TRUE;
        }
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__DOWN ) )
    {
        if( 
            ( m_pDebugger->m_dwMemAddr + 16 ) < 
            ( m_pDebugger->m_pCPU->getAddrSize( ) )
        )
        {
            m_pDebugger->m_dwMemAddr += 16;
            m_pDebugger->m_bRedrawScreen = TRUE;
        }
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__PGUP ) )
    {
        if( m_pDebugger->m_dwMemAddr >= m_dwPageLength * 16 )
        {
            m_pDebugger->m_dwMemAddr -= ( m_dwPageLength * 16 );
        }
        else
        {
            m_pDebugger->m_dwMemAddr = 0;
        }
        m_pDebugger->m_bRedrawScreen = TRUE;
    }
    else
    if( m_pKeyboard->switchOnWait( Keyboard::KEY__PGDN ) )
    {
        if( 
            ( m_pDebugger->m_dwMemAddr + ( m_dwPageLength * 16 ) ) <
            ( m_pDebugger->m_pCPU->getAddrSize( ) )
        )
        {
            m_pDebugger->m_dwMemAddr += ( m_dwPageLength * 16 );
        }
        else
        {
            m_pDebugger->m_dwMemAddr = m_pDebugger->m_pCPU->getAddrSize( ) - 16;
        }
        m_pDebugger->m_bRedrawScreen = TRUE;
    }

    return( Debugger::LEAVE_NO );
}






#endif
