///////////////////////////////////////////////////////////////////////////////
//
//  File:    dbgreg.cpp
//
//  Class:   DebugRegisters
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a window in the Replay+ debugger that 
//      displays the values of the registers for the current CPU being
//      debugged.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

#ifdef DEBUGGER

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>

//  Application Headers.
#include "dbgreg.h"
#include "cpu.h"
#include "clip.h"
#include "bitmap.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: DebugRegisters
//
//  Description:
//
//      This is the main constructor for the debug registers window object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//      pDebugger (input)
//          The debugger that the window belongs to.
//
//      dwX (input)
//          The X position of the window.
//
//      dwY (input)
//          The Y position of the window.
//
//      dwWidth (input)
//          The width of the window.
//
//      dwHeight (input)
//          The height of the window.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugRegisters::DebugRegisters(
    const KString& iName,
    Debugger*      pDebugger,
    const DWord    dwX,
    const DWord    dwY,
    const DWord    dwWidth,
    const DWord    dwHeight
)
:
    DebugWindow ( iName, pDebugger, dwX, dwY, dwWidth, dwHeight )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DebugRegisters
//
//  Description:
//
//      This is the destructor for the debugger registers window object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DebugRegisters::~DebugRegisters(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the debugger registers window.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
DebugRegisters::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "DebugRegisters" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is called when the window is to draw itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
DebugRegisters::draw(
)
{
    //  A temporary buffer.
    char strTemp[ 80 ];


    DebugWindow::draw( );

    //  Draw registers until we run out of registers or room.
    for( 
        DWord dwReg = m_pDebugger->m_dwRegister ; 
        m_pDebugger->m_pCPU->dbgRegister( dwReg ) != NULL ;
        dwReg += 1
    )
    {
        //  If we're out of room then quit.
        if( m_pCurClipping->m_nMinY > m_pCurClipping->m_nMaxY )
        {
            break;
        }

        //  Draw the register.
        sprintf( strTemp, "%-79s", m_pDebugger->m_pCPU->dbgRegister( dwReg ) );
        writeLine( strTemp );
    }

    //  Clear out the remainder of the window to get rid of any junk.
    m_pBitmap->clearArea( m_pCurClipping, m_pDebugger->sm_bColourBG );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: execute
//
//  Description:
//
//      This member is called when the window is to execute.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      One of the "leave" status codes.
//
///////////////////////////////////////////////////////////////////////////////
Debugger::Leave
DebugRegisters::execute(
)
{
    return( Debugger::LEAVE_NO );
}






#endif
