///////////////////////////////////////////////////////////////////////////////
//
//  File:    canvasd.cpp
//
//  Class:   CanvasDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents the Display of the Replay application on
//      the DOS platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "canvasd.h"
#include "paged.h"
#include "bitmapd.h"
#include "colour.h"

//  Allegro Headers.
#include <allegro.h>



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a DOS Canvas object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
CanvasDOS*
CanvasDOS::s_build(
    const KString& iName
)
{
    //  Create the new object.
    CanvasDOS* pThis = new CanvasDOS( iName );
 
    //  Initialize the new object.
    pThis->init( );

    //  Send back the new canvas.
    return( pThis );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: CanvasDOS
//
//  Description:
//
//      This is the main constructor for the DOS Canvas object.  This is
//      a protected member.  Clients wishing to create a canvas object should
//      do so through the factory method.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CanvasDOS::CanvasDOS(
    const KString& iName
)
:
    Canvas         ( iName )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the canvas object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasDOS::init(
)
{
    //  Initialize the Allegro graphics library.
    allegro_init( );

    //  Now that we've determined the visual characteristics we can call
    //  the base init( ).  The base init( ) uses some of this information
    //  (such as # of available colours) so this init( ) shouldn't be done
    //  any sooner.
    Canvas::init( );
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CanvasDOS
//
//  Description:
//
//      This is the main destructor for the DOS Canvas object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CanvasDOS::~CanvasDOS(
)
{
    //  Set the graphics mode back to regular text.
    set_gfx_mode( GFX_TEXT, 0, 0, 0, 0 );

    //  Shut down allegro.
    allegro_exit( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the canvas object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CanvasDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "CanvasDOS" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addPage
//
//  Description:
//
//      This member is used to add a page to the canvas.
//
//  Parameters:
//
//      pageName (input)
//          The name of the page.
//
//      ePageNumber (input)
//          The page number.
//
//      dwWidth (input)
//          The width of the page.
//
//      dwHeight (input)
//          The height of the page.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasDOS::addPage(
    const KString& pageName,
    PageNumber     ePageNumber,
    DWord          dwWidth,
    DWord          dwHeight
)
{
    //  Make sure the page doesn't exist already.
    ASSERT( m_apPageList[ ePageNumber ] == NULL );

    //  OK, create the new page.
    m_apPageList[ ePageNumber ] = new PageDOS( 
        pageName, dwWidth, dwHeight, getTranspose( )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: turnPage
//
//  Description:
//
//      This member is called to set the current page on the canvas.
//
//  Parameters:
//
//      ePageNumber (input)
//          The page to turn to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasDOS::turnPage(
    PageNumber ePageNumber
)
{
    //  Check the parameters.
    ASSERT( ePageNumber < PAGE_COUNT );
    ASSERT( m_apPageList[ ePageNumber ] != NULL );

    //  If the page hasn't changed then just return.
    if( m_pPage == m_apPageList[ ePageNumber ] )
    {
        return;
    }

    //  Call the base class.
    Canvas::turnPage( ePageNumber );

    //  Tell the page to set the correct screen mode.
    ASSERT( m_pPage != NULL );
    ( ( PageDOS* )m_pPage )->setGraphicsMode( );
}






///////////////////////////////////////////////////////////////////////////////
//
//  Function: createBitmap
//
//  Description:
//
//      This member is used to create a blank bitmap.
//
//  Parameters:
//
//      iName (input)
//          instance name of the object.
//
//      dwWidth (input)
//          The width of the page.
//
//      dwHeight (input)
//          The height of the page.
//
//      bForScreen (input)
//          Is the bitmap a screen bitmap.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Bitmap*
CanvasDOS::createBitmap(
    const KString& iName,
    DWord          dwWidth,
    DWord          dwHeight,
    Byte           bForScreen /* = FALSE */

)
{
    //  Return a new bitmap. 
    return( BitmapDOS::s_build( iName, dwWidth, dwHeight, bForScreen ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: draw
//
//  Description:
//
//      This member is used to draw a bitmap on the current page.
//
//  Parameters:
//
//      pBitmap (input)
//          The bitmap to draw.
//
//      nX (input)
//          The X position to draw the bitmap at.
//
//      nY (input)
//          The Y position to draw the bitmap at.
//
//      dwWidth (input)
//          The width to draw.
//
//      dwHeight (input)
//          The height to draw.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CanvasDOS::draw(
    Bitmap* pBitmap,
    int32   nX,
    int32   nY,
    DWord   dwWidth,
    DWord   dwHeight
)
{
    //  There must be a page selected.
    ASSERT( m_pPage != NULL );

    //  The bitmap must be for this platform.
    ASSERT( pBitmap->getClassName( ) == "BitmapDOS" );

    //  The bitmap must be a screen bitmap.
    ASSERT( pBitmap->isScreen( ) );

    //  The width and height must be no greater than the bitmap.
    ASSERT( dwWidth  <= pBitmap->getWidth( ) );
    ASSERT( dwHeight <= pBitmap->getHeight( ) );
    
    //  Used to swap variables.
    int32 nSwap;
    DWord dwSwap;

    //  A convenient pointer to the DOS page.
    PageDOS* pPage = ( PageDOS* )m_pPage;

    //  If the user has requested that the complete bitmap be drawn to the
    //  origin of the page then we classify this as an "easy" draw.
    Byte bEasy =
        ( dwWidth  == pBitmap->getWidth( )  ) &&
        ( dwHeight == pBitmap->getHeight( ) ) &&
        ( nX == 0 ) &&
        ( nY == 0 );

    //  If the canvas is transposed then we need to swap the X/Y related 
    //  parameters.
    if( getTranspose( ) )
    {
        //  Swap the X/Y locations.
        nSwap = nX;
        nX    = nY;
        nY    = nSwap;

        //  Now swap the width/height of the area to draw.
        dwSwap   = dwWidth;
        dwWidth  = dwHeight;
        dwHeight = dwSwap;
    }
    
    //  If the canvas is X Flipped then we need to adjust the X coordinate.
    if( getFlipX( ) )
    {
        nX = 
            ( getTranspose( ) ? pPage->getHeight( ) : pPage->getWidth( ) ) -
            dwWidth -
            nX;
    }
    
    //  If the canvas is Y Flipped then we need to adjust the Y coordinate.
    if( getFlipY( ) )
    {
        nY = 
            ( getTranspose( ) ? pPage->getWidth( ) : pPage->getHeight( ) ) -
            dwHeight -
            nY;
    }
    

    //  Let the page do the actual drawing.
    pPage->draw( bEasy, pBitmap, nX, nY, dwWidth, dwHeight );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getNumColours
//
//  Description:
//
//      This member returns the maximum number of colours supported by
//      the canvas.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The number of colours supported.
//
///////////////////////////////////////////////////////////////////////////////
DWord
CanvasDOS::getNumColours( 
) const
{
    //  Currently we're always returning 256 colours.  In the future, this
    //  should be changed to return the real number of colours supported.
    return( 256 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: changeColour
//
//  Description:
//
//      This member is called to change a colour in the palette.
//
//  Parameters:
//
//      dwIndex (input)
//          The colour to change.
//
//      dwRed (input)
//          The red component to set.
//
//      dwGreen (input)
//          The green component to set.
//
//      dwBlue (input)
//          The blue component to set.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
CanvasDOS::changeColour( 
    DWord dwIndex, 
    DWord dwRed, 
    DWord dwGreen, 
    DWord dwBlue
)
{
    ASSERT( dwIndex < getNumColours( ) );
    
    //  A colour structure and a pointer to the colour object we're interested
    //  in.
    RGB     rgb;
    Colour* pColour = &( m_pPaletteColours[ dwIndex ] );

    //  Save the RGB values.
    pColour->set( dwRed, dwGreen, dwBlue );

    //  Set the R/G/B values.
    rgb.r = ( ( dwRed   * 65535 ) / 255 ) >> 2;
    rgb.g = ( ( dwGreen * 65535 ) / 255 ) >> 2;
    rgb.b = ( ( dwBlue  * 65535 ) / 255 ) >> 2;

    //  Now change the colour.
    set_color( dwIndex, &rgb );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member is called to allow the canvas to update it's display.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      FALSE indicating that there is no expose event.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CanvasDOS::update(
)
{
    //  If there is no page then we don't really care about any events.
    if( m_pPage == NULL )
    {
        return( FALSE );
    }

    //  If there is an update needed then say so.
    if( m_bUpdateNeeded )
    {
        m_bUpdateNeeded = FALSE;
        return( TRUE );
    }

    //  Nothing.
    return( FALSE );
}
