///////////////////////////////////////////////////////////////////////////////
//
//  File:    dbuffer.cpp
//
//  Class:   DOSBuffer
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a buffer that is stored in low memory for
//      access by RealMode calls.  In particular, it is used by IPX 
//      network access calls.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.

//  Application Headers.
#include "dbuffer.h"


///////////////////////////////////////////////////////////////////////////////
//
//  Function: DOSBuffer
//
//  Description:
//
//      This is the main constructor for a DOS Buffer object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwSize (input)
//          The size of the buffer in bytes.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DOSBuffer::DOSBuffer(
    const KString&  iName,
    const DWord     dwSize
)
 :
    RepBase           ( iName ),
    m_dwSize          ( dwSize ),
    m_wSegment        ( 0x0000 ),
    m_wOffset         ( 0x0000 ),
    m_dwAddress       ( 0x000000 ),
    m_dwLinearAddress ( 0x000000 )
{
    //  Temporary variables used to calculate the address of the buffer.
    unsigned int nAddress;
    unsigned int nPage;
    unsigned int nSegment;
    unsigned int nOffset;


    //  Time to allocate the required memory for the buffer.  
    //  However, we don't want the buffer to cross over a 64k boundary
    //  (Why? Performance?) so we actually allocate twice the required
    //  space.  By allocating twice the required amount, we guarantee
    //  that we'll have access to a contiguous block of memory that will
    //  fit our buffer and not cross a boundary.
    // 
    //  The size is allocated in paragraphs, not bytes and is calculated
    //  as ( reqdsize + 15 ) >> 4.
    ASSERT( m_dwSize > 0 );
    m_info.size = ( 2 * m_dwSize + 15 ) >> 4;
    CONFIRM(
        _go32_dpmi_allocate_dos_memory( &m_info ) == 0,
        "Could not allocate 2 * %ld bytes of conventional memory for %s.",
        m_dwSize,
        iName.data( )
    ); 

    //  Assign the address of the buffer.
    nAddress = m_info.rm_segment << 4;
   
    //  Assign the page.
    nPage = nAddress & 0xffff;

    //  If the buffer crosses a 64k boundary then adjust the address by
    //  using the extra space that we allocated.
    if( ( nPage + m_dwSize ) > 0xffff )
    {
        nAddress = ( nAddress - nPage ) + 0x10000;
    }

    //  Assign the segment and offset of the address.
    nSegment   = nAddress / 16;
    nOffset    = nAddress % 16;
    m_wSegment = nSegment;
    m_wOffset  = nOffset;

    //  Assign the address of the buffer.
    m_dwLinearAddress = nAddress;
    m_dwAddress       = ( ( DWord )nSegment << 16 ) | nOffset;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~DOSBuffer
//
//  Description:
//
//      This is the main destructor for the DOS Buffer object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
DOSBuffer::~DOSBuffer(
)
{
    //  Free the memory.
    _go32_dpmi_free_dos_memory( &m_info );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the DOS Buffer object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
DOSBuffer::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "DOSBuffer" );

    return( className );
}
