///////////////////////////////////////////////////////////////////////////////
//
//  File:    gmode.cpp
//
//  Class:   GraphicsMode
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a DOS graphics mode.  It's purpose is to
//      keep track of the characteristics of the current graphics mode and
//      be able to set the hardware to this mode.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <dos.h>
#include <io.h>
#include <fcntl.h>

//  Application Headers.
#include "gmode.h"

//  Allegro Headers.
#include <allegro.h>


///////////////////////////////////////////////////////////////////////////////
//
//  Function: GraphicsMode
//
//  Description:
//
//      This is the main constructor for a Graphics Mode object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the graphics mode.
//
//      dwHeight (input)
//          The height of the graphics mode.
//
//      bScanLines (input)
//          Indicates whether this mode shows scan lines or not.
//
//      bTweaked (input)
//          Indicates whether this is a tweaked mode or not.  If it's not
//          a tweaked mode then it's considered a vesa mode.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GraphicsMode::GraphicsMode(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bScanLines,
    const Byte      bTweaked
)
 :
    RepBase          ( iName ),
    m_dwWidth        ( dwWidth ),
    m_dwHeight       ( dwHeight ),
    m_bScanLines     ( bScanLines ),
    m_bTweaked       ( bTweaked ),
    m_pRegisterList  ( NULL ),
    m_dwNumRegisters ( 0 )
{
    ASSERT( dwWidth > 0 );
    ASSERT( dwHeight > 0 );

    //  If this is a tweaked mode then allocate a register list.
    if( m_bTweaked )
    {
        m_pRegisterList = new Register[ MAX_REGISTERS ];
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GraphicsMode
//
//  Description:
//
//      This is the main destructor for the GraphicsMode object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GraphicsMode::~GraphicsMode(
)
{
    //  Clean up the register list.
    delete [] m_pRegisterList;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the GraphicsMode object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
GraphicsMode::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "GraphicsMode" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: set
//
//  Description:
//
//      This member is called to activate the graphics mode.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GraphicsMode::set(
) const
{
    //  Retrieve the current palette.
    PALETTE pal;
    get_palette( pal );


    //  First set the initial graphics mode.
    if( m_bTweaked )
    {
        CONFIRM(
            set_gfx_mode( GFX_VGA, 320, 200, 0, 0 ) == 0,
            "Could not set graphics mode %d (320 x 200).", 
            GFX_VGA
        );
    }
    else
    {
        CONFIRM(
            set_gfx_mode( GFX_VESA1, m_dwWidth, m_dwHeight, 0, 0 ) == 0,
            "Could not set graphics mode %d (%d x %d).",
            GFX_VESA1,
            m_dwWidth,
            m_dwHeight
        );
    }

    //  If this is a tweaked mode then set the register values.
    if( m_bTweaked )
    {
        //  Ready the VGA registers to accept the configuration register
        //  settings.
        outportb( 0x3d4, 0x11 );
        int nV = inportb( 0x3d5 ) & 0x7f;
        outportb( 0x3d4, 0x11 );
        outportb( 0x3d5, nV );

        //  Now loop through the configuration registers.
        for( DWord dwI = 0 ; dwI < m_dwNumRegisters ; dwI += 1 )
        {
            //  A convenient pointer to the current register.
            Register* pRegister = &( m_pRegisterList[ dwI ] );

            //  Which port?
            switch( pRegister->m_dwPort )
            {
                case ATTRCON_ADDR:
                {
                    //  Reset the read/write flip-flop.
                    inportb( STATUS_ADDR );
                    outportb( ATTRCON_ADDR, pRegister->m_bIndex | 0x20 );

                    //  Ensure VGA output is enabled.
                    outportb( ATTRCON_ADDR, pRegister->m_bValue );

                    break;
                }

                case MISC_ADDR:
                case VGAENABLE_ADDR:
                {
                    //  Directly to the port.
                    outportb( pRegister->m_dwPort, pRegister->m_bValue );
                    break;
                }

                case SEQ_ADDR:
                case GRACON_ADDR:
                case CRTC_ADDR:
                default:
                {
                    //  Index to port.
                    outportb( pRegister->m_dwPort, pRegister->m_bIndex );

                    //  Value to port + 1.
                    outportb( pRegister->m_dwPort + 1, pRegister->m_bValue );
 
                    break;
                }
            }
        }
    }

    //  Restore the palette.
    set_palette( pal );
}
