///////////////////////////////////////////////////////////////////////////////
//
//  File:       gmode.h
//
//  Class:      GraphicsMode - Concrete
//  
//  Hierarchy:  GraphicsMode - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class represents a DOS graphics mode.  It's purpose is to
//      keep track of the characteristics of the current graphics mode and
//      be able to set the hardware to this mode.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _gmode_h_
#define _gmode_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Allegro Headers.
#include <allegro.h>

//  Application Headers.
#include "repbase.h"


///////////////////////////////////////////////////////////////////////////////
//  GraphicsMode Class Definition
///////////////////////////////////////////////////////////////////////////////
class GraphicsMode : public RepBase
{
    public:

        //  Creation/Deletion.
        GraphicsMode ( 
            const KString& iName, 
            const DWord    dwWidth, 
            const DWord    dwHeight, 
            const Byte     bScanLines,
            const Byte     bTweaked
        );
        virtual ~GraphicsMode ( );
   
        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Interface.
        Byte  isLinear     ( ) const;
        DWord getWidth     ( ) const;
        DWord getHeight    ( ) const;
        DWord getAddress   ( 
            const DWord dwRow = 0 , const DWord dwCol = 0 
        ) const;
        void  set          ( ) const;
        Byte  match        ( 
            const DWord dwWidth, 
            const DWord dwHeight, 
            const Byte bScanLines = FALSE,
            const Byte bTweaked   = TRUE
        ) const;
        void  addRegister  ( 
            const DWord dwPort, const Byte bIndex, const Byte bValue 
        );

    protected:

        //  The following define the base port numbers to access VGA
        //  components.
        enum
        {
            ATTRCON_ADDR   = 0x3c0,
            MISC_ADDR      = 0x3c2,
            VGAENABLE_ADDR = 0x3c3,
            SEQ_ADDR       = 0x3c4,
            GRACON_ADDR    = 0x3ce,
            CRTC_ADDR      = 0x3d4,
            STATUS_ADDR    = 0x3da
        };

        //  The maximum number of registers for a tweaked mode.
        enum
        {
            MAX_REGISTERS = 32
        };

        //  The following embedded class is used to hold a VGA register.
        class Register
        {
            public:
                DWord m_dwPort;
                Byte  m_bIndex;
                Byte  m_bValue;
        };

        //  Member Data.
        DWord              m_dwWidth;
        DWord              m_dwHeight;
        Byte               m_bScanLines;
        Byte               m_bTweaked;
        Register*          m_pRegisterList;
        DWord              m_dwNumRegisters;
        
    private:

};

///////////////////////////////////////////////////////////////////////////////
//  GraphicsMode Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Byte
GraphicsMode::isLinear(
) const
{
    return( m_bTweaked );
}

inline
DWord
GraphicsMode::getWidth(
) const
{
    return( m_dwWidth );
} 

inline
DWord
GraphicsMode::getHeight(
) const
{
    return( m_dwHeight );
} 

inline
DWord 
GraphicsMode::getAddress( 
    const DWord dwRow /* = 0 */,
    const DWord dwCol /* = 0 */
) const
{
    ASSERT( dwRow < m_dwHeight );
    ASSERT( dwCol < m_dwWidth );

    if( m_bTweaked )
    {
        return( 0xa0000 + ( dwRow * m_dwWidth ) + dwCol );
    }
    else
    {
        return( bmp_write_line( screen, dwRow ) + dwCol );
    }
}

inline
Byte
GraphicsMode::match(
    const DWord dwWidth, 
    const DWord dwHeight, 
    const Byte bScanLines /* = FALSE */,
    const Byte bTweaked   /* = TRUE */
) const
{
    return( 
        ( dwWidth    <= m_dwWidth ) && 
        ( dwHeight   <= m_dwHeight ) &&
        ( bScanLines == m_bScanLines ) && 
        ( bTweaked   == m_bTweaked )
    );
}

inline
void
GraphicsMode::addRegister( 
    const DWord dwPort, 
    const Byte  bIndex, 
    const Byte  bValue 
)
{
    ASSERT( m_dwNumRegisters < MAX_REGISTERS );
    ASSERT( m_bTweaked );
    m_pRegisterList[ m_dwNumRegisters ].m_dwPort = dwPort;
    m_pRegisterList[ m_dwNumRegisters ].m_bIndex = bIndex;
    m_pRegisterList[ m_dwNumRegisters ].m_bValue = bValue;
    m_dwNumRegisters += 1;
}

#endif
