///////////////////////////////////////////////////////////////////////////////
//
//  File:    keybd.cpp
//
//  Class:   KeyboardDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The KeyboardDOS class encapsulates keyboard access for the DOS
//      platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <go32.h>
#include <pc.h>

//  Application Headers.
#include "keybd.h"
#include "replay.h"
#include "switch.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data.
///////////////////////////////////////////////////////////////////////////////
//  Indicates whether or not the keyboard handler has been installed.
Byte KeyboardDOS::sm_bKeyboardHandlerInstalled = FALSE;

//  The following variables are used to keep a queue of keyboard events.
KeyboardDOS::KeyEvent KeyboardDOS::sm_aKeyEventQueue[ 256 ];
Byte                  KeyboardDOS::sm_bQueueCount;

//  The following contain information on the new and old keyboard handlers.
_go32_dpmi_seginfo KeyboardDOS::sm_oldKeyboardHandler;
_go32_dpmi_seginfo KeyboardDOS::sm_newKeyboardHandler;

//  Tables of key codes.
const Keyboard::KeyboardKey KeyboardDOS::sm_aKeyCodes[ ] =
{
    KEY__NONE,              //  N/A                    0x000
    KEY__ESC,               //  Escape                 0x001
    KEY__1,                 //  1                      0x002
    KEY__2,                 //  2                      0x003
    KEY__3,                 //  3                      0x004
    KEY__4,                 //  4                      0x005
    KEY__5,                 //  5                      0x006
    KEY__6,                 //  6                      0x007
    KEY__7,                 //  7                      0x008
    KEY__8,                 //  8                      0x009
    KEY__9,                 //  9                      0x00a
    KEY__0,                 //  0                      0x00b
    KEY__MINUS,             //  -                      0x00c
    KEY__EQUAL,             //  =                      0x00d
    KEY__BS,                //  Backspace              0x00e
    KEY__TAB,               //  Tab                    0x00f
    KEY__Q,                 //  Q                      0x010
    KEY__W,                 //  W                      0x011
    KEY__E,                 //  E                      0x012
    KEY__R,                 //  R                      0x013
    KEY__T,                 //  T                      0x014
    KEY__Y,                 //  Y                      0x015
    KEY__U,                 //  U                      0x016
    KEY__I,                 //  I                      0x017
    KEY__O,                 //  O                      0x018
    KEY__P,                 //  P                      0x019
    KEY__LBRACE,            //  [                      0x01a
    KEY__RBRACE,            //  ]                      0x01b
    KEY__ENTER,             //  Enter                  0x01c
    KEY__LCTRL,             //  Left Control           0x01d
    KEY__A,                 //  A                      0x01e
    KEY__S,                 //  S                      0x01f
    KEY__D,                 //  D                      0x020
    KEY__F,                 //  F                      0x021
    KEY__G,                 //  G                      0x022
    KEY__H,                 //  H                      0x023
    KEY__J,                 //  J                      0x024
    KEY__K,                 //  K                      0x025
    KEY__L,                 //  L                      0x026
    KEY__COLON,             //  ;                      0x027
    KEY__QUOTE,             //  '                      0x028
    KEY__NONE,              //  N/A                    0x029
    KEY__LSHIFT,            //  Left Shift             0x02a
    KEY__BSLASH,            //  Backslash              0x02b
    KEY__Z,                 //  Z                      0x02c
    KEY__X,                 //  X                      0x02d
    KEY__C,                 //  C                      0x02e
    KEY__V,                 //  V                      0x02f
    KEY__B,                 //  B                      0x030
    KEY__N,                 //  N                      0x031
    KEY__M,                 //  M                      0x032
    KEY__COMMA,             //  ,                      0x033
    KEY__PERIOD,            //  .                      0x034
    KEY__SLASH,             //  /                      0x035
    KEY__RSHIFT,            //  Right Shift            0x036
    KEY__MULT,              //  *                      0x037
    KEY__LALT,              //  Left Alt               0x038
    KEY__SPACE,             //  Space                  0x039
    KEY__NONE,              //  N/A                    0x03a
    KEY__F1,                //  F1                     0x03b
    KEY__F2,                //  F2                     0x03c
    KEY__F3,                //  F3                     0x03d
    KEY__F4,                //  F4                     0x03e
    KEY__F5,                //  F5                     0x03f
    KEY__F6,                //  F6                     0x040
    KEY__F7,                //  F7                     0x041
    KEY__F8,                //  F8                     0x042
    KEY__F9,                //  F9                     0x043
    KEY__F10,               //  F10                    0x044
    KEY__NONE,              //  Numlock                0x045
    KEY__SLOCK,             //  Scroll Lock            0x046
    KEY__7,                 //  Numpad 7               0x047
    KEY__8,                 //  Numpad 8               0x048
    KEY__9,                 //  Numpad 9               0x049
    KEY__SUBT,              //  Subtract               0x04a
    KEY__4,                 //  Numpad 4               0x04b
    KEY__5,                 //  Numpad 5               0x04c
    KEY__6,                 //  Numpad 6               0x04d
    KEY__ADD,               //  Add                    0x04e
    KEY__1,                 //  Numpad 1               0x04f
    KEY__2,                 //  Numpad 2               0x050
    KEY__3,                 //  Numpad 3               0x051
    KEY__0,                 //  Numpad 0               0x052
    KEY__PERIOD,            //  .                      0x053
    KEY__NONE,              //  N/A                    0x054
    KEY__NONE,              //  N/A                    0x055
    KEY__NONE,              //  N/A                    0x056
    KEY__F11,               //  F11                    0x057
    KEY__F12,               //  F12                    0x058
    KEY__NONE,              //  N/A                    0x059
    KEY__NONE,              //  N/A                    0x05a
    KEY__NONE,              //  N/A                    0x05b
    KEY__NONE,              //  N/A                    0x05c
    KEY__NONE,              //  N/A                    0x05d
    KEY__NONE,              //  N/A                    0x05e
    KEY__NONE,              //  N/A                    0x05f
    KEY__NONE,              //  N/A                    0x060
    KEY__NONE,              //  N/A                    0x061
    KEY__NONE,              //  N/A                    0x062
    KEY__NONE,              //  N/A                    0x063
    KEY__NONE,              //  N/A                    0x064
    KEY__NONE,              //  N/A                    0x065
    KEY__NONE,              //  N/A                    0x066
    KEY__NONE,              //  N/A                    0x067
    KEY__NONE,              //  N/A                    0x068
    KEY__NONE,              //  N/A                    0x069
    KEY__NONE,              //  N/A                    0x06a
    KEY__NONE,              //  N/A                    0x06b
    KEY__NONE,              //  N/A                    0x06c
    KEY__NONE,              //  N/A                    0x06d
    KEY__NONE,              //  N/A                    0x06e
    KEY__NONE,              //  N/A                    0x06f
    KEY__NONE,              //  N/A                    0x070
    KEY__NONE,              //  N/A                    0x071
    KEY__NONE,              //  N/A                    0x072
    KEY__NONE,              //  N/A                    0x073
    KEY__NONE,              //  N/A                    0x074
    KEY__NONE,              //  N/A                    0x075
    KEY__NONE,              //  N/A                    0x076
    KEY__NONE,              //  N/A                    0x077
    KEY__NONE,              //  N/A                    0x078
    KEY__NONE,              //  N/A                    0x079
    KEY__NONE,              //  N/A                    0x07a
    KEY__NONE,              //  N/A                    0x07b
    KEY__NONE,              //  N/A                    0x07c
    KEY__NONE,              //  N/A                    0x07d
    KEY__NONE,              //  N/A                    0x07e
    KEY__NONE,              //  N/A                    0x07f
    KEY__NONE,              //  N/A                    0x080
    KEY__NONE,              //  N/A                    0x081
    KEY__NONE,              //  N/A                    0x082
    KEY__NONE,              //  N/A                    0x083
    KEY__NONE,              //  N/A                    0x084
    KEY__NONE,              //  N/A                    0x085
    KEY__NONE,              //  N/A                    0x086
    KEY__NONE,              //  N/A                    0x087
    KEY__NONE,              //  N/A                    0x088
    KEY__NONE,              //  N/A                    0x089
    KEY__NONE,              //  N/A                    0x08a
    KEY__NONE,              //  N/A                    0x08b
    KEY__NONE,              //  N/A                    0x08c
    KEY__EQUAL,             //  Numpad =               0x08d
    KEY__NONE,              //  N/A                    0x08e
    KEY__NONE,              //  N/A                    0x08f
    KEY__NONE,              //  N/A                    0x090
    KEY__NONE,              //  N/A                    0x091
    KEY__COLON,             //  :                      0x092
    KEY__MINUS,             //  Underline              0x093
    KEY__NONE,              //  N/A                    0x094
    KEY__NONE,              //  N/A                    0x095
    KEY__NONE,              //  N/A                    0x096
    KEY__NONE,              //  N/A                    0x097
    KEY__NONE,              //  N/A                    0x098
    KEY__NONE,              //  N/A                    0x099
    KEY__NONE,              //  N/A                    0x09a
    KEY__NONE,              //  N/A                    0x09b
    KEY__ENTER,             //  Numpad Enter           0x09c
    KEY__RCTRL,             //  Right Control          0x09d
    KEY__NONE,              //  N/A                    0x09e
    KEY__NONE,              //  N/A                    0x09f
    KEY__NONE,              //  N/A                    0x0a0
    KEY__NONE,              //  N/A                    0x0a1
    KEY__NONE,              //  N/A                    0x0a2
    KEY__NONE,              //  N/A                    0x0a3
    KEY__NONE,              //  N/A                    0x0a4
    KEY__NONE,              //  N/A                    0x0a5
    KEY__NONE,              //  N/A                    0x0a6
    KEY__NONE,              //  N/A                    0x0a7
    KEY__NONE,              //  N/A                    0x0a8
    KEY__NONE,              //  N/A                    0x0a9
    KEY__NONE,              //  N/A                    0x0aa
    KEY__NONE,              //  N/A                    0x0ab
    KEY__NONE,              //  N/A                    0x0ac
    KEY__NONE,              //  N/A                    0x0ad
    KEY__NONE,              //  N/A                    0x0ae
    KEY__NONE,              //  N/A                    0x0af
    KEY__NONE,              //  N/A                    0x0b0
    KEY__NONE,              //  N/A                    0x0b1
    KEY__NONE,              //  N/A                    0x0b2
    KEY__COMMA,             //  Numpad Comma           0x0b3
    KEY__NONE,              //  N/A                    0x0b4
    KEY__DIV,               //  Numpad /               0x0b5
    KEY__NONE,              //  N/A                    0x0b6
    KEY__SYSREQ,            //  SysReq                 0x0b7
    KEY__RALT,              //  Right Alt              0x0b8
    KEY__NONE,              //  N/A                    0x0b9
    KEY__NONE,              //  N/A                    0x0ba
    KEY__NONE,              //  N/A                    0x0bb
    KEY__NONE,              //  N/A                    0x0bc
    KEY__NONE,              //  N/A                    0x0bd
    KEY__NONE,              //  N/A                    0x0be
    KEY__NONE,              //  N/A                    0x0bf
    KEY__NONE,              //  N/A                    0x0c0
    KEY__NONE,              //  N/A                    0x0c1
    KEY__NONE,              //  N/A                    0x0c2
    KEY__NONE,              //  N/A                    0x0c3
    KEY__NONE,              //  N/A                    0x0c4
    KEY__NONE,              //  N/A                    0x0c5
    KEY__NONE,              //  N/A                    0x0c6
    KEY__HOME,              //  Home                   0x0c7
    KEY__UP,                //  Up                     0x0c8
    KEY__PGUP,              //  PgUp                   0x0c9
    KEY__NONE,              //  N/A                    0x0ca
    KEY__LEFT,              //  Left                   0x0cb
    KEY__NONE,              //  N/A                    0x0cc
    KEY__RIGHT,             //  Right                  0x0cd
    KEY__NONE,              //  N/A                    0x0ce
    KEY__END,               //  End                    0x0cf
    KEY__DOWN,              //  Down                   0x0d0
    KEY__PGDN,              //  PgDn                   0x0d1
    KEY__INS,               //  Insert                 0x0d2
    KEY__DEL,               //  Delete                 0x0d3
    KEY__NONE,              //  N/A                    0x0d4
    KEY__NONE,              //  N/A                    0x0d5
    KEY__NONE,              //  N/A                    0x0d6
    KEY__NONE,              //  N/A                    0x0d7
    KEY__NONE,              //  N/A                    0x0d8
    KEY__NONE,              //  N/A                    0x0d9
    KEY__NONE,              //  N/A                    0x0da
    KEY__NONE,              //  N/A                    0x0db
    KEY__NONE,              //  N/A                    0x0dc
    KEY__NONE,              //  N/A                    0x0dd
    KEY__NONE,              //  N/A                    0x0de
    KEY__NONE,              //  N/A                    0x0df
    KEY__NONE,              //  N/A                    0x0e0
    KEY__NONE,              //  N/A                    0x0e1
    KEY__NONE,              //  N/A                    0x0e2
    KEY__NONE,              //  N/A                    0x0e3
    KEY__NONE,              //  N/A                    0x0e4
    KEY__NONE,              //  N/A                    0x0e5
    KEY__NONE,              //  N/A                    0x0e6
    KEY__NONE,              //  N/A                    0x0e7
    KEY__NONE,              //  N/A                    0x0e8
    KEY__NONE,              //  N/A                    0x0e9
    KEY__NONE,              //  N/A                    0x0ea
    KEY__NONE,              //  N/A                    0x0eb
    KEY__NONE,              //  N/A                    0x0ec
    KEY__NONE,              //  N/A                    0x0ed
    KEY__NONE,              //  N/A                    0x0ee
    KEY__NONE,              //  N/A                    0x0ef
    KEY__NONE,              //  N/A                    0x0f0
    KEY__NONE,              //  N/A                    0x0f1
    KEY__NONE,              //  N/A                    0x0f2
    KEY__NONE,              //  N/A                    0x0f3
    KEY__NONE,              //  N/A                    0x0f4
    KEY__NONE,              //  N/A                    0x0f5
    KEY__NONE,              //  N/A                    0x0f6
    KEY__NONE,              //  N/A                    0x0f7
    KEY__NONE,              //  N/A                    0x0f8
    KEY__NONE,              //  N/A                    0x0f9
    KEY__NONE,              //  N/A                    0x0fa
    KEY__NONE,              //  N/A                    0x0fb
    KEY__NONE,              //  N/A                    0x0fc
    KEY__NONE,              //  N/A                    0x0fd
    KEY__NONE,              //  N/A                    0x0fe
    KEY__NONE,              //  N/A                    0x0ff
};



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a DOS Keyboard object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object. 
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
KeyboardDOS*
KeyboardDOS::s_build(
    const KString& iName
)
{
    //  Create the new object.
    KeyboardDOS* pThis = new KeyboardDOS( iName );
 
    //  Initialize the new object.
    pThis->init( );

    //  Send back the new keyboard.
    return( pThis );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: KeyboardDOS
//
//  Description:
//
//      This is the main constructor for the DOS Keyboard object.  This is
//      a protected member.  Clients wishing to create a keyboard object should
//      do so through the factory method.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KeyboardDOS::KeyboardDOS(
    const KString&  iName
)
:
    Keyboard   ( iName )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the keyboard object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
KeyboardDOS::init(
)
{
    //  Call the base class.
    Keyboard::init( );

    //  Install the keyboard handler.
    installKeyboardHandler( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~KeyboardDOS
//
//  Description:
//
//      This is the destructor for a DOS keyboard object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
KeyboardDOS::~KeyboardDOS(
)
{
    //  Uninstall the keyboard handler.
    uninstallKeyboardHandler( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the keyboard object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
KeyboardDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "KeyboardDOS" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member should be called periodically to allow the
//      keyboard object to update itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if a key state has changed.
//      FALSE if no key state has changed.
//
///////////////////////////////////////////////////////////////////////////////
Byte
KeyboardDOS::update(
)
{
    //  Loop while there are more key events.
    for( Byte bI = 0 ; bI < sm_bQueueCount ; bI += 1 )
    {
        //  A convenience to the current event.
        KeyEvent* pEvent = &( sm_aKeyEventQueue[ bI ] );

        //  Set the state of the key and note the last key pressed.
        if( pEvent->m_bPressed )
        {
            m_switchList[ pEvent->m_eKey ]->physicalOn( TRUE );
        }
        else
        {
            m_switchList[ pEvent->m_eKey ]->physicalOn( FALSE );
            m_dwLastSwitchOn = pEvent->m_eKey;
        }
    }
    
    //  Clear the event queue.
    if( sm_bQueueCount > 0 )
    {
        sm_bQueueCount = 0;
        return( TRUE );
    } 
     
    //  If there were no events in the queue then no keys were changed.
    return( FALSE );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: installKeyboardHandler
//
//  Description:
//
//      This member is called to install Replay+'s keyboard handler.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
KeyboardDOS::installKeyboardHandler(
)
{
    //  If the keyboard handler has already been installed then just return.
    if( sm_bKeyboardHandlerInstalled )
    {
        return;
    }

    //  Indicate that the keyboard handler has been installed.
    sm_bKeyboardHandlerInstalled = TRUE;

    //  Clear the event queue.
    sm_bQueueCount = 0;
    
    //  For an interrupt handler, we must lock the data we are going to
    //  use within the handler.
    _go32_dpmi_lock_data( &sm_aKeyEventQueue, sizeof( sm_aKeyEventQueue ) );
    _go32_dpmi_lock_data( &sm_bQueueCount,    sizeof( sm_bQueueCount ) );

    //   We also have to lock the actual code of the handler.
    _go32_dpmi_lock_code(
        s_keyboardHandler,
        ( int )s_keyboardHandlerEnd - ( int )s_keyboardHandler
    );
    
    //  Install the new keyboard handler.
    sm_newKeyboardHandler.pm_selector = _go32_my_cs( );
    sm_newKeyboardHandler.pm_offset   = ( int )s_keyboardHandler;
    _go32_dpmi_get_protected_mode_interrupt_vector( 9, &sm_oldKeyboardHandler );
    _go32_dpmi_allocate_iret_wrapper( &sm_newKeyboardHandler );
    _go32_dpmi_set_protected_mode_interrupt_vector( 9, &sm_newKeyboardHandler );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: uninstallKeyboardHandler
//
//  Description:
//
//      This member is called to uninstall Replay+'s keyboard handler.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
KeyboardDOS::uninstallKeyboardHandler(
)
{
    //  If the handler hasn't been installed then return.
    if( !sm_bKeyboardHandlerInstalled )
    {
        return;
    }

    //  Reinstall the old keyboard handler.
    _go32_dpmi_set_protected_mode_interrupt_vector( 9, &sm_oldKeyboardHandler );
    _go32_dpmi_free_iret_wrapper( &sm_newKeyboardHandler );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_keyboardHandler
//
//  Description:
//
//      This is Replay+'s keyboard handler.  We place key presses/releases
//      into an event queue so that processing is similar to event driven
//      platforms such as X or Windows.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
KeyboardDOS::s_keyboardHandler(
)
{
    //  The following static variable indicates whether this is an extended
    //  scan code.
    static Byte bIsExtended = FALSE;

    //  A pointer to the current event in the queue.
    KeyEvent* pKeyEvent = &( sm_aKeyEventQueue[ sm_bQueueCount ] );

    //  Clear the interrupt flag to prevent multiple calls.  Also preserve
    //  our registers.
    asm( "cli; pusha" );

    //  Port 60 contains the scan code.
    int nScanCode = inportb( 0x60 );

    //   Enable keyboard processing and set/clear the confirmation bit.
    Byte bData = inportb( 0x61 );
    bData |= 0x82;
    outportb( 0x61, bData );
    bData &= 0x7f;
    outportb( 0x61, bData );

    //  We don't bother with the PrintScrn/Pause keys.
    if( nScanCode != 0xe1 ) 
    {
        //  If this is an extended keycode then simply indicate this,
        //  otherwise process the key.
        if( nScanCode == 0xe0 )
        {
            bIsExtended = TRUE;
        }
        else
        {
            //  Has the key been pressed or released?
            pKeyEvent->m_bPressed = ( nScanCode & 0x80 ) ? FALSE : TRUE;

            //  Adjust the key if this is an extended key.
            if( bIsExtended )
            {
                nScanCode += 0x80;
            }
           
            //  Adjust if the key was not pressed.
            if( !pKeyEvent->m_bPressed )
            {
                nScanCode -= 0x80;
            }
 
            //  Set the key of the event.
            pKeyEvent->m_eKey = sm_aKeyCodes[ nScanCode ];

            //  If the key was valid then increment the queue count.
            if( 
                ( pKeyEvent->m_eKey != KEY__NONE ) && ( sm_bQueueCount < 0xff ) 
            )
            {
                sm_bQueueCount += 1;
            }

            //  Reset the extended flag.
            bIsExtended = FALSE;
        }
    }
 
    //  Renable interrupts.
    outportb( 0x20, 0x20 );

    //  Restore our registers and reenable the interrupts.
    asm( "popa; sti" );    
}
//  Marks the end of the keyboard handler.
void KeyboardDOS::s_keyboardHandlerEnd( ) { }
