///////////////////////////////////////////////////////////////////////////////
//
//  File:    netipxd.cpp
//
//  Class:   NetworkIPXDOS
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The NetworkIPXDOS class implements network code between two
//      or more Replay processes.  The IPX protocol is used for communication.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "netipxd.h"
#include "replay.h"
#include "config.h"
#include "dbuffer.h"
#include "clock.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
Byte                 NetworkIPXDOS::sm_bDefaultSoftInt = 0x60;
Word                 NetworkIPXDOS::sm_wDefaultSocketNumber = 0x869c;
_go32_dpmi_registers NetworkIPXDOS::sm_cbRegs;
_go32_dpmi_seginfo   NetworkIPXDOS::sm_cbInfo;
NetworkIPXDOS*       NetworkIPXDOS::sm_pThis = NULL;
DWord                NetworkIPXDOS::sm_dwConnectTimeout = 10000000;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: NetworkIPXDOS
//
//  Description:
//
//      This is the main constructor for a network object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetworkIPXDOS::NetworkIPXDOS(
    const KString&  iName
)
:
    Network             ( iName ),
    m_bSoftInt          ( sm_bDefaultSoftInt ),
    m_pConnectList      ( NULL ),
    m_wSocketNumber     ( sm_wDefaultSocketNumber ),
    m_pSendPacketList   ( NULL ),
    m_pRecvPacketList   ( NULL ),
    m_dwNumPackets      ( 0 ),
    m_dwCallbackAddress ( 0x000000 )
{
    //  The configuration object.
    Configuration& config = Configuration::s_instance( );


    //  Cache the pointer to this object.
    ASSERT( sm_pThis == NULL );
    sm_pThis = this;

    //  If the network is disabled then there's no need to continue.
    if( !m_bEnabled )
    {
        return;
    }

    //  Allocate space for the list of connections.
    m_pConnectList = new NodeAddress[ getMaxConnections( ) ];

    //  Assign the socket number.
    int32 nSocketNumber;
    if( config.getParam( "-socket", &nSocketNumber ) )
    {
        m_wSocketNumber = nSocketNumber;
    }


    //  Assign the software interrupt used by IPX.
    int32 nSoftwareInterrupt;
    if( config.getParam( "-irq", &nSoftwareInterrupt ) )
    {
        m_bSoftInt = ( Byte )nSoftwareInterrupt; 
    }


    //  Perform the initialization and if it succeeded then get the local
    //  address of the machine..
    initializeIPX( );
    if( m_bEnabled )
    {
        getLocalAddress( );
    } 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~NetworkIPXDOS
//
//  Description:
//
//      This is the destructor for a network object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
NetworkIPXDOS::~NetworkIPXDOS(
)
{
    //  Free up the list of connections.
    delete [] m_pConnectList;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the network object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
NetworkIPXDOS::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "NetworkIPXDOS" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: initializeIPX
//
//  Description:
//
//      This function attempts to initialize the IPX protocol.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::initializeIPX(
)
{
    //  Registers for communicating outside of protected mode.
    _go32_dpmi_registers r;


    //  Clear the registers.
    memset( ( void* )&r, 0x00, sizeof( r ) );

    //  The following code checks to ensure that the IPX network is available.
    //  A return of 0xff indicates success.
    r.x.ax = 0x7a00;
    _go32_dpmi_simulate_int( 0x2f, &r );
    if( r.h.al != 0xff )
    {
        CHECK0( FALSE, "IPX Network not present" );
        m_bEnabled = FALSE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getLocalAddress
//
//  Description:
//
//      This function is called to retrieve the network address of this
//      machine.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::getLocalAddress(
)
{
    //  Registers for communicating outside of protected mode.
    _go32_dpmi_registers r;

    //  A DOS buffer that the local address will be retrieved into.
    DOSBuffer localAddress( "LocalAddress", 32 );


    //  Clear the registers and the local address.
    memset( ( void* )&r, 0x00, sizeof( r ) );
    memset( ( void* )&m_localAddress, 0x00, sizeof( m_localAddress ) );


    //  The address is retrieved by calling subfunction 9.
    r.x.bx = 0x0009;
    r.x.es = localAddress.getSegment( );
    r.x.si = localAddress.getOffset( );

    //  Place the local address into the DOS buffer, perform the interrupt
    //  and then retrieve the value.
    localAddress.in( ( void* )&m_localAddress, sizeof( m_localAddress ) );
    _go32_dpmi_simulate_int( m_bSoftInt, &r );
    localAddress.out( ( void* )&m_localAddress, sizeof( m_localAddress ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: openNetwork
//
//  Description:
//
//      This member is called to open the network connection.
//
//  Parameters:
//
//      eType (input)
//          Indicates whether or not the connection should be opened as
//          client or server.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkIPXDOS::openNetwork(
    ConnectType eType
)
{
    ASSERT( m_bEnabled );

    //  Make sure either client or server was specified.
    CONFIRM( eType != UNKNOWN, "Network must be opened as client or server" );

    //  Call the base class.
    Network::openNetwork( eType );

    //  Perform various initialization.
    openSocket( );

    //  Initialize the packet receiving callback.
    initializeCallback( );

    //  If we're a client then we only need 1 send and 1 receive packet, 
    //  but if we're a server we need enough packets for each possible 
    //  connection.
    initializePackets( eType == CLIENT ? 1 : getMaxConnections( ) );
    
    //  The network was successfully opened.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: openSocket
//
//  Description:
//
//      This function is called to open the socket that is used for 
//      IPX communication.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::openSocket(
)
{
    //  Registers for communicating outside of protected mode.
    _go32_dpmi_registers r;


    //  Clear the registers.
    memset( ( void* )&r, 0x00, sizeof( r ) );

    //  Perform the open (note that the socket number must be converted to
    //  big endian).
    r.x.bx = 0x0000;
    r.h.al = 0x00;
    r.x.dx = swapEndian( m_wSocketNumber );
    _go32_dpmi_simulate_int( m_bSoftInt, &r );
    CONFIRM( 
        r.h.al == 0x00, 
        "Could not open socket 0x%04x (0x%02x)",
        m_wSocketNumber,
        r.h.al
    );
    m_wSocketNumber = swapEndian( r.x.dx );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: closeSocket
//
//  Description:
//
//      This function is called to close the socket that is used for 
//      IPX communication.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::closeSocket(
)
{
    //  Registers for communicating outside of protected mode.
    _go32_dpmi_registers r;


    //  Clear the registers.
    memset( ( void* )&r, 0x00, sizeof( r ) );

    //  Perform the open (note that the socket number must be converted to
    //  big endian).
    r.x.bx = 0x0001;
    r.x.dx = swapEndian( m_wSocketNumber );
    _go32_dpmi_simulate_int( m_bSoftInt, &r );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: initializeCallback
//
//  Description:
//
//      This function is called to initialize the callback that is used for
//      receipt of packets.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::initializeCallback(
)
{
    //  Segment/Offset of the callback.
    Word wSegment;
    Word wOffset;


    //  Clear the registers & information.
    memset( ( void* )&sm_cbRegs, 0x00, sizeof( sm_cbRegs ) );
    memset( ( void* )&sm_cbInfo, 0x00, sizeof( sm_cbInfo ) );

    //  Create the callback.
    sm_cbInfo.pm_offset = ( unsigned int )s_packetReceived;
    _go32_dpmi_allocate_real_mode_callback_retf( &sm_cbInfo, &sm_cbRegs );
    
    //  Save the address for use later.
    wSegment            = sm_cbInfo.rm_segment;
    wOffset             = sm_cbInfo.rm_offset;
    m_dwCallbackAddress = ( ( DWord )wSegment << 16 ) | wOffset;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: initializePackets
//
//  Description:
//
//      This function is called to create and initialize the number of
//      packets specified.
//
//  Parameters:
//
//      dwNumPackets (input)
//          The number of packets (each of send and recieve) to create
//          and initialize.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::initializePackets(
    const DWord dwNumPackets
)
{
    //  A temporary pointer to a packet list.
    Packet* pPacket;


    ASSERT( dwNumPackets > 0 );
    m_dwNumPackets = dwNumPackets;


    //  Allocate the send packet list.
    pPacket = new Packet[ dwNumPackets ];
    m_pSendPacketList = pPacket;

    //  Initialize each of the send packets.
    for( DWord dwI = 0 ; dwI < dwNumPackets ; dwI += 1 )
    {
        //  Allocate the buffers.
        pPacket->m_pECBBuffer = new DOSBuffer( "ECB", sizeof( ECB ) );
        pPacket->m_pIPXBuffer = new DOSBuffer( "IPX", sizeof( IPX ) );

        //  Clear the structures.
        memset( ( void* )&( pPacket->m_ecb ), 0x00, sizeof( ECB ) );
        memset( ( void* )&( pPacket->m_ipx ), 0x00, sizeof( IPX ) );

        //  Initialize relevant data.
        pPacket->m_ecb.m_wSocketNumber     = swapEndian( m_wSocketNumber );
        pPacket->m_ecb.m_wFragmentCount    = 1;
        pPacket->m_ecb.m_dwFragmentAddress = 
            pPacket->m_pIPXBuffer->getAddress( );
        pPacket->m_ecb.m_wFragmentSize     = sizeof( IPX );
        memset( ( void* )( pPacket->m_ecb.m_abImmediateAddress ), 0xff, 6 );
        pPacket->m_ipx.m_wCheckSum         = 0xffff;
        memcpy( 
            ( void* )( pPacket->m_ipx.m_destination.m_abNetwork ),
            ( void* )( m_localAddress.m_abNetwork ),
            4
        );
        memset( ( void* )( pPacket->m_ipx.m_destination.m_abNode ), 0xff, 6 );
        pPacket->m_ipx.m_destination.m_wSocketNumber = 
            swapEndian( m_wSocketNumber );
        memcpy( 
            ( void* )( pPacket->m_ipx.m_source.m_abNetwork ),
            ( void* )( m_localAddress.m_abNetwork ),
            4
        );
        memcpy( 
            ( void* )( pPacket->m_ipx.m_source.m_abNode ),
            ( void* )( m_localAddress.m_abNode ),
            4
        );
        pPacket->m_ipx.m_source.m_wSocketNumber = swapEndian( m_wSocketNumber );
        
        //  Copy the ECB and IPX into the DOS buffers.
        pPacket->m_pECBBuffer->in( 
            ( void* )&( pPacket->m_ecb ), sizeof( ECB ) 
        );
        pPacket->m_pIPXBuffer->in( 
            ( void* )&( pPacket->m_ipx ), sizeof( IPX ) 
        ) ; 
        
        //  Move to the next send packet.
        pPacket += 1;
    }


    //  Allocate the receive packet list.
    pPacket = new Packet[ dwNumPackets ];
    m_pRecvPacketList = pPacket;

    //  Initialize each of the receive packets.
    for( DWord dwI = 0 ; dwI < dwNumPackets ; dwI += 1 )
    {
        //  Allocate the buffers.
        pPacket->m_pECBBuffer = new DOSBuffer( "ECB", sizeof( ECB ) );
        pPacket->m_pIPXBuffer = new DOSBuffer( "IPX", sizeof( IPX ) );

        //  Clear the structures.
        memset( ( void* )&( pPacket->m_ecb ), 0x00, sizeof( ECB ) );
        memset( ( void* )&( pPacket->m_ipx ), 0x00, sizeof( IPX ) );

        //  Initialize relevant data.
        pPacket->m_ecb.m_wSocketNumber     = swapEndian( m_wSocketNumber );
        pPacket->m_ecb.m_dwESRAddress      = m_dwCallbackAddress;
        pPacket->m_ecb.m_wFragmentCount    = 1;
        pPacket->m_ecb.m_dwFragmentAddress = 
            pPacket->m_pIPXBuffer->getAddress( );
        pPacket->m_ecb.m_wFragmentSize     = sizeof( IPX );
        
        //  Copy the ECB and IPX into the DOS buffers.
        pPacket->m_pECBBuffer->in( 
            ( void* )&( pPacket->m_ecb ), sizeof( ECB ) 
        );
        pPacket->m_pIPXBuffer->in( 
            ( void* )&( pPacket->m_ipx ), sizeof( IPX )
        ) ; 
        
        //  Listen for a packet.
        listenForPacket( pPacket );

        //  Transfer the message from the DOS Buffer.
        pPacket->m_pIPXBuffer->out( 
            ( void* )&( pPacket->m_ipx ), sizeof( IPX )
        );

        //  Move to the next receive packet.
        pPacket += 1;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: listenForPacket
//
//  Description:
//
//      This member is called to listen for a packet on the connection.
//
//  Parameters:
//
//      pPacket (input/output)
//          The desination packet for the listen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::listenForPacket(
    Packet* pPacket
)
{
    //  Registers for communicating outside of protected mode.
    _go32_dpmi_registers r;


    ASSERT( pPacket != NULL );

    //  Clear the registers.
    memset( ( void* )&r, 0x00, sizeof( r ) );

    //  Perform the listen (subfunction 4).
    r.x.bx = 0x0004;
    r.x.es = pPacket->m_pECBBuffer->getSegment( );
    r.x.si = pPacket->m_pECBBuffer->getOffset( );
    _go32_dpmi_simulate_int( m_bSoftInt, &r );
    ASSERT( r.h.al == 0x00 );

    //  Transfer the contents of the ECB to the real buffer.
    pPacket->m_pECBBuffer->out( ( void* )&( pPacket->m_ecb ), sizeof( ECB ) );
}

    

///////////////////////////////////////////////////////////////////////////////
//
//  Function: sendPacket
//
//  Description:
//
//      This member is called to send a packet.  The packet can be
//      broadcast or sent to a specific connection.
//
//  Parameters:
//
//      pMsg (output)
//          A pointer to the buffer containing the message to send.
//
//      nConnection (input)
//          The connection the packet is to be sent to.  -1 indicates
//          that the packet should be broadcast.
//
//  Returns:
//
//      TRUE  if errors were detected in the sending.
//      FALSE if everything is A-OK.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkIPXDOS::sendPacket(
    Msg*        pMsg, 
    const int32 nConnection /* = -1 */
)
{
    //  A pointer to the packet being sent.
    Packet* pPacket;


    ASSERT( nConnection < ( int32 )m_dwNumConnections );

    //  Assign the packet to send.  We use the packet corresponding to the
    //  connection specified or the first one if broadcasting.
    if( nConnection < 0 )
    {
        pPacket = &( m_pSendPacketList[ 0 ] );
        for( Byte bI = 0 ; bI < 6 ; bI += 1 )
        {
            pPacket->m_ecb.m_abImmediateAddress[ bI ] = 0xff;
            pPacket->m_ipx.m_destination.m_abNode[ bI ] = 0xff;
        }
    }
    else
    {
        pPacket = &( m_pSendPacketList[ nConnection ] );
        for( Byte bI = 0 ; bI < 6 ; bI += 1 )
        {
            pPacket->m_ecb.m_abImmediateAddress[ bI ] = 
                m_pConnectList[ nConnection ].m_abNode[ bI ];
            pPacket->m_ipx.m_destination.m_abNode[ bI ] =   
                m_pConnectList[ nConnection ].m_abNode[ bI ];
        }
    }

    //  Copy the message into the packet.
    memcpy( ( void* )&( pPacket->m_ipx.m_msg ), ( void* )pMsg, sizeof( Msg ) );

    //  Copy the message into the buffer.
    pPacket->m_pECBBuffer->in(
        ( void* )&( pPacket->m_ecb ), sizeof( ECB )
    );
    pPacket->m_pIPXBuffer->in( 
        ( void* )&( pPacket->m_ipx ), sizeof( IPX ) 
    );


    // Perform the interrupt that sends the message.
    _go32_dpmi_registers r;
    memset( ( void* )&r, 0x00, sizeof( r ) );
    r.x.bx = 0x0003;
    r.x.es = pPacket->m_pECBBuffer->getSegment( );
    r.x.si = pPacket->m_pECBBuffer->getOffset( );
    _go32_dpmi_simulate_int( m_bSoftInt, &r );

    Byte bReturn = r.h.al ? TRUE : FALSE;

    //  Wait until the packet has successfully been sent.
    memset( ( void* )&r, 0x00, sizeof( r ) );
    r.x.bx = 0x000a;
    _go32_dpmi_simulate_int( m_bSoftInt, &r );

    //  All finished.
    return( bReturn );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getPacket
//
//  Description:
//
//      This member is called to retrieve a packet.  The packet can come
//      from any sender or from a particular sender if specified.
//
//  Parameters:
//
//      pMsg (output)
//          A pointer to the buffer to place the received message in.
//
//      nConnection (input)
//          The connection the packet is to be retrieved for.  -1 indicates
//          that the packet can come from any connection.
//
//  Returns:
//
//      A pointer to the packet if found.
//      NULL otherwise.
//
///////////////////////////////////////////////////////////////////////////////
NetworkIPXDOS::Packet*
NetworkIPXDOS::getPacket(
    Msg*        pMsg, 
    const int32 nConnection /* = -1 */
)
{
    ASSERT( nConnection < ( int32 )m_dwNumConnections );

    //  Loop through each of the receive packets looking for one that is
    //  available for receipt and matches the specified connection.
    for( DWord dwI = 0 ; dwI < m_dwNumPackets ; dwI += 1 )
    {
        Packet* pPacket = &( m_pRecvPacketList[ dwI ] );

        //  If the packet isn't available, continue on to the next packet.
        if( pPacket->m_ecb.m_bInUse )
        {
            continue;
        }

        //  If there is no connection specified or the sender of the packet
        //  matches the specified connection then we have found our packet.
        if( 
            ( nConnection < 0 ) || 
            ( 
                memcmp(
                    ( void* )( m_pConnectList[ nConnection ].m_abNode ),
                    ( void* )( pPacket->m_ipx.m_source.m_abNode ), 
                    6
                ) == 0
            )
        )
        {
            ASSERT( pPacket->m_ecb.m_bCompletionCode == 0 );
 
            //  Copy the message out.
            memcpy(
                ( void* )pMsg,
                ( void* )&( pPacket->m_ipx.m_msg ), 
                sizeof( Msg )
            );

            //  Stick the packet back on the eligible to receive list.
            listenForPacket( pPacket );
             
            return( pPacket );
        }
    }

    //  A packet wasn't found.
    return( NULL );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: acceptClient
//
//  Description:
//
//      This member is called when the server is to accept client connections.
//      It performs a non-blocking accept.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if a client connected
//      FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkIPXDOS::acceptClient(
)
{
    //  Client connections can only be accepted if we are serving as the 
    //  server.
    CONFIRM( m_eConnectType == SERVER, "Must be server to accept clients" );

    //  We can't accept more connections than allowed.
    CONFIRM( 
        getNumConnections( ) < getMaxConnections( ), 
        "Cannot accept more than %d connections.",   
        getMaxConnections( )
    );

    //  If there is no packet pending or the pending packet is not a client
    //  message then there are no clients attempting to connect.
    Msg msg;
    Packet* pPacket = getPacket( &msg );
    if( ( pPacket == NULL ) || ( msg.eType != MSG_CLIENT ) )
    {
        return( FALSE );
    }

    //  OK, a client is attempting to connect, add their address to the list
    //  of connections.
    memcpy( 
        ( void* )( m_pConnectList[ m_dwNumConnections ].m_abNode ),
        ( void* )( pPacket->m_ipx.m_source.m_abNode ),
        6
    );
    m_dwNumConnections += 1;
   
    //  Send out confirmation to the client that they have been accepted.
    MsgServer msgServer;
    msgServer.eType = MSG_SERVER;
    sendPacket( &msg, m_dwNumConnections - 1 );

    //  A client has connected.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: connectToServer
//
//  Description:
//
//      This member is called when the client is to connect to the server.
//      It performs a non-blocking connect.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if the client connected
//      FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
NetworkIPXDOS::connectToServer(
)
{
    //  We can only connect to a server if we're a client.
    CONFIRM( m_eConnectType == CLIENT, "Must be client to connect to server." );

    //  We can only connect to a server if we haven't already connected.
    CONFIRM( m_dwNumConnections == 0, "Can't connect - already connected." );

    //  Broadcast that we are a client and want connect.
    MsgClient msgClient;
    msgClient.eType = MSG_CLIENT;
    sendPacket( &msgClient );

    //  Now we wait a while for a response.
    Clock* pClock = Replay::s_instance( ).getClock( );
    DWord dwStart = pClock->getTime( ) ;
    while( ( pClock->getTime( ) - dwStart ) < sm_dwConnectTimeout )
    {
        //  If we have received a "server" packet then we have connected.
        Msg msg;
        Packet* pPacket = getPacket( &msg );
        if( pPacket != NULL )
        {
            if( msg.eType == MSG_SERVER )
            {
                //  OK, the server has responded, add its address to our
                //  list of connections.
                memcpy( 
                    ( void* )( m_pConnectList[ m_dwNumConnections ].m_abNode ),
                    ( void* )( pPacket->m_ipx.m_source.m_abNode ),
                    6
                );
                m_dwNumConnections += 1;

                return( TRUE );
            }
        }
    } 
        
    //  If we have reached here then we have timed out and no connection
    //  has been made.
    return( FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: closeNetwork
//
//  Description:
//
//      This member is called to close the network down completely.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::closeNetwork(
)
{
    //  If the network isn't opened then just return.
    if( !isOpen( ) )
    {
        return;
    }

    //  Close each connection. 
    while( m_dwNumConnections > 0 )
    {
        closeConnection( 0 );
    }

    //  Close the socket.
    closeSocket( );

    //  Free the packet lists.
    for( DWord dwI = 0 ; dwI < m_dwNumPackets ; dwI += 1 )
    {
        delete m_pSendPacketList[ dwI ].m_pECBBuffer;
        delete m_pSendPacketList[ dwI ].m_pIPXBuffer;
        delete m_pRecvPacketList[ dwI ].m_pECBBuffer;
        delete m_pRecvPacketList[ dwI ].m_pIPXBuffer;
    }
    delete [] m_pSendPacketList;
    delete [] m_pRecvPacketList;

    //  Call the base class.
    Network::closeNetwork( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: closeConnection
//
//  Description:
//
//      This member is called to close a specific connection within the
//      network.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to close down.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::closeConnection(
    DWord dwConnection
)
{
    //  Make sure that the specified connection exists.
    CONFIRM( 
        dwConnection < m_dwNumConnections,
        "Connection %d cannot be closed as it does not exist.",
        dwConnection
    );

    //  We now have one less entry in the list.
    m_dwNumConnections -= 1;

    //  If this wasn't the last connection then move the connection at the
    //  end of the list into this position.
    if( dwConnection < m_dwNumConnections )
    {
        memcpy(
            ( void* )( m_pConnectList[ dwConnection ].m_abNode ),
            ( void* )( m_pConnectList[ m_dwNumConnections ].m_abNode ),
             6
        );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readMsg
//
//  Description:
//
//      This member is called to read a message from the specified connection.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to read from.
//
//      pMsg (output)
//          The buffer to hold the message.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::readMsg(
    const DWord dwConnection,
    Msg*        pMsg
)
{
    ASSERT( dwConnection < m_dwNumConnections );

    //  If there is a pending message for the specified connection then read
    //  it.  Otherwise, indicate that there is no message to read.
    if( !getPacket( pMsg, dwConnection ) )
    {
        pMsg->eType = MSG_NONE;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: writeMsg
//
//  Description:
//
//      This member is called to write a message to the specified connection.
//
//  Parameters:
//
//      dwConnection (input)
//          Indicates which connection to write to.
//
//      pMsg (output)
//          The buffer that contains the message.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
NetworkIPXDOS::writeMsg(
    const DWord dwConnection,
    Msg*        pMsg
)
{
    ASSERT( dwConnection < m_dwNumConnections );
  
    //  Write data to the connection.  If there is a problem then close the
    //  connection.
    if( sendPacket( pMsg, dwConnection ) )
    {
        if( m_eConnectType == SERVER )
        {
            closeConnection( dwConnection );
        }
        else
        {
            closeNetwork( );
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_packetReceived
//
//  Description:
//
//      This member is called whenever a packet arrives.
//
//  Parameters:
//
//      pRegs (input)
//          The registers.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
NetworkIPXDOS::s_packetReceived( 
    _go32_dpmi_registers* pRegs 
)
{
    pRegs->d.eax = 4;

    //  I'm not sure how to determine which packet has been received so
    //  for now, we move the information from all receive packets over.
    for( DWord dwI = 0 ; dwI < sm_pThis->m_dwNumPackets ; dwI += 1 )
    {
        //  A convenience to the current packet.
        Packet* pPacket = &( sm_pThis->m_pRecvPacketList[ dwI ] );
        
        //  Transfer the buffers.
        pPacket->m_pECBBuffer->out( 
            ( void* )&( pPacket->m_ecb ), sizeof( ECB )
        );
        pPacket->m_pIPXBuffer->out( 
            ( void* )&( pPacket->m_ipx ), sizeof( IPX )
        );
    }
}
