///////////////////////////////////////////////////////////////////////////////
//
//  File:       netipxd.h
//
//  Class:      NetworkIPXDOS - Concrete
//  
//  Hierarchy:  NetworkIPXDOS - Network - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      The NetworkIPXDOS class implements network code between two
//      or more Replay processes.  The IPX protocol is used for communication.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _netipxd_h_
#define _netipxd_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.

//  Application Headers.
#include "network.h"
#include "dpmi.h"


//  Forward class declarations.
class DOSBuffer;

///////////////////////////////////////////////////////////////////////////////
//  NetworkIPXDOS Class Definition
///////////////////////////////////////////////////////////////////////////////
class NetworkIPXDOS : public Network
{
    public:

        //  Creation/Deletion.
                 NetworkIPXDOS ( const KString& iName );
        virtual ~NetworkIPXDOS ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Connection.
        virtual Byte openNetwork     ( ConnectType eType );
        virtual Byte acceptClient    ( );
        virtual Byte connectToServer ( );
        virtual void closeNetwork    ( );
        virtual void closeConnection ( DWord dwConnection );

        //  Message passing.
        virtual void readMsg  ( const DWord dwConnection, Msg* pMsg );
        virtual void writeMsg ( const DWord dwConnection, Msg* pMsg );


    protected:

        //  The following class holds a node address.
        class NodeAddress
        {
            public:
                Byte m_abNode[ 6 ];
        };

        //  The following class holds a local network address.
        class LocalAddress
        {
            public:
                Byte m_abNetwork[ 4 ];
                Byte m_abNode[ 6 ];
        };

        //  The following class holds a network address.
        class NetworkAddress
        {
            public:
                Byte m_abNetwork[ 4 ];
                Byte m_abNode[ 6 ];
                Word m_wSocketNumber;
        };
    
        //  The following class holds an Event Control Block (ECB).
        class ECB
        {
            public:
                DWord m_dwLinkAddress;
                DWord m_dwESRAddress;
                Byte  m_bInUse;
                Byte  m_bCompletionCode;
                Word  m_wSocketNumber;
                Byte  m_abIPXWorkspace[ 4 ];
                Byte  m_abDriverWorkspace[ 12 ];
                Byte  m_abImmediateAddress[ 6 ];
                Word  m_wFragmentCount;
                DWord m_dwFragmentAddress;
                Word  m_wFragmentSize;
        };

        //  The following class holds the IPX Header.
        class IPX
        {
            public:
                Word           m_wCheckSum;
                Word           m_wLength;
                Byte           m_bTransportControl;
                Byte           m_bPacketType;
                NetworkAddress m_destination;
                NetworkAddress m_source;
                Msg            m_msg;
        };

        //  The following class holds a packet.
        class Packet
        {
            public:
                ECB        m_ecb;
                IPX        m_ipx;
                DOSBuffer* m_pECBBuffer;
                DOSBuffer* m_pIPXBuffer;
        };
                 
        //  Member Functions.
        Word    swapEndian         ( const Word wSource ) const;
        void    initializeIPX      ( );
        void    getLocalAddress    ( );
        void    openSocket         ( );
        void    closeSocket        ( );
        void    initializeCallback ( );
        void    initializePackets  ( const DWord dwNumPackets );
        void    listenForPacket    ( Packet* pPacket );
        Byte    sendPacket         ( Msg* pMsg, const int32 nConnection = -1 );
        Packet* getPacket          ( Msg* pMsg, const int32 nConnection = -1 );

        //  Member Data.
        Byte         m_bSoftInt;
        LocalAddress m_localAddress;
        NodeAddress* m_pConnectList;
        Word         m_wSocketNumber;
        Packet*      m_pSendPacketList;
        Packet*      m_pRecvPacketList;
        DWord        m_dwNumPackets;
        DWord        m_dwCallbackAddress;

        //  Static Member Functions.
        static void s_packetReceived ( _go32_dpmi_registers* pRegs );

        //  Static Member Data.
        static Byte                 sm_bDefaultSoftInt;
        static Word                 sm_wDefaultSocketNumber;
        static _go32_dpmi_registers sm_cbRegs;
        static _go32_dpmi_seginfo   sm_cbInfo;
        static NetworkIPXDOS*       sm_pThis;
        static DWord                sm_dwConnectTimeout;


    private:

};

///////////////////////////////////////////////////////////////////////////////
//  NetworkIPXDOS Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Word
NetworkIPXDOS::swapEndian(
    const Word wSource
) const
{
    return( ( wSource >> 8 ) | ( ( wSource & 0xff ) << 8 ) );
}

    
#endif
