///////////////////////////////////////////////////////////////////////////////
//
//  File:    gfile.cpp
//
//  Class:   GameFile
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The GameFile class serves a base class for a group of classes that
//      are responsible for manipulating files used exclusively for the 
//      emulated games (i.e. ROMs, Samples, Information, etc.).  
//
//      The reason why this is a separate hierarchy from the AppFile is that
//      an application file (e.g. Registry) is always a plain disk file found
//      in a specific location that can be both read and written to.  Game
//      files, however, can be found in one of numerous places depending
//      on the preference of the user.  Maybe the user likes to keep their
//      ROMs uncompressed in directories named for the games, maybe they
//      like to keep them in a zipfile, maybe they like to have them gziped
//      or unix compressed.  
//
//      GameFile provides an abstract interface to retrieve these files
//      and the subclasses are used to implement a certain method.
//
//      An additional note is that Game Files are always read from and never
//      written to which simplifies the interface.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "gfile.h"




///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameFile
//
//  Description:
//
//      This is the main constructor for a game file object.
//
//  Parameters:
//
//      gameId (input)
//          The id of the game to which the file belongs.
//
//      fileName (input)
//          The name of the file.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameFile::GameFile(
    const KString& gameId, 
    const KString& fileName 
)
:
    RepBase           ( fileName ),
    m_gameId          ( gameId ),
    m_fileName        ( fileName ),
    m_pbBuffer        ( NULL ),
    m_dwLength        ( 0 )
{
    //  No default behaviour.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameFile
//
//  Description:
//
//      This is the destructor for a game file object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameFile::~GameFile(
)
{
    //  Free the buffer.
    delete [] m_pbBuffer;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: read
//
//  Description:
//
//      This member is used to read the file into the specified buffer.
//
//  Parameters:
//
//      pbDest (output)
//          The buffer to contain the file (or a piece thereof).
//
//      dwMaxLength (input)
//          The maximum number of bytes to place in the buffer.
//
//      dwOffset (input)
//          The offset within the file to start copying the file from.
//
//  Returns:
//
//      The number of characters read.
//
///////////////////////////////////////////////////////////////////////////////
const
DWord
GameFile::read(
    Byte*       pbDest, 
    const DWord dwMaxLength, 
    const DWord dwOffset /* = 0x0000 */
)
{
    //  Make sure a file is open.
    CONFIRM( isOpen( ), "A GameFile is being read that is not opened" );

    //  Make sure the offset is valid.
    CONFIRM( 
        dwOffset < m_dwLength, 
        "Attempting to read 0x%08x from a 0x%08x byte file (%s).",
        dwOffset,
        m_dwLength,
        m_fileName.data( )
    );

    //  Calculate the number of characters to copy.
    DWord dwCharsToCopy;

    //  The number of characters available to copy from the offset to the
    //  end of the file.
    DWord dwCharsAvailable = m_dwLength - dwOffset;


    //  Determine the number of characters to copy.
    if( dwCharsAvailable < dwMaxLength )
    {
        dwCharsToCopy = dwCharsAvailable;
    }
    else
    {
        dwCharsToCopy = dwMaxLength;
    }

    //  Do the copy.
    memcpy( 
        ( void* )pbDest, ( void* )( m_pbBuffer + dwOffset ), dwCharsToCopy
    );

    //  Return the number of characters copied.
    return( dwCharsToCopy );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: readSkip
//
//  Description:
//
//      This member is used to read the file into the specified buffer 
//      while skipping every other byte.  This is typically used for ROMs
//      that are stored with the even bytes in one chip and the odd bytes
//      in another chip.
//
//  Parameters:
//
//      pbDest (output)
//          The buffer to contain the file (or a piece thereof).
//
//      dwMaxLength (input)
//          The maximum number of bytes to place in the buffer.
//
//      dwOffset (input)
//          The offset within the file to start copying the file from.
//
//  Returns:
//
//      The number of characters read.
//
///////////////////////////////////////////////////////////////////////////////
const 
DWord
GameFile::readSkip(
    Byte*       pbDest, 
    const DWord dwMaxLength, 
    const DWord dwOffset /* = 0x0000 */
)
{
    //  A pointer used to traverse the destination buffer.
    Byte* pbTraverse = pbDest;


    //  Make sure a file is open.
    CONFIRM( isOpen( ), "A GameFile is being read that is not opened" );

    //  Make sure the offset is valid.
    CONFIRM( 
        dwOffset < m_dwLength, 
        "Attempting to read 0x%08x from a 0x%08x byte file (%s).",
        dwOffset,
        m_dwLength,
        m_fileName.data( )
    );

    //  Calculate the number of characters to copy.
    DWord dwCharsToCopy;

    //  The number of characters available to copy from the offset to the
    //  end of the file.
    DWord dwCharsAvailable = m_dwLength - dwOffset;


    //  Determine the number of characters to copy.
    if( dwCharsAvailable < dwMaxLength )
    {
        dwCharsToCopy = dwCharsAvailable;
    }
    else
    {
        dwCharsToCopy = dwMaxLength;
    }

    //  Do the copy.
    for( DWord dwI = 0 ; dwI < dwCharsToCopy ; dwI += 1 )
    {
        *pbTraverse  = m_pbBuffer[ dwOffset + dwI ];
        pbTraverse  += 2;
    }

    //  Return the number of characters copied.
    return( dwCharsToCopy );
}
