///////////////////////////////////////////////////////////////////////////////
//
//  File:    ctrlmap.cpp
//
//  Class:   CtrlMap
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to map a control in the game (e.g. Player 1 Up,
//      Player 2 Fire, Coin 1 Inserted, etc.) to a user's control (e.g.
//      The 'A' key, Joystick Up, etc.).
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "ctrlmap.h"
#include "config.h"
#include "registry.h"
#include "keyb.h"
#include "joy.h"
#include "input.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data Initialization.
///////////////////////////////////////////////////////////////////////////////
//  The global control mappings.  This is declared as an an array of 
//  indeterminate size instead of size GLOBAL_CONTROL_COUNT so that we can 
//  check to make sure that all controls have defaults.  If we had specified 
//  the size of GLOBAL_CONTROL_COUNT we could have left entries out of this 
//  list without the compiler complaining.  Instead, a size mismatch can
//  be caught with a simple ASSERT.
KPtrList<CtrlMap> CtrlMap::sm_globalControlList( 32 );

//  The following indicates whether the global controls have been initialized
//  or not.  Initializing global controls involves loading saved values.
Byte CtrlMap::sm_bGlobalControlsInitialized = FALSE;




///////////////////////////////////////////////////////////////////////////////
//
//  Function: CtrlMap
//
//  Description:
//
//      This is the main constructor for a control map object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWhen (input)
//          This indicates when the key is active.  This can be used in
//          network games to determine whether or not a key is controlled
//          by a particular player.
//
//      eGlobal (input)
//          The global control this is related to.
//
//      pbLocation (input)
//          A pointer to the byte that the value of the control is to
//          be written to.
//
//      bMask (input)
//          The bits affected by the control.
//
//      eDefaultKey (input)
//          The default key that this control is mapped to.
//
//      eDefaultJoy (input)
//          The default joystick control that this control is mapped to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CtrlMap::CtrlMap(
    const KString&        iName,
    DWord                 dwWhen,
    GlobalControl         eGlobal,
    Byte*                 pbLocation,
    Byte                  bMask,
    Keyboard::KeyboardKey eDefaultKey  /* = Keyboard::KEY__NONE */,
    Joystick::JoystickJoy eDefaultJoy  /* = Joystick::JOY_NONE */
)
:
    RepBase                       ( iName ),
    m_dwWhen                      ( dwWhen ),
    m_eGlobal                     ( eGlobal ),
    m_pbLocation                  ( pbLocation ),
    m_bMask                       ( bMask )
{
    //  Call a common construction member function.
    CtrlMapCtor( eGlobal, eDefaultKey, eDefaultJoy );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: CtrlMap
//
//  Description:
//
//      This is a secondary constructor for a control map object that allows
//      the control map to be bound to an input.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWhen (input)
//          This indicates when the key is active.  This can be used in
//          network games to determine whether or not a key is controlled
//          by a particular player.
//
//      eGlobal (input)
//          The global control this is related to.
//
//      pInput (input)
//          A pointer to the input that the control is bound to.
//
//      bMask (input)
//          The bits affected by the control.
//
//      eDefaultKey (input)
//          The default key that this control is mapped to.
//
//      eDefaultJoy (input)
//          The default joystick control that this control is mapped to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CtrlMap::CtrlMap(
    const KString&        iName,
    DWord                 dwWhen,
    GlobalControl         eGlobal,
    Input*                pInput,
    Byte                  bMask,
    Keyboard::KeyboardKey eDefaultKey  /* = Keyboard::KEY__NONE */,
    Joystick::JoystickJoy eDefaultJoy  /* = Joystick::JOY_NONE */
)
:
    RepBase                       ( iName ),
    m_dwWhen                      ( dwWhen ),
    m_eGlobal                     ( eGlobal ),
    m_pbLocation                  ( pInput->getLocation( ) ),
    m_bMask                       ( bMask )
{
    //  Call a common construction member function.
    CtrlMapCtor( eGlobal, eDefaultKey, eDefaultJoy );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: CtrlMap
//
//  Description:
//
//      This is a private constructor used by this class to create the
//      global control maps.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWhen (input)
//          This indicates when the key is active.  This can be used in
//          network games to determine whether or not a key is controlled
//          by a particular player.
//
//      eGlobal (input)
//          The global control this is related to.
//
//      eDefaultKey (input)
//          The default key that this control is mapped to.
//
//      eDefaultJoy (input)
//          The default joystick control that this control is mapped to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CtrlMap::CtrlMap(
    const KString&        iName,
    DWord                 dwWhen,
    GlobalControl         eGlobal,
    Keyboard::KeyboardKey eDefaultKey,
    Joystick::JoystickJoy eDefaultJoy
)
:
    RepBase                       ( iName ),
    m_dwWhen                      ( dwWhen ),
    m_eGlobal                     ( eGlobal ),
    m_pbLocation                  ( NULL ),
    m_bMask                       ( 0x00 )
{
    //  Call a common construction member function.
    CtrlMapCtor( eGlobal, eDefaultKey, eDefaultJoy );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: CtrlMapCtor
//
//  Description:
//
//      This function contains the code that is common between constructors.
//
//  Parameters:
//
//      eGlobal (input)
//          The global control this is related to.
//
//      eDefaultKey (input)
//          The default key that this control is mapped to.
//
//      eDefaultJoy (input)
//          The default joystick control that this control is mapped to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
CtrlMap::CtrlMapCtor(
    GlobalControl         eGlobal,
    Keyboard::KeyboardKey eDefaultKey,
    Joystick::JoystickJoy eDefaultJoy
)
{
    //  Make sure the global controls are initialized.
    if( !sm_bGlobalControlsInitialized )
    {
        sm_bGlobalControlsInitialized = TRUE;
        s_initializeGlobalControls( );
    }

    //  Assign the sources, defaults.
    m_eSource[ KEYBOARD ]         = FROM_GAME;
    m_dwSwitch[ KEYBOARD ]        = eDefaultKey;
    m_dwDefaultSwitch[ KEYBOARD ] = eDefaultKey;
    m_eSource[ JOYSTICK ]         = FROM_GAME;
    m_dwSwitch[ JOYSTICK ]        = eDefaultJoy;
    m_dwDefaultSwitch[ JOYSTICK ] = eDefaultJoy;

    //  If the default key or joystick is not specified and a matching global
    //  control is specified then use the default from it.
    for( DWord dwI = 0 ; dwI < ASSOC_COUNT ; dwI += 1 )
    {
        if( 
            ( m_dwSwitch[ dwI ] == DigitalController::SWITCH_NONE ) && 
            ( eGlobal != NO_GLOBAL ) &&
            ( ( unsigned int )eGlobal < sm_globalControlList.entries( ) )
        )
        {
            m_dwSwitch[ dwI ] = 
                sm_globalControlList[ eGlobal ]->m_dwSwitch[ dwI ];
            m_dwDefaultSwitch[ dwI ] = m_dwSwitch[ dwI ];
        }
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~CtrlMap
//
//  Description:
//
//      This is the destructor for a control map object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
CtrlMap::~CtrlMap(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the control map object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
CtrlMap::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "CtrlMap" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: isActive
//
//  Description:
//
//      This returns whether or not the current control map is active
//      for the state passed in.
//
//      This is for future modifications to network support that will 
//      disallow controls from the wrong nodes (e.g. Player 2 won't be
//      allowed to press Player 1's keys).  For now it's the honour system.
//
//  Parameters:
//
//      dwPlayerNum (input)
//          The player requesting the activation status.
//
//      dwWhosTurn (input)
//          The player whos turn it currently is.
//
//      bUpright (input)
//          Indicates whether the machine is an upright or cocktail.
//
//  Returns:
//
//      TRUE  if the control map is active for the specified state.
//      FALSE otherwise.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CtrlMap::isActive(
    const DWord dwPlayerNum,
    const DWord dwWhosTurn,
    const Byte  bUpright
)
{
    //  If the control is assigned solely to the player then it is active.
    if( m_dwWhen & ( PLAYER << ( dwPlayerNum - 1 ) ) )
    {
        return( TRUE );
    }
    //  Otherwise it must be the player's turn for the control to be active.
    else
    if( dwPlayerNum == dwWhosTurn )
    {
        //  Check for matching upright or cocktail specification.
        if( 
            ( bUpright && ( m_dwWhen & ( UPRIGHT  << ( dwPlayerNum - 1 ) ) ) ) 
            ||
            (             ( m_dwWhen & ( COCKTAIL << ( dwPlayerNum - 1 ) ) ) )
        )
        {
            return( TRUE );
        }
    }

    //  The control is not active.
    return( FALSE );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_initializeGlobalControls
//
//  Description:
//
//      This static member is called to initialize the global controls.  This
//      involves retrieving any previously saved values.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if the global controls were initialized successfully.
//      FALSE if the global controls could not be initialized.
//
///////////////////////////////////////////////////////////////////////////////
Byte
CtrlMap::s_initializeGlobalControls(
)
{
    //  The registry containing the settings for Replay.
    Registry* pSettings = Configuration::s_instance( ).getSettingsRegistry( );

    //  A temporary control.
    int32 nSwitch;

    //  Add the global controls to the list.
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Up",
            PLAYER1,
            P1_UP,
            Keyboard::KEY__UP,
            Joystick::JOY1_UP
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Down",
            PLAYER1,
            P1_DOWN,
            Keyboard::KEY__DOWN,
            Joystick::JOY1_DOWN
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Left",
            PLAYER1,
            P1_LEFT,
            Keyboard::KEY__LEFT,
            Joystick::JOY1_LEFT
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Right",
            PLAYER1,
            P1_RIGHT,
            Keyboard::KEY__RIGHT,
            Joystick::JOY1_RIGHT
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Action 1",
            PLAYER1,
            P1_ACTION1,
            Keyboard::KEY__RALT,
            Joystick::JOY1_B1
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Action 2",
            PLAYER1,
            P1_ACTION2,
            Keyboard::KEY__RCTRL,
            Joystick::JOY1_B2
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Action 3",
            PLAYER1,
            P2_ACTION3,
            Keyboard::KEY__SLASH,
            Joystick::JOY1_B3
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Action 4",
            PLAYER1,
            P2_ACTION4,
            Keyboard::KEY__RSHIFT,
            Joystick::JOY1_B4
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Up",
            PLAYER2,
            P2_UP,
            Keyboard::KEY__T,
            Joystick::JOY2_UP
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Down",
            PLAYER2,
            P2_DOWN,
            Keyboard::KEY__G,
            Joystick::JOY2_DOWN
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Left",
            PLAYER2,
            P2_LEFT,
            Keyboard::KEY__F,
            Joystick::JOY2_LEFT
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Right",
            PLAYER2,
            P2_RIGHT,
            Keyboard::KEY__H,
            Joystick::JOY2_RIGHT
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Action 1",
            PLAYER2,
            P2_ACTION1,
            Keyboard::KEY__LCTRL,
            Joystick::JOY2_B1
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Action 2",
            PLAYER2,
            P2_ACTION2,
            Keyboard::KEY__LALT,
            Joystick::JOY2_B2
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Action 3",
            PLAYER2,
            P2_ACTION3,
            Keyboard::KEY__LSHIFT,
            Joystick::JOY2_B3
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Action 4",
            PLAYER2,
            P2_ACTION4,
            Keyboard::KEY__Z,
            Joystick::JOY2_B4
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 1 Start",
            PLAYER1,
            P1_START,
            Keyboard::KEY__1,
            Joystick::JOY_NONE
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Player 2 Start",
            PLAYER1,
            P2_START,
            Keyboard::KEY__2,
            Joystick::JOY_NONE
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Coin 1 Insert",
            PLAYER1,
            COIN1,
            Keyboard::KEY__3,
            Joystick::JOY_NONE
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Coin 2 Insert",
            PLAYER1,
            COIN2,
            Keyboard::KEY__4,
            Joystick::JOY_NONE
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Coin 3 Insert",
            PLAYER1,
            COIN3,
            Keyboard::KEY__5,
            Joystick::JOY_NONE
        )
    );
    sm_globalControlList.add(
        new CtrlMap(
            "Coin 4 Insert",
            PLAYER1,
            COIN4,
            Keyboard::KEY__6,
            Joystick::JOY_NONE
        )
    );

    //  Check to make sure we have descriptions for each control.
    ASSERT( sm_globalControlList.entries( ) == GLOBAL_CONTROL_COUNT );

    //  Load any values from the settings registry.
    for( DWord dwI = 0 ; dwI < GLOBAL_CONTROL_COUNT ; dwI += 1 )
    {
        //  For each control type.
        for( DWord dwJ = 0 ; dwJ < ASSOC_COUNT ; dwJ += 1 )
        {
            //  Set the source to global.
            sm_globalControlList[ dwI ]->setSource( 
                ( Association )dwJ, FROM_GLOBAL 
            );

            //  If a registry entry exists for the current key then load it in.
            KString resourceName( 
                sm_globalControlList[ dwI ]->getInstanceName( ) 
            );
            resourceName += ( dwJ == KEYBOARD ? ".key" : ".joy" );
            if( pSettings->getValueSize( "replay", resourceName ) )
            {
                //  First extract the value from the registry into a temporary
                //  buffer.
                pSettings->getValue(
                    "replay", resourceName, ( Byte* )&nSwitch, sizeof( nSwitch )
                );

                //  Now set the value of the control map to the retrieved 
                //  switch and indicate that the mapping has been specified by 
                //  a user.
                sm_globalControlList[ dwI ]->setSource( 
                    ( Association )dwJ, FROM_USER 
                );
                sm_globalControlList[ dwI ]->set( 
                    ( Association )dwJ, ( DWord )nSwitch 
                );
            }
        }
    }

    //  Mark the global controls as initialized.
    sm_bGlobalControlsInitialized = TRUE;

    return( TRUE );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_getGlobalControl
//
//  Description:
//
//      This static member can be used to retrieve a specified global control.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      A pointer to the global control object.
//
///////////////////////////////////////////////////////////////////////////////
CtrlMap*
CtrlMap::s_getGlobalControl( 
    GlobalControl eControl 
)
{
    //  Check the parameters.
    ASSERT( eControl < GLOBAL_CONTROL_COUNT );
    ASSERT( ( unsigned int )eControl < sm_globalControlList.entries( ) );

    //  return the requested control.
    return( sm_globalControlList[ eControl ] );
}
