///////////////////////////////////////////////////////////////////////////////
//
//  File:       ctrlmap.h
//
//  Class:      CtrlMap - Concrete
//  
//  Hierarchy:  CtrlMap - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class is used to map a control in the game (e.g. Player 1 Up,
//      Player 2 Fire, Coin 1 Inserted, etc.) to a user's control (e.g.
//      The 'A' key, Joystick Up, etc.).
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _ctrlmap_h_
#define _ctrlmap_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "keyb.h"
#include "joy.h"

//  Forward Class Declarations.
class CtrlMap;
class Input;


///////////////////////////////////////////////////////////////////////////////
//  CtrlMap Class Definition
///////////////////////////////////////////////////////////////////////////////
class CtrlMap : public RepBase
{
    public:

        //  The global controls.  These are common controls that appear
        //  in many games and will have default mappings throughout Replay.
        typedef enum
        {
            P1_UP,      P1_DOWN,    P1_LEFT,    P1_RIGHT,
            P1_ACTION1, P1_ACTION2, P1_ACTION3, P1_ACTION4,
            P2_UP,      P2_DOWN,    P2_LEFT,    P2_RIGHT,
            P2_ACTION1, P2_ACTION2, P2_ACTION3, P2_ACTION4,
            P1_START,   P2_START,   
            COIN1,      COIN2,      COIN3,      COIN4,

            GLOBAL_CONTROL_COUNT,

            NO_GLOBAL
        }
        GlobalControl;
      
        //  The various associations that the control mapping has with 
        //  physical controls.
        typedef enum
        {
            KEYBOARD,
            JOYSTICK,

            ASSOC_COUNT
        }
        Association;

        //  The various sources of the mapping.
        typedef enum
        {
            FROM_USER,
            FROM_GLOBAL,
            FROM_GAME
        }
        MappedFromWhere;
 
        //  The following are used to indicate when the control is valid.
        typedef enum
        {
            PLAYER    = 0x000001,
            PLAYER1   = 0x000001,
            PLAYER2   = 0x000002,
            PLAYER3   = 0x000004,
            PLAYER4   = 0x000008,
            PLAYER5   = 0x000010,
            PLAYER6   = 0x000020,
            PLAYER7   = 0x000040,
            PLAYER8   = 0x000080,

            UPRIGHT   = 0x000100,
            UPRIGHT1  = 0x000100,
            UPRIGHT2  = 0x000200,
            UPRIGHT3  = 0x000400,
            UPRIGHT4  = 0x000800,
            UPRIGHT5  = 0x001000,
            UPRIGHT6  = 0x002000,
            UPRIGHT7  = 0x004000,
            UPRIGHT8  = 0x008000,

            COCKTAIL  = 0x010000,
            COCKTAIL1 = 0x010000,
            COCKTAIL2 = 0x020000,
            COCKTAIL3 = 0x040000,
            COCKTAIL4 = 0x080000,
            COCKTAIL5 = 0x100000,
            COCKTAIL6 = 0x200000,
            COCKTAIL7 = 0x400000,
            COCKTAIL8 = 0x800000
        }
        When;
      
        //  Creation/Deletion.
        CtrlMap ( 
            const KString&        iName, 
            DWord                 dwWhen,
            GlobalControl         eGlobal,
            Byte*                 pbLocation,
            Byte                  bMask,
            Keyboard::KeyboardKey eDefaultKey = Keyboard::KEY__NONE,
            Joystick::JoystickJoy eDefaultJoy = Joystick::JOY_NONE
        );
        CtrlMap ( 
            const KString&        iName, 
            DWord                 dwWhen,
            GlobalControl         eGlobal,
            Input*                pInput,
            Byte                  bMask,
            Keyboard::KeyboardKey eDefaultKey = Keyboard::KEY__NONE,
            Joystick::JoystickJoy eDefaultJoy = Joystick::JOY_NONE
        );
        virtual ~CtrlMap ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Interface.
        Byte*           getLocation ( ) const;
        Byte            getMask     ( ) const;
        GlobalControl   getGlobal   ( ) const;

        MappedFromWhere getSource   ( Association eAssoc ) const;
        void            setSource   ( 
            Association eAssoc, MappedFromWhere eSource 
        );
        DWord           getDefault  ( Association eAssoc ) const;
        DWord           get         ( Association eAssoc ) const;
        void            set         ( Association eAssoc, DWord dwSwitch );

        Byte            isActive    ( 
            const DWord dwPlayerNum, 
            const DWord dwWhosTurn, 
            const Byte  bUpright 
        );

        //  Static Interface.
        static CtrlMap*        s_getGlobalControl ( GlobalControl eControl );

    protected:

        //  Creation/Deletion.
        CtrlMap ( 
            const KString&        iName, 
            DWord                 dwWhen,
            GlobalControl         eGlobal,
            Keyboard::KeyboardKey eDefaultKey,
            Joystick::JoystickJoy eDefaultJoy
        );
        void
        CtrlMapCtor ( 
            GlobalControl         eGlobal,
            Keyboard::KeyboardKey eDefaultKey,
            Joystick::JoystickJoy eDefaultJoy
        );

        //  Static Local Interface.
        static Byte            s_initializeGlobalControls ( );

        //  Member Data.
        DWord                  m_dwWhen;
        GlobalControl          m_eGlobal;
        Byte*                  m_pbLocation;
        Byte                   m_bMask;
        MappedFromWhere        m_eSource[ ASSOC_COUNT ];
        DWord                  m_dwSwitch[ ASSOC_COUNT ];
        DWord                  m_dwDefaultSwitch[ ASSOC_COUNT ];

        //  Static Member Data.
        static KPtrList<CtrlMap> sm_globalControlList;
        static Byte              sm_bGlobalControlsInitialized;

    private:

};


///////////////////////////////////////////////////////////////////////////////
//  Input Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Byte*
CtrlMap::getLocation(
) const
{
    return( m_pbLocation );
}

inline
Byte
CtrlMap::getMask(
) const
{
    return( m_bMask );
}

inline
CtrlMap::MappedFromWhere
CtrlMap::getSource(
    CtrlMap::Association eAssoc
) const
{
    ASSERT( eAssoc < ASSOC_COUNT );

    return( m_eSource[ eAssoc ] );
}

inline
void
CtrlMap::setSource(
    CtrlMap::Association     eAssoc,
    CtrlMap::MappedFromWhere eSource
)
{
    ASSERT( eAssoc < ASSOC_COUNT );

    m_eSource[ eAssoc ] = eSource;
}

inline
DWord
CtrlMap::getDefault(
    CtrlMap::Association eAssoc
) const
{
    ASSERT( eAssoc < ASSOC_COUNT );

    return( m_dwDefaultSwitch[ eAssoc ] );
}

inline
DWord
CtrlMap::get(
    CtrlMap::Association eAssoc
) const
{
    ASSERT( eAssoc < ASSOC_COUNT );

    return( m_dwSwitch[ eAssoc ] );
}

inline
void
CtrlMap::set(
    CtrlMap::Association eAssoc,
    DWord                dwSwitch
)
{
    ASSERT( 
        ( ( eAssoc == KEYBOARD ) && ( dwSwitch < Keyboard::KEY__COUNT ) ) ||
        ( ( eAssoc == JOYSTICK ) && ( dwSwitch < Joystick::JOY_COUNT ) ) 
    );
    
    m_dwSwitch[ eAssoc ] = dwSwitch;
}

inline
CtrlMap::GlobalControl
CtrlMap::getGlobal(
) const
{
    return( m_eGlobal );
}

#endif
