///////////////////////////////////////////////////////////////////////////////
//
//  File:    game.cpp
//
//  Class:   Game
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class serves as a common base for each game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////
//  Application Headers.
#include "game.h"
#include "gameinfo.h"
#include "repfact.h"
#include "registry.h"
#include "space.h"
#include "screen.h"
#include "buffer.h"
#include "hiscore.h"
#include "bitmap.h"
#include "input.h"
#include "ctrlmap.h"
#include "dip.h"
#include "gfxset.h"
#include "ctable.h"
#include "clip.h"
#include "sample.h"
#include "sdev.h"
#include "schedule.h"
#include "disk.h"
#include "appfile.h"
#ifdef DEBUGGER
#include "debugger.h"
#endif


///////////////////////////////////////////////////////////////////////////////
//  Static Data.
///////////////////////////////////////////////////////////////////////////////

//  The following static member is used for quick access to the current game
//  object in the static functions used for handlers and what have you.
Game* Game::sm_pGame = NULL;


///////////////////////////////////////////////////////////////////////////////
//
//  Function: Game
//
//  Description:
//
//      This is the main constructor for a game object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Game::Game(
    const KString& iName
)
:
    RepBase             ( iName ),
    m_pReplay           ( NULL ),
    m_nCPUInterleave    ( 1 ),
    m_bScreenFlippedX   ( FALSE ),
    m_bScreenFlippedY   ( FALSE ),
    m_dwMaxPlayers      ( 2 ),
    m_pSettingsRegistry ( NULL ),
    m_pHiScoreRegistry  ( NULL ),
    m_bitmapList        ( 5 ),
    m_inputList         ( 10 ),
    m_ctrlMapList       ( 20 ),
    m_dipSwitchList     ( 10 ),
    m_spaceList         ( 3 ),
    m_cpuList           ( 3 ),
    m_gfxSetList        ( 5 ),
    m_bufferList        ( 3 ),
    m_tempBufferList    ( 3 ),
    m_dirtyList         ( 3 ),
    m_colourTableList   ( 3 ),
    m_clippingList      ( 3 ),
    m_sampleList        ( 5 ),
    m_soundDeviceList   ( 2 ),
    m_pScheduler        ( 0 ),
    m_pScreen           ( NULL ),
    m_pColourPROM       ( NULL ),
    m_pSoundPROM        ( NULL ),
    m_pHiScore          ( NULL ),
    m_bSoundDisabled    ( Configuration::s_instance( ).getParam( "-nosound" ) ),
    m_dwStartStep       ( 0 ),
    m_nWatchDog         ( -1 )
{
    //  For convenience, we hold on to some commonly used objects.
    m_pReplay = &( Replay::s_instance( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Game
//
//  Description:
//
//      This is the destructor for the Game base object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Game::~Game(
)
{
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUp
//
//  Description:
//
//      This member is called when the game is to be initialized.  Most 
//      initialization should be done here, not in the constructor since
//      all games objects are created up front which would eat up a lot
//      of memory if every game initialized itself at that point.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//
//      TRUE  to indicate that there is more to do and that the function
//            should be called again.
//      FALSE to indicate that the startup is complete.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Game::startUp(
)
{
    //  Perform the current startup task.
    switch( m_dwStartStep )
    {
        case 0:  startUpStart( );          break;
        case 1:  startUpSpace( );          break;
        case 2:  startUpROMs( );           break;
        case 3:  startUpCPU( );            break;
        case 4:  startUpInput( );          break;
        case 5:  startUpGraphics( );       break;
        case 6:  startUpColour( );         break;
        case 7:  startUpSound( );          break;
        case 8:  startUpMemMap( );         break;
        case 9:  startUpOther( );          break;
        case 10: startUpEnd( );            break;

        default: return( FALSE );
    }

    //  We haven't yet completed the startup so move to the next step.
    m_dwStartStep += 1;

    //  Indicate that more is necessary.
    return( TRUE );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXX
//
//  Description:
//
//      The following members are used to start up various subsystems for
//      the current game.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::startUpStart(
)
{
    //  We must create the configuration and high score registries for
    //  the game.
    ASSERT( m_pGameInfo != NULL );
    m_pSettingsRegistry = 
        new Registry( m_pGameInfo->getGameId( ), Registry::CONFIG );
    m_pHiScoreRegistry = 
        new Registry( m_pGameInfo->getGameId( ), Registry::SCORES );

    //  A screen object must be created up front.
    m_pScreen = new Screen( m_pGameInfo->getGameId( ) );
}

void
Game::startUpSpace(
)
{
    //  No default action.
}

void
Game::startUpROMs(
)
{
    //  No default action.
}

void
Game::startUpCPU(
)
{
    //  No default action.
}

void
Game::startUpInput(
)
{
    //  No default action.
}

void
Game::startUpGraphics(
)
{
    //  No default action.
}

void
Game::startUpColour(
)
{
    //  No default action.
}

void
Game::startUpSound(
)
{
    //  No default action.
}

void
Game::startUpMemMap(
)
{
    //  No default action.
}

void
Game::startUpOther(
)
{
    //  No default action.
}

void
Game::startUpEnd(
)
{
    //  Reset the game.
    reset( );

    //  Set up the controls and the dip switches.
    setControls( );
    setDipSwitches( );

    //  We don't want any CPU interleave if there is only one CPU and
    //  we don't want an interleave of less than 1.
    if( ( m_cpuList.entries( ) == 1 ) || ( m_nCPUInterleave < 1 ) )
    {
        m_nCPUInterleave = 1;
    }

    //  Schedule the CPUs for execution.
    m_pScheduler = new Scheduler( "Scheduler" );
    m_pScheduler->scheduleCPUs( m_cpuList, m_nCPUInterleave, m_pScreen );

    //  Get rid of the temporary address spaces.
    for( DWord dwI = 0 ; dwI < m_tempBufferList.entries( ) ; dwI += 1 )
    {
        delete m_tempBufferList[ dwI ];
    }
    m_tempBufferList.clear( );

#ifdef DEBUGGER
    //  Notify the debugger of it.
    Debugger::s_instance( ).clear( this );
#endif
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: shutDown
//
//  Description:
//
//      This member is called when the game is to be cleaned up.  Any 
//      initialization done in startUp() should be cleaned up here, not
//      in the destructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::shutDown(
)
{
    //  Indices used for iterations.
    DWord dwI;


    //  Free up the Screen.
    delete m_pScreen;
    m_pScreen = NULL;

    //  Free up the Colour PROM.
    delete m_pColourPROM;
    m_pColourPROM = NULL;

    //  Free up the Sound PROM.
    delete m_pSoundPROM;
    m_pSoundPROM = NULL;

    //  Free up the HiScore (but save them first!).
    if( m_pHiScore != NULL )
    {
        m_pHiScore->saveScores( );
        delete m_pHiScore;
        m_pHiScore = NULL;
    }

    //  Clean up the Bitmaps.
    for( dwI = 0 ; dwI < m_bitmapList.entries( ) ; dwI += 1 )
    {
        delete m_bitmapList[ dwI ];
    }
    m_bitmapList.clear( );

    //  Clean up the Inputs.
    for( dwI = 0 ; dwI < m_inputList.entries( ) ; dwI += 1 )
    {
        delete m_inputList[ dwI ];
    }
    m_inputList.clear( );

    //  Clean up the Control Maps.
    for( dwI = 0 ; dwI < m_ctrlMapList.entries( ) ; dwI += 1 )
    {
        delete m_ctrlMapList[ dwI ];
    }
    m_ctrlMapList.clear( );

    //  Clean up the Dip Switches.
    for( dwI = 0 ; dwI < m_dipSwitchList.entries( ) ; dwI += 1 )
    {
        delete m_dipSwitchList[ dwI ];
    }
    m_dipSwitchList.clear( );

    //  Clean up the address spaces.
    for( dwI = 0 ; dwI < m_spaceList.entries( ) ; dwI += 1 )
    {
        delete m_spaceList[ dwI ];
    }
    m_spaceList.clear( );

    //  Clean up the CPUs.
    for( dwI = 0 ; dwI < m_cpuList.entries( ) ; dwI += 1 )
    {
        delete m_cpuList[ dwI ];
    }
    m_cpuList.clear( );

    //  Clean up the graphic sets.
    for( dwI = 0 ; dwI < m_gfxSetList.entries( ) ; dwI += 1 )
    {
        delete m_gfxSetList[ dwI ];
    }
    m_gfxSetList.clear( );

    //  Clean up the buffers.
    for( dwI = 0 ; dwI < m_bufferList.entries( ) ; dwI += 1 )
    {
        delete m_bufferList[ dwI ];
    }
    m_bufferList.clear( );

    //  Clean up the dirty buffers.
    for( dwI = 0 ; dwI < m_dirtyList.entries( ) ; dwI += 1 )
    {
        delete m_dirtyList[ dwI ];
    }
    m_dirtyList.clear( );

    //  Clean up the colour tables.
    for( dwI = 0 ; dwI < m_colourTableList.entries( ) ; dwI += 1 )
    {
        delete m_colourTableList[ dwI ];
    }
    m_colourTableList.clear( );

    //  Clean up the clipping regions.
    for( dwI = 0 ; dwI < m_clippingList.entries( ) ; dwI += 1 )
    {
        delete m_clippingList[ dwI ];
    }
    m_clippingList.clear( );

    //  Clean up the samples.
    for( dwI = 0 ; dwI < m_sampleList.entries( ) ; dwI += 1 )
    {
        delete m_sampleList[ dwI ];
    }
    m_sampleList.clear( );

    //  Clean up the sound devices.
    for( dwI = 0 ; dwI < m_soundDeviceList.entries( ) ; dwI += 1 )
    {
        delete m_soundDeviceList[ dwI ];
    }
    m_soundDeviceList.clear( );

    //  Dispose of the scheduler.
    delete m_pScheduler;
    m_pScheduler = NULL;

    //  Dispose of the registries.
    delete m_pSettingsRegistry;
    delete m_pHiScoreRegistry;
    m_pSettingsRegistry = NULL;
    m_pHiScoreRegistry  = NULL;

    //  Reset the start step to 0.
    m_dwStartStep = 0;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: reset
//
//  Description:
//
//      This member is called to reset the game.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::reset(
)
{
    //  A shortcut to the canvas.
    Canvas* pCanvas = m_pReplay->getCanvas( );

    //  Save the high scores.
    if( m_pHiScore != NULL )
    {
        m_pHiScore->saveScores( );
    }

    //  Reset all CPUs for the game.
    for( DWord dwI = 0 ; dwI < m_cpuList.entries( ) ; dwI += 1 )
    {
        //  Reset the CPU.
        m_cpuList[ dwI ]->reset( );
    }

    //  Clear the screen bitmaps.
    m_pScreen->clear( );

    //  Attempt to load the high scores if the scores are to be loaded 
    //  immediately (i.e. trigger count is 0).
    if( ( m_pHiScore != NULL ) && ( m_pHiScore->getTrigger( ) == 0 ) )
    {
        m_pHiScore->loadScores( );
    }

    //  If the screen has been flipped around, then flip it back.
    if( m_bScreenFlippedX )
    {
        pCanvas->setOrientation( 
            pCanvas->getTranspose( ), 
            !pCanvas->getFlipX( ), 
            pCanvas->getFlipY( ) 
        );
        m_bScreenFlippedX = FALSE;
    }
    if( m_bScreenFlippedY )
    {
        pCanvas->setOrientation( 
            pCanvas->getTranspose( ), 
            pCanvas->getFlipX( ), 
            !pCanvas->getFlipY( ) 
        );
        m_bScreenFlippedY = FALSE;
    }
    

    //  Reset the watchdog.
    m_nWatchDog = -1;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: run
//
//  Description:
//
//      This member is called to run one frame's worth of the game.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::run(
)
{
    //  Squirrel this away for use in any static functions that are called.
    sm_pGame = this;

    //  Update the state of the CPUs.
    m_pScheduler->updateCPUs( );

    //  Update the state of the display.
    ASSERT( m_pScreen != NULL );
    updateDisplay( m_pScreen->getBitmap( ), m_pScreen->getTempBitmap( ) );

    //  Update the state of the sound if it's enabled.
    if( !m_bSoundDisabled )
    {
        updateSound( );
    }

    //  If the watchdog counter is in use then check it.
    if( m_nWatchDog >= 0 )
    {
        m_nWatchDog -= 1;
        if( m_nWatchDog < 0 )
        {
            CHECK0( FALSE, "Watchdog has reset the emulation." );
            reset( );
        }
    } 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateDisplay
//
//  Description:
//
//      This member is called when a game is to update its Display.
//
//  Parameters:
//
//      pScreen (input/output)
//          A pointer to the screen bitmap.
//
//      pTempScreen (input/output)
//          A pointer to the double buffer screen bitmap.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::updateDisplay(
    Bitmap* /* pScreen is unused     */,
    Bitmap* /* pTempScreen is unused */
)
{
    //  No default behaviour.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: updateSound
//
//  Description:
//
//      This member is called when a game is to update it's Sound.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::updateSound(
)
{
    //  Update each sound device.
    for( DWord dwI = 0 ; dwI < m_soundDeviceList.entries( ) ; dwI += 1 )
    {
        m_soundDeviceList[ dwI ]->update( );
    } 
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: createCPU
//
//  Description:
//
//      This member is called to create a CPU for the game.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the CPU.
//
//      eCPUType (input)
//          The type of processor.
//
//      pSpace (input)
//          The address space the CPU is to operate within.
//
//      ePurpose (input)
//          The CPU's purpose .
//
//      bCONSERVATIVE (input)
//          A TEMPORARY FLAG USED TO ALLOW US TO PICK THE 'C' BASED Z80
//          CORE DUE TO THE INSTABILITY OF THE ASM.
//
//  Returns:
//
//      A pointer to the newly created CPU.
//
///////////////////////////////////////////////////////////////////////////////
CPU* 
Game::createCPU( 
    const KString&        iName, 
    const CPU::CPUType    eCPUType, 
    AddrSpace*            pSpace,
    const CPU::CPUPurpose ePurpose      /* = CPU::GAME */,
    const Byte            bCONSERVATIVE /* = FALSE */
)
{
    //  A pointer to the new CPU.
    CPU* pCPU;

    //  Create the CPU.
    pCPU = ReplayFactory::s_instance( ).createCPU( 
        iName, eCPUType, this, pSpace, ePurpose, bCONSERVATIVE
    );

    //  Set up the default interrupt handler (this can be overridden).
    pCPU->interruptHandler( s_irqInterrupt );

    return( pCPU );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setControls
//
//  Description:
//
//      This member is called to set the controls for the game.  The controls
//      should have been added to list of control mappings.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::setControls( 
)
{
    //  The keyboard and joystick objects.
    DigitalController* pController[ CtrlMap::ASSOC_COUNT ];
    pController[ CtrlMap::KEYBOARD ] = m_pReplay->getKeyboard( );
    pController[ CtrlMap::JOYSTICK ] = m_pReplay->getJoystick( );

    //  A buffer used to retrieve registry values.
    int32 nValue;


    //  Make sure the keyboard and joystick do not currently have any 
    //  mappings specified.
    pController[ CtrlMap::KEYBOARD ]->clearMappings( );
    pController[ CtrlMap::JOYSTICK ]->clearMappings( );

    //  Set each control found in the list of control mappings.
    for( DWord dwI = 0 ; dwI < m_ctrlMapList.entries( ) ; dwI += 1 )
    {
       //  Loop through each physical control.
       for( DWord dwJ = 0 ; dwJ < CtrlMap::ASSOC_COUNT ; dwJ += 1 )
       {
           //  There are three possible sources for the control.  If
           //  the control has been specified by the user then this control
           //  will take precedence.  If not and the control matches a global
           //  control then the global control will be used.  The final option
           //  is to just use the control specified by the game.
           //  So we modify the control mapping accordingly.
           KString resourceName( m_ctrlMapList[ dwI ]->getInstanceName( ) );
           resourceName += ( dwJ == CtrlMap::KEYBOARD ? ".key" : ".joy" );
           if( settingsGetValue( resourceName, &nValue ) )
           {
               //  The user has overridden the control so set it up 
               //  appropriately.
               m_ctrlMapList[ dwI ]->setSource( 
                   ( CtrlMap::Association )dwJ, CtrlMap::FROM_USER 
               );
               m_ctrlMapList[ dwI ]->set( 
                   ( CtrlMap::Association )dwJ, ( DWord )nValue
               );
           }
           else
           if( m_ctrlMapList[ dwI ]->getGlobal( ) != CtrlMap::NO_GLOBAL )
           {
               //  Override the control with the global value.
               CtrlMap* pGlobal = CtrlMap::s_getGlobalControl( 
                   m_ctrlMapList[ dwI ]->getGlobal( )
               );
               m_ctrlMapList[ dwI ]->setSource( 
                   ( CtrlMap::Association )dwJ, CtrlMap::FROM_GLOBAL 
               );
               m_ctrlMapList[ dwI ]->set( 
                   ( CtrlMap::Association )dwJ, 
                   pGlobal->get( ( CtrlMap::Association )dwJ ) 
               );
           }

           //  Set the control with the controller.  The controller will take 
           //  care of rejecting controls when the control is NONE.
           pController[ dwJ ]->addMapping( 
               m_ctrlMapList[ dwI ]->get( ( CtrlMap::Association )dwJ ),
               m_ctrlMapList[ dwI ]->getLocation( ),
               m_ctrlMapList[ dwI ]->getMask( )
           );
        }
    }
}


    
///////////////////////////////////////////////////////////////////////////////
//
//  Function: setDipSwitches
//
//  Description:
//
//      This member is called to initialize the dip switches for the game.  
//      The dip switches should have been added to the list of dip switches.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::setDipSwitches( 
)
{
    //  A dip switch value.
    Byte bValue;

    //  Set each switch found in the list of dip switches.
    for( DWord dwI = 0 ; dwI < m_dipSwitchList.entries( ) ; dwI += 1 )
    {
       //  If there is a saved setting for the dip switch then use it.
       if( 
           settingsGetValue( 
               m_dipSwitchList[ dwI ]->getInstanceName( ), &bValue
           )
       )
       {
           //  We've found a saved setting so let the dip know.
           m_dipSwitchList[ dwI ]->setValue( bValue );
       }
    }
}


    
///////////////////////////////////////////////////////////////////////////////
//
//  Function: resetControls
//
//  Description:
//
//      This member is called to reset the controls of the game based
//      on the current settings of the control mappings.  This is to be
//      called by external clients who may have changed the joystick or
//      keyboard mappings and want the changes reflected.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::resetControls( 
)
{
    //  Get the keyboard and joystick objects.
    Keyboard* pKeyboard = m_pReplay->getKeyboard( );
    Joystick* pJoystick = m_pReplay->getJoystick( );

    //  Make sure the keyboard and joystick do not currently have any mappings 
    //  specified.
    pKeyboard->clearMappings( );
    pJoystick->clearMappings( );

    //  Set each control found in the list of control mappings.
    for( DWord dwI = 0 ; dwI < m_ctrlMapList.entries( ) ; dwI += 1 )
    {
       //  Set the control with the keyboard and joystick.  The keyboard and
       //  joystick will take care of rejecting controls when the value is
       //  NONE.
       pKeyboard->addMapping( 
           m_ctrlMapList[ dwI ]->get( CtrlMap::KEYBOARD ),
           m_ctrlMapList[ dwI ]->getLocation( ),
           m_ctrlMapList[ dwI ]->getMask( )
       );
       pJoystick->addMapping( 
           m_ctrlMapList[ dwI ]->get( CtrlMap::JOYSTICK ),
           m_ctrlMapList[ dwI ]->getLocation( ),
           m_ctrlMapList[ dwI ]->getMask( )
       );
    }
}


    
///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member is called to save the current state of the game in 
//      a new save game file.  This can be retrieved with the load member.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Game::save( 
)
{
    //  A pointer to the disk object.
    Disk* pDisk = m_pReplay->getDisk( );

    //  A pointer to the new save file.
    AppFile* pSaveFile;

    //  Indicates whether or not the save was successful or not.
    Byte bSuccess;


    //  Build the name of the file to save to.
    ASSERT( m_pGameInfo != NULL );
    KString fileName  = pDisk->getPath( );
    fileName         += pDisk->getDirectorySeparator( );
    fileName         += "save";
    fileName         += pDisk->getDirectorySeparator( );
    fileName         += m_pGameInfo->getGameId( );
    fileName         += ".sav";
      
    //  Ensure the directory components exist.
    pDisk->makeDirectoryComponents( fileName );

    //  Open the application file.
    pSaveFile = new AppFile( fileName );

    //  Perform the real save.
    if( pSaveFile->isOpen( ) )
    {
        bSuccess = save( pSaveFile );
    }
    else
    {
        bSuccess = TRUE;
    }

    //  Delete the save file.
    ASSERT( pSaveFile != NULL );
    delete pSaveFile;

    //  Return the sucess status.
    return( bSuccess );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member is called to save the current state of the game in 
//      a save game file.  This can be retrieved with the load member.
//
//  Parameters:
//
//      pSaveFile (input)
//          A pointer to the file to save to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Game::save( 
    AppFile* pSaveFile
)
{
    //  Indices used for iterations.
    DWord dwI;

    //  The number of errors encountered during the save.
    DWord dwNumErrors = 0;

    //  We always allow the base class to save itself first.
    dwNumErrors += RepBase::save( pSaveFile );

    //  Do the saving.
    for( dwI = 0 ; dwI < m_bitmapList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_bitmapList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_inputList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_inputList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_ctrlMapList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_ctrlMapList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_dipSwitchList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_dipSwitchList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_spaceList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_spaceList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_cpuList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_cpuList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_gfxSetList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_gfxSetList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_bufferList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_bufferList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_dirtyList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_dirtyList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_colourTableList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_colourTableList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_clippingList.entries( ) ; dwI += 1 )
    {
        pSaveFile->clearTotal( );
        pSaveFile->write( 
            ( Byte* )&( m_clippingList[ dwI ]->m_nMinX ), 
            sizeof( m_clippingList[ dwI ]->m_nMinX )
        );
        pSaveFile->write( 
            ( Byte* )&( m_clippingList[ dwI ]->m_nMaxX ), 
            sizeof( m_clippingList[ dwI ]->m_nMaxX )
        );
        pSaveFile->write( 
            ( Byte* )&( m_clippingList[ dwI ]->m_nMinY ), 
            sizeof( m_clippingList[ dwI ]->m_nMinY )
        );
        pSaveFile->write( 
            ( Byte* )&( m_clippingList[ dwI ]->m_nMaxY ), 
            sizeof( m_clippingList[ dwI ]->m_nMaxY )
        );
        if( 
            pSaveFile->total( ) != 
            ( 4 * sizeof( m_clippingList[ dwI ]->m_nMinX ) )
        )
        {
            dwNumErrors += 1;
        }
    }
    for( dwI = 0 ; dwI < m_sampleList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_sampleList[ dwI ]->save( pSaveFile );
    }
    for( dwI = 0 ; dwI < m_soundDeviceList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_soundDeviceList[ dwI ]->save( pSaveFile );
    }
    if( m_pScreen )
    {
        dwNumErrors += m_pScreen->save( pSaveFile );
    }
    if( m_pColourPROM )
    {
        dwNumErrors += m_pColourPROM->save( pSaveFile );
    }
    if( m_pSoundPROM )
    {
        dwNumErrors += m_pSoundPROM->save( pSaveFile );
    }
    if( m_pHiScore )
    {
        dwNumErrors += m_pHiScore->save( pSaveFile );
    }
    dwNumErrors += m_pReplay->getCanvas( )->save( pSaveFile );

    //  If there were errors encountered then indicate this to the client.
    return( dwNumErrors ? TRUE : FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member is called to load the current state of the game from
//      a save game file that was created in the save member.
//
//  Parameters:
//
//      pLoadFile (input)
//          A pointer to the file to load from.  
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Game::load( 
)
{
    //  A pointer to the disk object.
    Disk* pDisk = m_pReplay->getDisk( );

    //  A pointer to the load file.
    AppFile* pLoadFile;

    //  Indicates whether or not the save was successful or not.
    Byte bSuccess;


    //  Build the name of the file to save to.
    ASSERT( m_pGameInfo != NULL );
    KString fileName  = pDisk->getPath( );
    fileName         += pDisk->getDirectorySeparator( );
    fileName         += "save";
    fileName         += pDisk->getDirectorySeparator( );
    fileName         += m_pGameInfo->getGameId( );
    fileName         += ".sav";
  
    //  Open the application file.
    pLoadFile = new AppFile( fileName, FALSE );

    //  Perform the real load.
    if( pLoadFile->isOpen( ) )
    {
        bSuccess = load( pLoadFile );
    }
    else
    {
        bSuccess = TRUE;
    }


    //  Delete the load file.
    ASSERT( pLoadFile != NULL );
    delete pLoadFile;

    //  Return the sucess status.
    return( bSuccess );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member is called to load the current state of the game from
//      a save game file that was created in the save member.
//
//  Parameters:
//
//      pLoadFile (input)
//          A pointer to the file to load from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Game::load( 
    AppFile* pLoadFile
)
{
    //  Indices used for iterations.
    DWord dwI;

    //  The number of errors encountered during the load.
    DWord dwNumErrors = 0;


    //  Clear out the pointers to the current CPU.
    CPU::sm_pCPU   = NULL;
    CPU::sm_pSpace = NULL;
    
    //  We always allow the base class to load itself first.
    dwNumErrors += RepBase::load( pLoadFile );


    //  Do the loading.
    for( dwI = 0 ; dwI < m_bitmapList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_bitmapList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_inputList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_inputList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_ctrlMapList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_ctrlMapList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_dipSwitchList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_dipSwitchList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_spaceList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_spaceList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_cpuList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_cpuList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_gfxSetList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_gfxSetList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_bufferList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_bufferList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_dirtyList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_dirtyList[ dwI ]->load( pLoadFile );

        //  Mark everything as dirty so that it will be redrawn.
        m_dirtyList[ dwI ]->clear( TRUE ); 
    }
    for( dwI = 0 ; dwI < m_colourTableList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_colourTableList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_clippingList.entries( ) ; dwI += 1 )
    {
        pLoadFile->clearTotal( );
        pLoadFile->read( 
            ( Byte* )&( m_clippingList[ dwI ]->m_nMinX ), 
            sizeof( m_clippingList[ dwI ]->m_nMinX )
        );
        pLoadFile->read( 
            ( Byte* )&( m_clippingList[ dwI ]->m_nMaxX ), 
            sizeof( m_clippingList[ dwI ]->m_nMaxX )
        );
        pLoadFile->read( 
            ( Byte* )&( m_clippingList[ dwI ]->m_nMinY ), 
            sizeof( m_clippingList[ dwI ]->m_nMinY )
        );
        pLoadFile->read( 
            ( Byte* )&( m_clippingList[ dwI ]->m_nMaxY ), 
            sizeof( m_clippingList[ dwI ]->m_nMaxY )
        );
        if( 
            pLoadFile->total( ) != 
            ( 4 * sizeof( m_clippingList[ dwI ]->m_nMinX ) )
        )
        {
            dwNumErrors += 1;
        }
    }
    for( dwI = 0 ; dwI < m_sampleList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_sampleList[ dwI ]->load( pLoadFile );
    }
    for( dwI = 0 ; dwI < m_soundDeviceList.entries( ) ; dwI += 1 )
    {
        dwNumErrors += m_soundDeviceList[ dwI ]->load( pLoadFile );
    }
    if( m_pScreen )
    {
        dwNumErrors += m_pScreen->load( pLoadFile );
    }
    if( m_pColourPROM )
    {
        dwNumErrors += m_pColourPROM->load( pLoadFile );
    }
    if( m_pSoundPROM )
    {
        dwNumErrors += m_pSoundPROM->load( pLoadFile );
    }
    if( m_pHiScore )
    {
        dwNumErrors += m_pHiScore->load( pLoadFile );
    }
    dwNumErrors += m_pReplay->getCanvas( )->load( pLoadFile );

    //  If there were errors encountered then indicate this to the client.
    return( dwNumErrors ? TRUE : FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: settingsGetValueSize
//
//  Description:
//
//      This member is called to retrieve the size of the named value from
//      the game's settings registry.
//
//  Parameters:
//
//      name (input)
//          The name of the value.
//
//  Returns:
//
//      The size of the value if found.
//      0 if not found.
//
///////////////////////////////////////////////////////////////////////////////
const
DWord
Game::settingsGetValueSize(
    const KString& name
)
{
    ASSERT( m_pGameInfo != NULL );

    //  Get the size.
    return( 
        m_pSettingsRegistry->getValueSize( m_pGameInfo->getGameId( ), name )
    );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: settingsGetValue
//
//  Description:
//
//      This member is called to retrieve the specified byte stream value
//      from the settings registry.
//
//  Parameters:
//
//      name (input)
//          The name of the value.
//
//      pbResult (input)
//          The destination buffer.
//
//      dwMaxLength (input)
//          The maximum number of bytes allowed in the buffer.
//
//  Returns:
//
//      The number of bytes written to the buffer if found and valid.
//      0 if not found or invalid.
//
///////////////////////////////////////////////////////////////////////////////
const
DWord
Game::settingsGetValue(
    const KString& name,
    Byte*          pbResult,
    DWord          dwMaxLength /* = 1 */
)
{
    //  Preconditions.
    ASSERT( m_pGameInfo != NULL );
    ASSERT( pbResult != NULL );
    ASSERT( dwMaxLength > 0 );

    //  Grab the value if it exists.
    return( 
        m_pSettingsRegistry->getValue( 
            m_pGameInfo->getGameId( ),
            name, 
            pbResult, 
            dwMaxLength
        )
    );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: settingsGetValue
//
//  Description:
//
//      This member is called to retrieve the specified integer value
//      from the game's settings registry.
//
//  Parameters:
//
//      name (input)
//          The name of the value.
//
//      pnResult (input)
//          The destination buffer.
//
//  Returns:
//
//      1 if the integer was obtained.
//      0 if not found or invalid.
//
///////////////////////////////////////////////////////////////////////////////
const
DWord
Game::settingsGetValue(
    const KString& name,
    int32*         pnResult
)
{
    //  Preconditions.
    ASSERT( pnResult != NULL );

    //  The integer obtained.
    int32 nResult;

    
    //  If the integer was obtained then return 1, otherwise return 0.
    if( 
        settingsGetValue( name, ( Byte* )&nResult, sizeof( int32 ) ) == 
        sizeof( int32 )
    )
    {
        *pnResult = nResult;
        return( 1 );
    }
    else
    {
        return( 0 );
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: settingsGetValue
//
//  Description:
//
//      This member is called to retrieve the specified string value
//      from the game's settings registry.
//
//  Parameters:
//
//      name (input)
//          The name of the value.
//
//      pResult (input)
//          The destination string.
//
//  Returns:
//
//      The length of the string if the string was obtained.
//      0 if not found.
//
///////////////////////////////////////////////////////////////////////////////
const
DWord
Game::settingsGetValue(
    const KString& name,
    KString*       pResult
)
{
    //  Preconditions.
    ASSERT( pResult != NULL );

    //  Get the size of the desired value.
    DWord dwSize = settingsGetValueSize( name );

    //  The size to allocate for the string.
    DWord dwStringSize;


    //  If the value doesn't exist then just return, otherwise get the value.
    if( dwSize == 0 )
    {
        return( 0 );
    }
    else
    {
        //  Allocate some space for the value.
        Byte* pbBuffer = new Byte[ dwSize ];

        //  Get the value into the temporary buffer.
        dwStringSize = settingsGetValue( name, pbBuffer, dwSize );
        ASSERT( dwStringSize == dwSize );

        //  Assign it to the specified string.
        *pResult = ( char* )pbBuffer;

        //  All done with the temporary buffer.
        delete [] pbBuffer;

        //  We were successful.
        return( dwStringSize );
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: settingsSetValue
//
//  Description:
//
//      This member is called to set the specified settings registry value 
//      based on a given byte stream.
//
//  Parameters:
//
//      name (input)
//          The name of the value.
//
//      pbValue (input)
//          The value buffer.
//
//      dwLength (input)
//          The length of the buffer.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::settingsSetValue(
    const KString& name,
    Byte*          pbValue,
    DWord          dwLength /* = 1 */
)
{
    //  Preconditions.
    ASSERT( m_pGameInfo != NULL );
    ASSERT( pbValue != NULL );
    ASSERT( dwLength > 0 );

    //  Set the value.
    m_pSettingsRegistry->setValue( 
        m_pGameInfo->getGameId( ), name, pbValue, dwLength
    );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: settingsSetValue
//
//  Description:
//
//      This member is called to set the specified settings registry value 
//      based on a given integer.
//
//  Parameters:
//
//      name (input)
//          The name of the value.
//
//      nValue (input)
//          The value.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::settingsSetValue(
    const KString& name,
    int32          nValue
)
{
    //  Set the value.
    settingsSetValue( name, ( Byte* )&nValue, sizeof( int32 ) );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: settingsSetValue
//
//  Description:
//
//      This member is called to set the specified settings registry value 
//      based on a given string.
//
//  Parameters:
//
//      name (input)
//          The name of the value.
//
//      value (input)
//          The value.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::settingsSetValue(
    const KString& name,
    KString&       value
)
{
    //  Set the value.
    settingsSetValue( name, ( Byte* )value.data( ), value.length( ) );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: hiScorePostLoad
//
//  Description:
//
//      This member is called after the high scores have been loaded.
//      By default it does nothing but can be overriden by sub-classes
//      to perform processing in such a situation.
//
//  Parameters:
//
//      bExisting (input)
//          Indicates whether or not there were existing high scores to load.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::hiScorePostLoad( 
    const Byte /* bExisting is not used */
)
{
    //  Do nothing.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: hiScorePreSave
//
//  Description:
//
//      This member is called before the high scores have been saved.
//      By default it does nothing but can be overriden by sub-classes
//      to perform processing in such a situation.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::hiScorePreSave( 
)
{
    //  Do nothing.
}
