///////////////////////////////////////////////////////////////////////////////
//
//  File:    gameinfo.cpp
//
//  Class:   GameInfo
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class holds information on a particular emulated game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "gameinfo.h"
#include "game.h"
#include "replay.h"
#include "disk.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameInfo
//
//  Description:
//
//      This is the main constructor for a game information object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      ppGameInfoRaw (input)
//          This is the game information in raw form.  All games store there
//          information as a series of strings with certain identifying
//          strings indicating what the information is.  This is the raw
//          game information and must be parsed into a form usable by
//          this object.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameInfo::GameInfo(
    const KString& iName, 
    const char**   ppGameInfoRaw
)
:
    RepBase           ( iName ),
    m_gameId          ( ),
    m_gameName        ( ),
    m_versionList     ( 4 ),
    m_contributorList ( 4 ),
    m_reqdFileList    ( 12 )
{
    //  Parse the raw information.
    parseRawInfo( ppGameInfoRaw );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameInfo
//
//  Description:
//
//      This is the destructor for a game information object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameInfo::~GameInfo(
)
{
    //  Free the game information.
    for( DWord dwI = 0 ; dwI < m_versionList.entries( ) ; dwI += 1 )
    {
        delete m_versionList[ dwI ];
    }
    m_versionList.clear( );

    for( DWord dwI = 0 ; dwI < m_contributorList.entries( ) ; dwI += 1 )
    {
        delete m_contributorList[ dwI ];
    }
    m_contributorList.clear( );
  
    for( DWord dwI = 0 ; dwI < m_reqdFileList.entries( ) ; dwI += 1 )
    {
        delete m_reqdFileList[ dwI ];
    }
    m_reqdFileList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the game information object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
GameInfo::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "GameInfo" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: parseRawInfo
//
//  Description:
//
//      This member is responsible for filling out the member data by parsing
//      the game information from a raw list of strings.
//
//  Parameters:
//
//      ppGameInfoRaw (input)
//          The game information in raw form.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameInfo::parseRawInfo(
    const char** ppGameInfoRaw
)
{
    //  The current item we are parsing.
    enum
    {
        ITEM_NONE = 0,
        ITEM_ID,
        ITEM_NAME,
        ITEM_VERSION,
        ITEM_CONTRIBUTORS,
        ITEM_REQD_FILES
    }
    eCurrent = ITEM_NONE;
        
    //  A pointer used to traverse the list.
    const char* pTraverse;

    //  Traverse through the raw information.
    for( DWord dwI = 0 ; ; dwI += 1 )
    {
        //  Assign the traversal pointer.
        pTraverse = ppGameInfoRaw[ dwI ];

        //  If we have encountered a NULL then we have entered the start of
        //  a new section.
        if( pTraverse == NULL )
        {
            //  What is it? 
            dwI += 1;
            pTraverse = ppGameInfoRaw[ dwI ];
            if( strcmp( pTraverse, INFO_ID ) == 0 )
            {
                eCurrent = ITEM_ID;
            }
            else
            if( strcmp( pTraverse, INFO_NAME ) == 0 )
            {
                eCurrent = ITEM_NAME;
            }
            else
            if( strcmp( pTraverse, INFO_VERSION ) == 0 )
            {
                eCurrent = ITEM_VERSION;
            }
            else
            if( strcmp( pTraverse, INFO_CONTRIBUTORS ) == 0 )
            {
                eCurrent = ITEM_CONTRIBUTORS;
            }
            else
            if( strcmp( pTraverse, INFO_REQD_FILES ) == 0 )
            {
                eCurrent = ITEM_REQD_FILES;
            }
            else
            if( strcmp( pTraverse, INFO_END ) == 0 )
            {
                break;
            }
            else
            {
                fatalError( "Unknown token in game info (%s).", pTraverse );
            }
            dwI += 1;
            pTraverse = ppGameInfoRaw[ dwI ];
        }

        //  Add the item to our data.
        switch( eCurrent )
        {
            case ITEM_ID:           
            {
                m_gameId = pTraverse;        
                break;
            }
            case ITEM_NAME:         
            {
                m_gameName = pTraverse;      
                break;
            }
            case ITEM_VERSION:      
            {
                m_versionList.add( new KString( pTraverse ) );
                break;
            }
            case ITEM_CONTRIBUTORS:
            {
                m_contributorList.add( new KString( pTraverse ) );
                break;
            }
            case ITEM_REQD_FILES:
            {
                m_reqdFileList.add( new KString( pTraverse ) );
                break;
            }
            default:
            {
                //  Ignore.
                break;
            }
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: requiredFilesAvailable
//
//  Description:
//
//      This member returns whether or not all of the specified required
//      files are available.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if all of the files are available.
//      FALSE if any of the files are missing.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
GameInfo::requiredFilesAvailable(
)
{
    //  Run through the list of required files.
    for( DWord dwI = 0 ; dwI < m_reqdFileList.entries( ) ; dwI += 1 )
    {
        //  If the file cannot be found then return with a problem.
        if(
            Replay::s_instance( ).getDisk( )->gameFileExists(
                m_gameId, *( m_reqdFileList[ dwI ] )
            ) == FALSE
        )
        {
            return( FALSE );
        }
    }

    //  All files were found.
    return( TRUE );
}
