///////////////////////////////////////////////////////////////////////////////
//
//  File:    gameio.cpp
//
//  Class:   Game
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This file holds the static I/O functions for the game class.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "game.h"
#include "space.h"
#include "input.h"
#include "cqueue.h"
#include "ay8910.h"
#include "ym3812.h"
#include "hiscore.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_irqInterrupt
//
//  Description:
//
//      This is a default interrupt handler for games that require an IRQ
//      each time around.  It simply returns an
//      INT_IRQ if the interrupts are enabled or INT_NONE if not.
//
//  Parameters:
//
//      pCPU (input)
//          The CPU being interrupted.
//
//      pGame (input)
//          A pointer to the game.
//
//  Returns:
//
//      The interrupt value.
//
///////////////////////////////////////////////////////////////////////////////
int32
Game::s_irqInterrupt( 
    CPU*  pCPU,
    Game* /* pGame is unused */
)
{
    return( pCPU->intsEnabled( ) ? CPU::INT_IRQ : CPU::INT_NONE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_nmiInterrupt
//
//  Description:
//
//      This is a default interrupt handler for games that require an NMI
//      each time around.  It simply returns an
//      INT_NMI if the interrupts are enabled or INT_NONE if not.
//
//  Parameters:
//
//      pCPU (input)
//          The CPU being interrupted.
//
//      pGame (input)
//          A pointer to the game.
//
//  Returns:
//
//      The interrupt value.
//
///////////////////////////////////////////////////////////////////////////////
int32
Game::s_nmiInterrupt( 
    CPU*  pCPU,
    Game* /* pGame is unused */
)
{
    return( pCPU->intsEnabled( ) ? CPU::INT_NMI : CPU::INT_NONE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_noInterrupt
//
//  Description:
//
//      This is a default interrupt handler for games that do not want any 
//      interrupts.  It simply returns INT_NONE.
//
//  Parameters:
//
//      pCPU (input)
//          The CPU being interrupted.
//
//      pGame (input)
//          A pointer to the game.
//
//  Returns:
//
//      The interrupt value.
//
///////////////////////////////////////////////////////////////////////////////
int32
Game::s_noInterrupt( 
    CPU*  /* pCPU is unused  */,
    Game* /* pGame is unused */
)
{
    return( CPU::INT_NONE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_vectorInterrupt 
//
//  Description:
//
//      This is a default interrupt handler for a CPU that is to return
//      the interrupt vector of the CPU.
//
//  Parameters:
//
//      pCPU (input)
//          The CPU being interrupted.
//
//      pGame (input)
//          A pointer to the game.
//
//  Returns:
//
//      The interrupt value.
//
///////////////////////////////////////////////////////////////////////////////
int32
Game::s_vectorInterrupt( 
    CPU*  pCPU,
    Game* /* pGame is unused */
)
{
    //  Return the interrupt value.
    return( 
        pCPU->intsEnabled( ) ? pCPU->getIntVector( ) : ( int32 )CPU::INT_NONE 
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readNo
//
//  Description:
//
//      This is a default memory handler for no handler (just returns 0x00).
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the handler object.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readNo( 
    DWord        /* dwAddress is unused */, 
    ReadHandler* /* pHandler  is unused */
)
{
    return( 0x00 );
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readRAM
//
//  Description:
//
//      This is a default memory handler for reading a byte from RAM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readRAM( 
    DWord        dwAddress, 
    ReadHandler* /* pHandler is unused */
)
{
    return( CPU::sm_pSpace->getByte( dwAddress ) );
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readSharedRAM
//
//  Description:
//
//      This is a default memory handler for reading a byte from shared RAM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readSharedRAM( 
    DWord        dwAddress, 
    ReadHandler* pHandler
)
{
    //  The handler data contains a pointer to the start of the shared RAM.
    Byte* pbSharedRAM = ( Byte* )pHandler->getData( );
    return( pbSharedRAM[ dwAddress - pHandler->getStart( ) ] );
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readBuffer
//
//  Description:
//
//      This is a default memory handler for reading a byte from a buffer.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readBuffer( 
    DWord        dwAddress, 
    ReadHandler* pHandler
)
{
    //  The handler data contains a pointer to the start of the buffer.
    Byte* pbBuffer = ( Byte* )pHandler->getData( );
    return( pbBuffer[ dwAddress - pHandler->getStart( ) ] );
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readROM
//
//  Description:
//
//      This is a default memory handler for reading a byte from ROM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readROM( 
    DWord        dwAddress, 
    ReadHandler* /* pHandler is unused */
)
{
    return( CPU::sm_pSpace->getByte( dwAddress ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readInput
//
//  Description:
//
//      This is a default memory handler for reading a value from an Input.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readInput( 
    DWord        /* dwAddress is unused */,
    ReadHandler* pHandler 
)
{
    //  The data of the handler should contain the input.
    return( ( ( Input* )( pHandler->getData( ) ) )->getValue( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readInputs
//
//  Description:
//
//      This is a default memory handler for reading a value from a list
//      of inputs.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readInputs( 
    DWord        dwAddress,
    ReadHandler* pHandler 
)
{
    //  The data of the handler should contain the game pointer.
    Game* pThis = ( Game* )( pHandler->getData( ) );

    ASSERT( 
        ( dwAddress - pHandler->getStart( ) ) < pThis->m_inputList.entries( ) 
    );

    return( 
        pThis->m_inputList[ dwAddress - pHandler->getStart( ) ]->getValue( ) 
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readCommand
//
//  Description:
//
//      This is a default memory handler for reading a value from a command
//      queue.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readCommand( 
    DWord        /* dwAddress is unused */,
    ReadHandler* pHandler 
)
{
    //  The data of the handler should contain the command queue.
    return( ( ( CommandQueue* )( pHandler->getData( ) ) )->getCommand( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_read8910
//
//  Description:
//
//      This is a default memory handler for reading a value from an AY-3-8910
//      register.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_read8910( 
    DWord        /* dwAddress is unused */,
    ReadHandler* pHandler 
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Read from the 8910.
    return( ( ( AY8910* )( pHandler->getData( ) ) )->readRegister( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_read3812Read
//
//  Description:
//
//      This is a default memory handler for reading a value from a YM3812
//      register.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_read3812Read( 
    DWord        /* dwAddress is unused */,
    ReadHandler* pHandler 
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Read from the YM3812.
    return( ( ( YM3812* )( pHandler->getData( ) ) )->readRegister( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_read3812Status
//
//  Description:
//
//      This is a default memory handler for reading the status of a YM3812.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_read3812Status( 
    DWord        /* dwAddress is unused */,
    ReadHandler* pHandler 
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Get the status from the YM3812.
    return( ( ( YM3812* )( pHandler->getData( ) ) )->getStatus( ) );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readWatchDog
//
//  Description:
//
//      This is a default memory handler for the watchdog.  The watchdog is
//      reset each time this function is called.  If this doesn't happen
//      every 10 frames, the game resets.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to read a byte from.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
DWord
Game::s_readWatchDog( 
    DWord        /* dwAddress is unused */,
    ReadHandler* /* pHandler is unused  */
)
{
    //  Reset the watchdog.
    sm_pGame->m_nWatchDog = 10;
    return( 0x00 );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeNo
//
//  Description:
//
//      This is a default memory handler for no handler (do nothing).
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeNo( 
    DWord         /* dwAddress is unused */,
    Byte          /* bValue    is unused */,
    WriteHandler* /* pHandler  is unused */
)
{
    //  Do nothing.
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeRAM
//
//  Description:
//
//      This is a default memory handler for writing a byte to RAM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeRAM( 
    DWord         dwAddress, 
    Byte          bValue,
    WriteHandler* /* pHandler is unused */
)
{
    CPU::sm_pSpace->setByte( dwAddress, bValue );
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeSharedRAM
//
//  Description:
//
//      This is a default memory handler for writing a byte to shared RAM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeSharedRAM( 
    DWord         dwAddress, 
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The handler data contains a pointer to the start of the shared RAM.
    Byte* pbSharedRAM = ( Byte* )pHandler->getData( );
    pbSharedRAM[ dwAddress - pHandler->getStart( ) ] = bValue;
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeBuffer
//
//  Description:
//
//      This is a default memory handler for writing a byte to a buffer.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeBuffer( 
    DWord         dwAddress, 
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  The handler data contains a pointer to the start of the buffer.
    Byte* pbBuffer = ( Byte* )pHandler->getData( );
    pbBuffer[ dwAddress - pHandler->getStart( ) ] = bValue;
}
     


///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeROM
//
//  Description:
//
//      This is a default memory handler for writing a byte to ROM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeROM( 
#ifdef ROM_WRITE_ERROR
    DWord         dwAddress, 
#else
    DWord         /* dwAddress is unused */, 
#endif
    Byte          /* bValue    is unused */,
    WriteHandler* /* pHandler  is unused */
)
{
#ifdef ROM_WRITE_ERROR
    CONFIRM( 0, "Writing to ROM at 0x%05x\n", dwAddress );
#endif
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeDirtyByte
//
//  Description:
//
//      This is a default memory handler for writing a value to some
//      sort of attribute memory that has one byte per object.  For example, 
//      a memory location that indicates what character appears on the 
//      screen would make that character "dirty".
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeDirtyByte( 
    DWord         dwAddress, 
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  Only write the value if it has changed.
    if( CPU::sm_pSpace->getByte( dwAddress ) != bValue )
    {
        //  The dirty array is kept in the handler's client data.
        *( 
            ( Byte* )( pHandler->getData( ) ) +
            dwAddress - pHandler->getStart( )
        ) = TRUE;

        //  Set the value.
        CPU::sm_pSpace->setByte( dwAddress, bValue );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeDirtyWord
//
//  Description:
//
//      This is a default memory handler for writing a value to some
//      sort of attribute memory that has two bytes per object.  For example, 
//      two bytes in memory that contain the character colour and character
//      number would make a single character "dirty".
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeDirtyWord( 
    DWord         dwAddress, 
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  Only write the value if it has changed.
    if( CPU::sm_pSpace->getByte( dwAddress ) != bValue )
    {
        //  The dirty array is kept in the handler's client data.
        *( 
            ( Byte* )( pHandler->getData( ) ) +
            ( dwAddress - pHandler->getStart( ) ) / 2
        ) = TRUE;

        //  Set the value.
        CPU::sm_pSpace->setByte( dwAddress, bValue );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeHiScore
//
//  Description:
//
//      This is a default memory handler for writing a value to a high score
//      trigger address.  A trigger address is one that is the last address
//      written to before the high scores can be loaded.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeHiScore( 
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    //  Write the byte into the memory space.
    CPU::sm_pSpace->setByte( dwAddress, bValue );

    //  Tell the HiScore object to load itself.  It will only do the actual
    //  load after the requisite number of calls have been made to trigger
    //  the load.
    //  We only trigger this on a write to the last address of the handler
    //  in case of a multi-byte handler.
    if( dwAddress == pHandler->getStart( ) )
    {
        ( ( HiScore* )( pHandler->getData( ) ) )->loadScores( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeScreenRotate
//
//  Description:
//
//      This is a default memory handler for writing a value to a location
//      that causes the screen to rotate 180.  This is usually
//      needed in those games that were optionally produced in cocktail 
//      style cabinets where the screen would rotate for the second player.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeScreenRotate( 
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Point to the screen flipped member data of the game.
    Byte* pbScreenFlippedX = 
        &( ( ( Game* )( pHandler->getData( ) ) )->m_bScreenFlippedX );
    Byte* pbScreenFlippedY = 
        &( ( ( Game* )( pHandler->getData( ) ) )->m_bScreenFlippedY );

    //  If the value is not equal to the last value then rotate the screen
    //  180 degrees (i.e. two 90 degree turns).
    if( 
        ( bValue && !( *pbScreenFlippedX ) ) || 
        ( !bValue && *pbScreenFlippedX ) 
    )
    {
        //  Rotate the screen.
        Replay::s_instance( ).getCanvas( )->rotate( 2 );
        *pbScreenFlippedX = bValue;
        *pbScreenFlippedY = bValue;
    }

    //  Set the value in the regular address space.
    CPU::sm_pSpace->setByte( dwAddress, bValue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeScreenFlipX
//
//  Description:
//
//      This is a default memory handler for writing a value to a location
//      that causes the screen to flip in the X direction.  This is usually
//      needed in those games that were optionally produced in cocktail 
//      style cabinets where the screen would rotate for the second player.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeScreenFlipX( 
    DWord         /* dwAddress is not used */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Point to the screen flipped X member data of the game.
    Byte* pbScreenFlippedX = 
        &( ( ( Game* )( pHandler->getData( ) ) )->m_bScreenFlippedX );

    //  If the value is not equal to the last value then flip the screen.
    if( 
        ( bValue && !( *pbScreenFlippedX ) ) || 
        ( !bValue && *pbScreenFlippedX ) 
    )
    {
        Canvas* pCanvas = Replay::s_instance( ).getCanvas( );
        pCanvas->setOrientation( 
            pCanvas->getTranspose( ),
            !pCanvas->getFlipX( ),
            pCanvas->getFlipY( )
        );
        *pbScreenFlippedX = bValue;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeScreenFlipY
//
//  Description:
//
//      This is a default memory handler for writing a value to a location
//      that causes the screen to flip in the Y direction.  This is usually
//      needed in those games that were optionally produced in cocktail 
//      style cabinets where the screen would rotate for the second player.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeScreenFlipY( 
    DWord         /* dwAddress is not used */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Point to the screen flipped Y member data of the game.
    Byte* pbScreenFlippedY = 
        &( ( ( Game* )( pHandler->getData( ) ) )->m_bScreenFlippedY );

    //  If the value is not equal to the last value then flip the screen.
    if( 
        ( bValue && !( *pbScreenFlippedY ) ) || 
        ( !bValue && *pbScreenFlippedY ) 
    )
    {
        Canvas* pCanvas = Replay::s_instance( ).getCanvas( );
        pCanvas->setOrientation( 
            pCanvas->getTranspose( ),
            pCanvas->getFlipX( ),
            !pCanvas->getFlipY( )
        );
        *pbScreenFlippedY = bValue;
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeCommand
//
//  Description:
//
//      This is a default memory handler for writing a value to a command
//      queue.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeCommand( 
    DWord         /* dwAddress is not used */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Add the command to the queue.
    ( ( CommandQueue* )( pHandler->getData( ) ) )->addCommand( bValue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeCPUEnable
//
//  Description:
//
//      This is a default memory handler for enabling/disabling a CPU.
//      This handles two possibilities:
//          a)  A handler for a single location where the value indicates
//              whether to enable or disable.
//          b)  A handler for two locations where the address indicates
//              whether to enable or disable.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeCPUEnable( 
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Is this type (a) handler or type (b)?
    if( pHandler->getStart( ) == pHandler->getEnd( ) )
    {
        ( ( CPU* )( pHandler->getData( ) ) )->enable( bValue );
    }
    else
    {
        ASSERT( ( pHandler->getEnd( ) - pHandler->getStart( ) ) == 1 );
        ( ( CPU* )( pHandler->getData( ) ) )->enable( 
            dwAddress - pHandler->getStart( )
        );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeIntEnable
//
//  Description:
//
//      This is a default memory handler for enabling/disabling the interrupts
//      for a CPU.  This handles two possibilities:
//          a)  A handler for a single location where the value indicates
//              whether to enable or disable.
//          b)  A handler for two locations where the address indicates
//              whether to enable or disable.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeIntEnable( 
    DWord         dwAddress,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Is this type (a) handler or type (b)?
    if( pHandler->getStart( ) == pHandler->getEnd( ) )
    {
        ( ( CPU* )( pHandler->getData( ) ) )->enableInts( bValue );
    }
    else
    {
        ASSERT( ( pHandler->getEnd( ) - pHandler->getStart( ) ) == 1 );
        ( ( CPU* )( pHandler->getData( ) ) )->enableInts( 
            dwAddress - pHandler->getStart( )
        );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_write8910Write
//
//  Description:
//
//      This is a default memory handler for writing a value to an AY-3-8910
//      "write" register.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_write8910Write( 
    DWord         /* dwAddress is not used */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Write to the 8910.
    ( ( AY8910* )( pHandler->getData( ) ) )->writeRegister( bValue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_write8910Control
//
//  Description:
//
//      This is a default memory handler for writing a value to an AY-3-8910
//      "control" port.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_write8910Control( 
    DWord         /* dwAddress is not used */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Write to the 8910.
    ( ( AY8910* )( pHandler->getData( ) ) )->controlRegister( bValue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_write3812Write
//
//  Description:
//
//      This is a default memory handler for writing a value to a YM3812
//      "write" register.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_write3812Write( 
    DWord         /* dwAddress is not used */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Write to the 3812.
    ( ( YM3812* )( pHandler->getData( ) ) )->writeRegister( bValue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_write3812Control
//
//  Description:
//
//      This is a default memory handler for writing a value to a YM3812
//      "control" port.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      The byte requested.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_write3812Control( 
    DWord         /* dwAddress is not used */,
    Byte          bValue,
    WriteHandler* pHandler
)
{
    ASSERT( pHandler->getData( ) != NULL );

    //  Write to the 3812.
    ( ( YM3812* )( pHandler->getData( ) ) )->setRegister( bValue );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_writeWatchDog
//
//  Description:
//
//      This is a default memory handler for the watchdog.  The watchdog is
//      reset each time this function is called.  If this doesn't happen
//      every 10 frames, the game resets.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to write a byte to.
//
//      bValue (input)
//          The value to write.
//
//      pHandler (input)
//          A pointer to the memory handler.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game::s_writeWatchDog( 
    DWord         /* dwAddress is unused */,
    Byte          /* bValue is unused    */,
    WriteHandler* /* pHandler is unused  */
)
{
    //  Reset the watchdog.
    sm_pGame->m_nWatchDog = 10;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_isRAMHandler
//
//  Description:
//
//      This is a convenient function for checking whether the specified
//      function is one of the RAM handlers (read or write).
//
//  Parameters:
//
//      pfnHandler (input)
//          A pointer to the handler of interest.
//
//  Returns:
//
//      TRUE  if the handler is one of the RAM handlers.
//      FALSE if the handler is not one of the RAM handlers.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Game::s_isRAMHandler( 
    void* pfnHandler
)
{
    //  Check if it matches one of the read handlers.
    if( 
        ( pfnHandler == ( void* )s_readRAM ) || 
        ( pfnHandler == ( void* )s_writeRAM ) 
    )
    {
        return( TRUE );
    }
    else
    {
        return( FALSE );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_isROMHandler
//
//  Description:
//
//      This is a convenient function for checking whether the specified
//      function is one of the ROM handlers (read or write).
//
//  Parameters:
//
//      pfnHandler (input)
//          A pointer to the handler of interest.
//
//  Returns:
//
//      TRUE  if the handler is one of the ROM handlers.
//      FALSE if the handler is not one of the ROM handlers.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Game::s_isROMHandler( 
    void* pfnHandler
)
{
    //  Check if it matches one of the read handlers.
    if( 
        ( pfnHandler == ( void* )s_readROM ) || 
        ( pfnHandler == ( void* )s_writeROM ) 
    )
    {
        return( TRUE );
    }
    else
    {
        return( FALSE );
    }
}
