///////////////////////////////////////////////////////////////////////////////
//
//  File:    gamfact.cpp
//
//  Class:   GameFactory
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is responsible for creating game objects for the games
//      that Replay knows about.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////
//  System Headers.
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "config.h"
#include "gamfact.h"
#include "game.h"
#include "gameinfo.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a Game Factory object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object.
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
GameFactory*
GameFactory::s_build(
    const KString& iName
)
{
    //  Create the new object.
    GameFactory* pThis = new GameFactory( iName );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new object.
    return( pThis );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameFactory
//
//  Description:
//
//      This is the main constructor for a game factory object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameFactory::GameFactory(
    const KString& iName
)
:
    RepBase          ( iName ),
    m_gameFinderList ( GameFactory::s_compareGame, 32 ),
    m_pGame          ( NULL )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//  The following is used in the init() member to add game information to
//  the list of game information objects.
///////////////////////////////////////////////////////////////////////////////
#define ADD_GAME_INFO(ID)                                                     \
    extern const char** ID##_info( );                                         \
    extern Game*        ID##_create( );                                       \
    if(                                                                       \
        ( runGame == KStringNULL ) || ( strcmp( #ID, runGame.data( ) ) == 0 ) \
    )                                                                         \
    {                                                                         \
        gameFinderList.add(                                                   \
            new GameFinder(                                                   \
                #ID,                                                          \
                new GameInfo( #ID, ID##_info( ) ),                            \
                "",                                                           \
                ( void* )ID##_create                                          \
            )                                                                 \
        );                                                                    \
    }
///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the game factory object.  By using an 
//      init member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
#ifdef GAMES_static

extern "C" 
void
init( 
    KPtrSortList<GameFinder>& gameFinderList
)
{
    //  Has a specific game been specified?
    KString runGame;
    Configuration::s_instance( ).getParam( "-game", &runGame );

    ADD_GAME_INFO( amidar   );
    ADD_GAME_INFO( amidarjp );
    ADD_GAME_INFO( amigo    );
    //ADD_GAME_INFO( baddudes );
    ADD_GAME_INFO( bnj      );
    ADD_GAME_INFO( brubber  );
    ADD_GAME_INFO( btime    );
    ADD_GAME_INFO( btimeb   );
    ADD_GAME_INFO( caractn  );
    //ADD_GAME_INFO( foodf    );
    ADD_GAME_INFO( lnc      );
    ADD_GAME_INFO( pacnpal  );
    ADD_GAME_INFO( robocop  );
    ADD_GAME_INFO( robocopp );
    ADD_GAME_INFO( superpac );
    ADD_GAME_INFO( superpcn );
    ADD_GAME_INFO( turpin   );
    ADD_GAME_INFO( turtles  );
}
void
GameFactory::init(
)
{
    ::init( m_gameFinderList );
}

#else

void
GameFactory::init(
)
{
    fatalError( "Shared Games not supported on this platform." );
}

#endif
     

///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameFactory
//
//  Description:
//
//      This is the destructor for a Game Factory object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameFactory::~GameFactory(
)
{
    //  We should no longer have an active game.
    ASSERT( m_pGame == NULL );

    //  Free up the game information.
    for( DWord dwI = 0 ; dwI < m_gameFinderList.entries( ) ; dwI += 1 )
    {
        delete m_gameFinderList[ dwI ]->getGameInfo( );
        delete m_gameFinderList[ dwI ];
    }
    m_gameFinderList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Game Factory class.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
GameFactory::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "GameFactory" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: newGame
//
//  Description:
//
//      This member creates a new game based on the specified index.
//
//  Parameters:
//
//      dwGameId (input)
//          The index of the game to create.
//
//  Returns:
//
//      A pointer to the newly created game object.
//
///////////////////////////////////////////////////////////////////////////////
Game*
GameFactory::newGame(
    const DWord dwGameId
)
{
    ASSERT( m_pGame == NULL );
    ASSERT( dwGameId < m_gameFinderList.entries( ) );
    ASSERT( m_gameFinderList[ dwGameId ]->getHandle( ) != NULL );

    //  A pointer to the create function.
    Game*(* pfnCreate )( ) = 
        ( Game* ( * )( ) )m_gameFinderList[ dwGameId ]->getHandle( );
    m_pGame = ( *pfnCreate )( );

    //  Pass the game information to the game.
    m_pGame->setGameInfo( m_gameFinderList[ dwGameId ]->getGameInfo( ) );

    return( m_pGame );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: deleteGame
//
//  Description:
//
//      This member deletes the specified game.
//
//  Parameters:
//
//      pGame (input)
//          The game object to delete.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameFactory::deleteGame(
    Game* pGame
)
{
    ASSERT( pGame != NULL );
    ASSERT( pGame == m_pGame );
    delete pGame;
    m_pGame = NULL;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_compareGame
//
//  Description:
//
//      This member function is used to compare two games by name.
//
//  Parameters:
//
//      pGame1 (input)
//          A pointer to the first Game Finder object.
//
//      pGame2 (input)
//          A pointer to the second Game Finder object.
//
//  Returns:
//
//      -1 if the first game name is less than the second.
//       0 if the first game name is equal to the second.
//      +1 if the first game name is greater than the second.
//
///////////////////////////////////////////////////////////////////////////////
int
GameFactory::s_compareGame(
    const void* pGame1,
    const void* pGame2
)
{
    //  Cast the items to the appropriate type.
    GameFinder* pGameFinderA = *( ( GameFinder** )pGame1 );
    GameFinder* pGameFinderB = *( ( GameFinder** )pGame2 );

    //  Compare the names of the games.
    ASSERT( pGameFinderA->getGameInfo( ) != NULL );
    ASSERT( pGameFinderB->getGameInfo( ) != NULL );
    return(
        strcmp( 
            pGameFinderA->getGameInfo( )->getGameName( ).data( ),
            pGameFinderB->getGameInfo( )->getGameName( ).data( ) 
        )
    );
}
