///////////////////////////////////////////////////////////////////////////////
//
//  File:       gfxset.h
//
//  Class:      GfxSet - Concrete
//  
//  Hierarchy:  GfxSet - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class contains a set of graphics (sprites/chars/etc. for a game).
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _gfxset_h_
#define _gfxset_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "kplist.h"


//  Forward class declarations.
class Bitmap;


///////////////////////////////////////////////////////////////////////////////
//  GfxSet Class Definition
///////////////////////////////////////////////////////////////////////////////
class GfxSet : public RepBase
{
    public:

        //  Creation/Deletion.
                 GfxSet ( const KString& iName );
        virtual ~GfxSet ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Interface.
        void  setNumber     ( DWord dwNumber );
        DWord getNumber     ( ) const;
        void  setDimensions ( DWord dwWidth, DWord dwHeight );
        void  setBPP        ( Byte bBPP );
        void  setBitPlanes  ( DWord dwPlane0, ... );
        void  setXBits      ( DWord dwXStart0, ... );
        void  setYBits      ( DWord dwYStart0, ... );
        void  setIncrement  ( DWord dwIncrement );
        void  decode        ( Byte* pbBits );
        void  decode        ( Byte* pbBits, DWord dwChar );

        //  Access.
        Bitmap* operator[]  ( DWord dwIdx ) const;

    protected:

        //  Utility.
        Byte readBit       ( Byte* pbBits, DWord dwBit );
        void loadSmallFont ( );

        //  Constants.
        static const Byte  MAX_BPP;
        static const DWord MAX_GFX_WIDTH;
        static const DWord MAX_GFX_HEIGHT;

        //  Member data.
        DWord                 m_dwNumber;
        DWord                 m_dwWidth;
        DWord                 m_dwHeight;
        Byte                  m_bBPP;
        DWord*                m_pdwBPStart;
        DWord*                m_pdwXStart;
        DWord*                m_pdwYStart;
        DWord                 m_dwIncrement;
        KPtrList<Bitmap>      m_bitmapList;
 
    private:

};


///////////////////////////////////////////////////////////////////////////////
//  Input Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
void
GfxSet::setNumber(
    DWord dwNumber
)
{
    ASSERT( dwNumber > 0 );
    m_dwNumber = dwNumber;
}

inline
DWord
GfxSet::getNumber(
) const
{
    return( m_dwNumber );
}

inline
void
GfxSet::setDimensions(
    DWord dwWidth, 
    DWord dwHeight
)
{
    ASSERT( 
        ( dwWidth > 0 ) && 
        ( dwWidth <= MAX_GFX_WIDTH ) && 
        ( dwHeight > 0 ) && 
        ( dwWidth <= MAX_GFX_HEIGHT ) 
    );
    m_dwWidth  = dwWidth;
    m_dwHeight = dwHeight;
}

inline
void
GfxSet::setBPP(
    Byte bBPP
)
{
    ASSERT( ( bBPP > 0 ) && ( bBPP < GfxSet::MAX_BPP ) );
    m_bBPP = bBPP;
}

inline
void
GfxSet::setIncrement(
    DWord dwIncrement
)
{
    ASSERT( dwIncrement > 0 );
    m_dwIncrement = dwIncrement;
}

inline
Bitmap*
GfxSet::operator[](
    DWord dwIdx
) const
{
    ASSERT( dwIdx < m_dwNumber );

    return( m_bitmapList[ dwIdx ] );
}

#endif
