///////////////////////////////////////////////////////////////////////////////
//
//  File:    hiscore.cpp
//
//  Class:   HiScore
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class is used to keep track of the hi scores for a game.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <stdio.h>
#include <stdlib.h>

//  Application Headers.
#include "reptypes.h"
#include "hiscore.h"
#include "appfile.h"
#include "config.h"
#include "registry.h"
#include "game.h"
#include "gameinfo.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: HiScore
//
//  Description:
//
//      This is the main constructor for a hi score object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      pGame (input)
//          The game that the hi score object belongs to.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
HiScore::HiScore(
    const KString&  iName,
    Game*           pGame
)
:
    RepBase             ( iName ),
    m_bEnabled          ( TRUE ),
    m_bLoaded           ( FALSE ),
    m_pGame             ( pGame ),
    m_dwTriggerCount    ( 0 ),
    m_dwCurTriggerCount ( 0 ),
    m_rangeList         ( 3 )
{
    //  Make sure the parameters are valid.
    ASSERT( m_pGame != NULL );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~HiScore
//
//  Description:
//
//      This is the destructor for a hi score object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
HiScore::~HiScore(
)
{
    //  Delete the list of ranges.
    for( DWord dwI = 0 ; dwI < m_rangeList.entries( ) ; dwI += 1 )
    {
        delete m_rangeList[ dwI ];
    }
    m_rangeList.clear( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the hi score object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
HiScore::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "HiScore" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member will save the hi score object to a file for retrieval at 
//      a later date.
//
//  Parameters:
//
//      pSaveFile
//          The file to save the hi score object to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
HiScore::save(
    AppFile* pSaveFile
)
{
    ASSERT( pSaveFile != NULL );

    //  We always allow the base class to save itself first.
    RepBase::save( pSaveFile );

    //  Save the member data that keeps the state.
    pSaveFile->clearTotal( );
    pSaveFile->write( &m_bEnabled, sizeof( m_bEnabled ) );
    pSaveFile->write( &m_bLoaded, sizeof( m_bLoaded ) );
    pSaveFile->write( 
        ( Byte* )&m_dwCurTriggerCount, sizeof( m_dwCurTriggerCount ) 
    );

    //  Was it saved successfully?
    return(
        pSaveFile->total( ) != 
        ( 
            sizeof( m_bEnabled ) + 
            sizeof( m_bLoaded ) + 
            sizeof( m_dwCurTriggerCount ) 
        )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member will load the hi score object from a file that it was 
//      previously saved to.
//
//  Parameters:
//
//      pLoadFile
//          The file to load the hi score object from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
HiScore::load(
    AppFile* pLoadFile
)
{
    ASSERT( pLoadFile != NULL );

    //  We always allow the base class to load itself first.
    RepBase::load( pLoadFile );

    //  Load the member data that keeps the state.
    pLoadFile->clearTotal( );
    pLoadFile->read( &m_bEnabled, sizeof( m_bEnabled ) );
    pLoadFile->read( &m_bLoaded, sizeof( m_bLoaded ) );
    pLoadFile->read( 
        ( Byte* )&m_dwCurTriggerCount, sizeof( m_dwCurTriggerCount ) 
    );

    //  Was it loaded successfully?
    return(
        pLoadFile->total( ) != 
        ( 
            sizeof( m_bEnabled ) + 
            sizeof( m_bLoaded ) + 
            sizeof( m_dwCurTriggerCount ) 
        )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadScores
//
//  Description:
//
//      This member checks the loading conditions and if they indicate that
//      it is now time to load the hi scores then it does so.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if the high scores were just now loaded.
//      FALSE if the high scores were not just now loaded.
//
///////////////////////////////////////////////////////////////////////////////
Byte
HiScore::loadScores(
)
{
    //  Indicates whether there were existing high scores to load.
    Byte bExisting = FALSE;


    //  If the scores have already been loaded or the hiscore object is
    //  disabled then we're all done.
    if( m_bLoaded || !m_bEnabled )
    {
        return( FALSE );
    }

    //  If the trigger count is not zero then we have to check if the trigger
    //  has been reached.
    if( m_dwTriggerCount > 0 )
    {
        //  Up the current trigger count.
        m_dwCurTriggerCount += 1;

        //  If it's not time to load then just return.
        if( m_dwCurTriggerCount < m_dwTriggerCount )
        {
            return( FALSE );
        }
    }

    //  There should be at least one range.
    ASSERT( m_rangeList.entries( ) > 0 );

    //  Get the registry that contains the high scores.
    Registry* pRegistry = m_pGame->getHiScoreRegistry( );
    ASSERT( pRegistry != NULL );

    //  Load each of the high score ranges.
    for( DWord dwI = 0 ; dwI < m_rangeList.entries( ) ; dwI += 1 )
    {
        //  A convenient pointer to the current range.
        ScoreRange* pScoreRange = m_rangeList[ dwI ];

        //  Get the value from the registry.
        ASSERT( m_pGame->getGameInfo( ) != NULL );
        if( 
            pRegistry->getValue(
                m_pGame->getGameInfo( )->getGameId( ),
                pScoreRange->m_name,
                pScoreRange->m_pbLocation,
                pScoreRange->m_dwLength
            )
        )
        {
            bExisting = TRUE;
        }
    }

    //  Now let the game do any processing it requires.
    m_pGame->hiScorePostLoad( bExisting );

    //  The high scores have been loaded.
    m_bLoaded = TRUE;

    //  Inform the client that the scores were just loaded.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: saveScores
//
//  Description:
//
//      This member saves the high scores to the registry.  
//      WARNING:  This should only be called when the game is exiting or
//                being reset as it makes the assumption that the loading
//                trigger should be reset and the scores loaded again on
//                schedule.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if the high scores were saved.
//      FALSE if the high scores were not saved.
//
///////////////////////////////////////////////////////////////////////////////
Byte
HiScore::saveScores(
)
{
    //  If the scores haven't been loaded then we don't save.
    //  We allow saving if the object is disabled because it may have
    //  become disabled after a new high score was obtained.  Once the
    //  scores are saved, the disabling will kick in for attempted loads.
    if( !m_bLoaded )
    {
        return( FALSE );
    }

    //  There should be at least one range.
    ASSERT( m_rangeList.entries( ) > 0 );

    //  Let the game do any pre-processing.
    m_pGame->hiScorePreSave( );

    //  Get the registry that contains the high scores.
    Registry* pRegistry = m_pGame->getHiScoreRegistry( );

    //  Save each of the high score ranges.
    for( DWord dwI = 0 ; dwI < m_rangeList.entries( ) ; dwI += 1 )
    {
        //  A convenient pointer to the current range.
        ScoreRange* pScoreRange = m_rangeList[ dwI ];

        //  Set the value in the registry.
        ASSERT( m_pGame->getGameInfo( ) != NULL );
        pRegistry->setValue(
            m_pGame->getGameInfo( )->getGameId( ),
            pScoreRange->m_name,
            pScoreRange->m_pbLocation,
            pScoreRange->m_dwLength
        );
    }

    //  Mark the high scores as not having been loaded to force a load next 
    //  time.
    m_bLoaded = FALSE;

    //  Don't forget to reset the trigger count.
    m_dwCurTriggerCount = 0;

    //  The high scores have been saved.
    return( TRUE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: addRange
//
//  Description:
//
//      This member adds a range to the hi score.  A range is a group of
//      bytes within the game's memory space where the hi score data is
//      held.
//
//  Parameters:
//
//      pbLocation (input)
//          A pointer to the bytes where hi score data is stored.
//
//      dwLength (input)
//          The number of bytes of data.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void 
HiScore::addRange( 
    Byte* pbLocation, 
    DWord dwLength 
)
{
    //  Check the parameters.
    ASSERT( pbLocation != NULL );
    ASSERT( dwLength > 0 ); 

    //  Allocate a new range item.
    ScoreRange* pScoreRange = new ScoreRange;

    //  Create the name of the range.  Each range must have a unique name
    //  so that we can store it into the registry.
    sprintf( pScoreRange->m_name, "HiScore0x%04x", m_rangeList.entries( ) );

    //  Fill out the remainder of the new structure.
    pScoreRange->m_pbLocation = pbLocation;
    pScoreRange->m_dwLength   = dwLength;

    //  Add the new structure to our list of ranges.
    m_rangeList.add( pScoreRange );
}
