///////////////////////////////////////////////////////////////////////////////
//
//  File:    sample.cpp
//
//  Class:   Sample
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a sound sample.  A sound sample is simply
//      a buffer that holds sound data. 
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "sample.h"
#include "replay.h"
#include "disk.h"
#include "gfile.h"
#include "appfile.h"
#include "sound.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Sample
//
//  Description:
//
//      This is the main constructor for a sample object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwSize (input)
//          The size of the sample.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Sample::Sample(
    const KString&  iName,
    const DWord     dwSize /* = 0x01 */
)
:
    Buffer            ( iName, dwSize ),
    m_bInitialized    ( FALSE ),
    m_dwChannel       ( 0 ),
    m_dwFrequency     ( 0 ),
    m_bResolution     ( 0 ),
    m_bVolume         ( 0 )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Sample
//
//  Description:
//
//      This is the destructor for the sample object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Sample::~Sample(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the sample object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
Sample::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "Sample" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: save
//
//  Description:
//
//      This member is called to save the current state of the sample
//      a save game file.  This can be retrieved with the load member.
//
//  Parameters:
//
//      pSaveFile (input)
//          A pointer to the file to save to.
//
//  Returns:
//
//      TRUE if there were errors during the save.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Sample::save( 
    AppFile* pSaveFile
)
{
    //  The number of errors encountered during the save.
    DWord dwNumErrors = 0;

    //  We always allow the base class to save itself first.
    dwNumErrors += Buffer::save( pSaveFile );

    //  Now save our attributes.
    pSaveFile->clearTotal( );
    pSaveFile->write( &m_bInitialized, sizeof( m_bInitialized ) );
    pSaveFile->write( ( Byte* )&m_dwChannel, sizeof( m_dwChannel ) );
    pSaveFile->write( ( Byte* )&m_dwFrequency, sizeof( m_dwFrequency ) );
    pSaveFile->write( &m_bResolution, sizeof( m_bResolution ) );
    pSaveFile->write( &m_bVolume, sizeof( m_bVolume ) );
    if( pSaveFile->total( ) != ( 2 * sizeof( DWord ) + 3 * sizeof( Byte ) ) )
    {
        dwNumErrors += 1;
    }

    //  If there were errors encountered then indicate this to the client.
    return( dwNumErrors ? TRUE : FALSE );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: load
//
//  Description:
//
//      This member is called to load the current state of the sample from
//      a save game file that was created in the save member.
//
//  Parameters:
//
//      pLoadFile (input)
//          A pointer to the file to load from.
//
//  Returns:
//
//      TRUE if there were errors during the load.
//      FALSE if no errors were encountered.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Sample::load( 
    AppFile* pLoadFile
)
{
    //  The number of errors encountered during the load.
    DWord dwNumErrors = 0;


    //  We always allow the base class to load itself first.
    dwNumErrors += Buffer::load( pLoadFile );

    //  Now load our attributes.
    pLoadFile->clearTotal( );
    pLoadFile->read( &m_bInitialized, sizeof( m_bInitialized ) );
    pLoadFile->read( ( Byte* )&m_dwChannel, sizeof( m_dwChannel ) );
    pLoadFile->read( ( Byte* )&m_dwFrequency, sizeof( m_dwFrequency ) );
    pLoadFile->read( &m_bResolution, sizeof( m_bResolution ) );
    pLoadFile->read( &m_bVolume, sizeof( m_bVolume ) );
    if( pLoadFile->total( ) != ( 2 * sizeof( DWord ) + 3 * sizeof( Byte ) ) )
    {
        dwNumErrors += 1;
    }

    //  If there were errors encountered then indicate this to the client.
    return( dwNumErrors ? TRUE : FALSE );
}





///////////////////////////////////////////////////////////////////////////////
//
//  Function: setSize
//
//  Description:
//
//      This member sets a new size for the sample buffer.
//
//  Parameters:
//
//      dwSize (input)
//          The new size of the sample buffer.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Sample::setSize(
    const DWord dwSize
)
{
    ASSERT( dwSize > 0 );

    //  Setting the buffer size means the sample is considered initialized.
    m_bInitialized = TRUE;

    //  If the size hasn't changed then return.
    if( dwSize == m_dwSize )
    {
        return;
    }

    //  Reallocate the buffer.
    m_dwSize = dwSize;
    delete [] m_pbBuffer;
    m_pbBuffer = new Byte[ m_dwSize ]; 
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadSample
//
//  Description:
//
//      This member is called to load a sample file into the sample
//      buffer and set the parameters of the sample.
//
//  Parameters:
//
//      gameId (input)
//          The name of the game the Sample is for.
//
//      sampleName (input)
//          The name of the Sample.
//
//  Returns:
//
//      TRUE if the sample was loaded.
//      FALSE if the sample was not loaded.
//
///////////////////////////////////////////////////////////////////////////////
Byte
Sample::loadSample(
    const KString& gameId,
    const KString& sampleName
)
{
    //  The new size of the sample.
    DWord dwSize;

    //  The number of bytes read from the sample file.
    DWord dwBytesRead( 0 );

    //  Open the sample file.
    GameFile* pSampleFile = 
        Replay::s_instance( ).getDisk( )->gameFileOpen( gameId, sampleName );


    //  If the sample file was opened then load it in.
    if( pSampleFile != NULL )
    {
        m_bInitialized = TRUE;

        dwBytesRead += pSampleFile->read( 
            ( Byte* )&dwSize, sizeof( dwSize ), 4 
        );
        setSize( dwSize );
        dwBytesRead += pSampleFile->read( 
            ( Byte* )&m_dwFrequency, sizeof( m_dwFrequency ), 8 
        );
        dwBytesRead += pSampleFile->read( 
            ( Byte* )&m_bResolution, sizeof( m_bResolution ), 12 
        );
        dwBytesRead += pSampleFile->read( 
            ( Byte* )&m_bVolume, sizeof( m_bVolume ), 13 
        );
        dwBytesRead += pSampleFile->read( 
            m_pbBuffer, m_dwSize, 16 
        );
    
        CONFIRM( 
            dwBytesRead == m_dwSize + 10, 
            "Corrupt sample file %s.", 
            sampleName.data( )
        );
    }
    else
    {
        m_bInitialized = FALSE;
    }

    //  Was the sample loaded successfully?
    return( m_bInitialized );
}
