///////////////////////////////////////////////////////////////////////////////
//
//  File:    space.cpp
//
//  Class:   AddrSpace
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents an address space.  An address space is simply
//      a buffer that can be used to place data into.  They are most commonly
//      used as a CPU's address space.
//
//      Each address space has an addressable range depending on the CPU.
//      This is specified in the constructor as a number of bits.  The actual
//      size of the address space buffer may be the same size as the 
//      addressable range or it may be smaller or larger.
//
//      A single buffer is used to represent the address space and any 
//      banked memory.  Since the buffer can be less than the
//      size of the address space there are indices into the 
//      buffer indicating where the corresponding memory locations are
//      to be stored.  There are also pointers into the block to indicate
//      alternate memory banks.
//
//      The indices are kept in the form of 256 "page" pointers.  Each page
//      pointer points into the buffer if the range of the page is 
//      needed by the emulation or into a "spare" page for those addresses
//      that are not needed.
//
//      As an example, let's assume the game uses a 68000 processor.  This
//      processor has 24 address lines giving it an addressable range of
//      0x000000-0xffffff.  This is 16MB of space.  Now usually a game
//      doesn't use all 16MB and we would rather not allocate that much
//      memory if we aren't going to use it.  Now let's assume that the
//      game uses 0x000000-0x00ffff for the program rom and uses 
//      0x030000-0x03ffff for ram.  Also, let's say that the game uses
//      0x020000 bytes of ROM to store 2 banks of code that can be switched
//      into the game.  Therefore, the address space object would be set up as
//      follows:
//
//              [][][][]  -> buffer is 0x040000 bytes long where each [] 
//              ^ ^ ^ ^      represents 0x01000 bytes of memory.  1 for
//              | | | |      the program rom, one for the ram and two for
//              P P B B      the banked ram.
//              a a a a
//              g g n n ---> So the 0th page points to 0x000000 of the buffer.
//              e e k k      The 3rd page points to 0x010000 of the buffer.
//              0 3 0 1      The 0th bank points to 0x020000 of the buffer and
//                           the 1st bank points to 0x030000 of the buffer.
//
//
//                 [] -----> The remainder of the pages point to a "spare"
//                 ^         page.  This is so that we don't have to check
//       Page1 ----|         for a NULL page pointer during execution, we
//       Page2 ----|         can just allow reads/writes to go to this 
//       Page4 ----|         spare, unused page.
//       ...       |
//       Page5 ----|
//
//      In the case of a game with encrypted opcodes, an additional buffer
//      of the same size will be created which contains the decryted version
//      of the main buffer.  Similarly, there will be page pointers and bank
//      pointers into this buffer at the same locations as the regular
//      buffer.  There is still only one spare page that both the regular and
//      decrypted page pointers point to.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <string.h>
#include <stdarg.h>

//  Application Headers.
#include "reptypes.h"
#include "space.h"
#include "replay.h"
#include "disk.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Data Initialization.
///////////////////////////////////////////////////////////////////////////////
const DWord AddrSpace::sm_dwNumPages = 256;
const Byte  AddrSpace::sm_bMaxBanks  = 8;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: AddrSpace
//
//  Description:
//
//      This is the main constructor for an address space object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      bAddrBits (input)
//          The number of address bits that will be involved in addressing this
//          space.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
AddrSpace::AddrSpace(
    const KString&  iName,
    const Byte      bAddrBits /* = 16 */
)
:
    Buffer            ( iName, 0x10000 ),
    m_nEncryptType    ( CRYPT_NONE ),
    m_pbDecryptBuffer ( NULL ),
    m_bAddrBits       ( bAddrBits ),
    m_dwAddrSize      ( 1 << m_bAddrBits ),
    m_bPageBits       ( bAddrBits - 8 ),
    m_dwPageSize      ( 1 << m_bPageBits ),
    m_dwPageMask      ( m_dwPageSize - 1 ),
    m_ppbPages        ( NULL ),
    m_ppbDecryptPages ( NULL ),
    m_bNumBanks       ( 0 ),
    m_pdwBankSizes    ( NULL ),
    m_ppbBanks        ( NULL ),
    m_ppbDecryptBanks ( NULL ),
    m_pbSparePage     ( NULL ),
    m_bSwapped        ( FALSE )
{
    //  Make sure the address bits are a sensible number.  We currently
    //  accept 16 bit address spaces for cpus like the 6502 and Z80 and we
    //  accept 24 bit address spaces for cpus like the 68000.
    ASSERT( ( m_bAddrBits == 16 ) || ( m_bAddrBits == 24 ) );

    //  Allocate page and bank pointers.
    m_ppbPages        = new Byte*[ sm_dwNumPages ];
    m_ppbDecryptPages = new Byte*[ sm_dwNumPages ];
    m_pdwBankSizes    = new DWord[ sm_bMaxBanks ];
    m_ppbBanks        = new Byte*[ sm_bMaxBanks ];
    m_ppbDecryptBanks = new Byte*[ sm_bMaxBanks ];

    //  If this is a 16bit address space then we have allocated the full
    //  address space up front so we can set up the page pointers.  However,
    //  for a 24bit address space, we use the allocated space as the spare
    //  page and wait for the user to tell us which regions to allocate since
    //  we don't want to allocate the full addressable space.
    if( m_bAddrBits == 16 )
    {
        //  Assign the page pointers.
        for( DWord dwI = 0 ; dwI < sm_dwNumPages ; dwI += 1 )
        {
            m_ppbPages[ dwI ]        = m_pbBuffer + dwI * ( m_dwPageSize );
            m_ppbDecryptPages[ dwI ] = m_ppbPages[ dwI ];
        }
    }
    else
    {
        //  Use the buffer as the spare page and point all of the page pointers
        //  to it until we are told otherwise.
        m_pbSparePage = m_pbBuffer;
        m_pbBuffer    = NULL;
        m_dwSize      = 0;
        for( DWord dwI = 0 ; dwI < sm_dwNumPages ; dwI += 1 )
        {
            m_ppbPages[ dwI ]        = m_pbSparePage;
            m_ppbDecryptPages[ dwI ] = m_ppbPages[ dwI ];
        }
    }

    //  Initially, the decrypt buffer is the same as the regular buffer.
    m_pbDecryptBuffer = m_pbBuffer;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~AddrSpace
//
//  Description:
//
//      This is the destructor for the address space object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
AddrSpace::~AddrSpace(
)
{
    ASSERT( m_pbDecryptBuffer != NULL );

    //  If the decryption buffer is different from the main buffer than
    //  free it up.
    if( m_pbDecryptBuffer != m_pbBuffer )
    {
        delete [] m_pbDecryptBuffer;
    }

    //  Free up the spare page.
    delete [] m_pbSparePage;

    //  Free the page pointers.
    delete [] m_ppbPages;
    delete [] m_ppbDecryptPages;

    //  Free the bank information.
    delete [] m_pdwBankSizes;
    delete [] m_ppbBanks;
    delete [] m_ppbDecryptBanks;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Address Space object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
AddrSpace::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "AddrSpace" );

    return( className );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setSegments
//
//  Description:
//
//      This member is called to inform the address space of which memory
//      segments are required.
//
//  Parameters:
//
//      nStart (input)
//          The start token.
//
//      ... (input)
//          A list of memory segments.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AddrSpace::setSegments(
    const int32 nStart,
    ...
)
{
    //  The variable argument list.
    va_list ap;

    //  A pointer used to traverse the memory buffer.
    Byte* pbTraverse;


    //  Start should of course be the start token.
    ASSERT( nStart == SPACE_START );

    //  If this is a 16 bit address space then just return since the
    //  full address space has already been allocated.
    if( m_bAddrBits == 16 )
    {
        CHECK0( FALSE, "setSegments() call ignored for 16 bit address space." );
        return;
    }

    //  No decoding should have been done yet.
    ASSERT( m_nEncryptType == CRYPT_NONE );

    //  Start the variable argument list after the start token.
    va_start( ap, nStart );

    //  Initialize the size of the buffer to 0.
    m_dwSize = 0;

    //  Loop through until the END calculating the size of the required
    //  buffer.
    for( 
        int32 nToken  = va_arg( ap, int32 ) ; 
        nToken       != SPACE_END ; 
        nToken        = va_arg( ap, int32 )
    )
    {
        //  Add space based on the token.
        switch( nToken )
        {
            case SPACE_RANGE:
            {
                int32 nStart  = va_arg( ap, int32 ) >> m_bPageBits;
                int32 nEnd    = va_arg( ap, int32 ) >> m_bPageBits;

                ASSERT( ( DWord )nStart < sm_dwNumPages );
                ASSERT( ( DWord )nEnd   < sm_dwNumPages );
                ASSERT( nStart <= nEnd );

                m_dwSize += ( nEnd - nStart + 1 ) * m_dwPageSize;

                break;
            }
            case SPACE_BANK:
            {
                int32 nSize  = va_arg( ap, int32 );
                m_dwSize    += nSize;

                break;
            }
            default:
            {
                ASSERT( FALSE );

                break;
            }
        }
    }
    va_end( ap );
    
    //  Allocate the space for the new buffer.
    m_pbBuffer        = new Byte[ m_dwSize ];
    m_pbDecryptBuffer = m_pbBuffer;

    //  Clear the new buffer.
    memset( ( void* )m_pbBuffer, 0x00, m_dwSize );

    //  Reopen the variable arguments in preparation for setting up the
    //  page pointers.
    va_start( ap, nStart );
    
    //  Initialize the traversal pointer to the start of our buffer.
    pbTraverse = m_pbBuffer;

    //  Loop through until the END setting up the page pointers and
    //  bank pointers as we go.
    for( 
        int32 nToken  = va_arg( ap, int32 ) ; 
        nToken       != SPACE_END ; 
        nToken        = va_arg( ap, int32 )
    )
    {
        //  Set up pointers based on the token.
        switch( nToken )
        {
            case SPACE_RANGE:
            {
                int32 nStart  = va_arg( ap, int32 ) >> m_bPageBits;
                int32 nEnd    = va_arg( ap, int32 ) >> m_bPageBits;

                for( int32 nI = nStart ; nI <= nEnd ; nI += 1 )
                {
                    m_ppbPages[ nI ]         = pbTraverse;
                    m_ppbDecryptPages[ nI ]  = pbTraverse;

                    pbTraverse += m_dwPageSize;
                }

                break;
            }
            case SPACE_BANK:
            {
                ASSERT( m_bNumBanks < sm_bMaxBanks );

                int32 nSize  = va_arg( ap, int32 );

                m_ppbBanks[ m_bNumBanks ]         = pbTraverse;
                m_ppbDecryptBanks[ m_bNumBanks ]  = pbTraverse;
                m_pdwBankSizes[ m_bNumBanks ]     = nSize;
                m_bNumBanks                      += 1;

                pbTraverse += nSize;

                break;
            }
            default:
            {
                ASSERT( FALSE );
 
                break;
            }
        } 
    }
    va_end( ap );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadFile
//
//  Description:
//
//      This member is called to load a file (or a piece of) into the 
//      address space.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to store the file at.
//
//      gameId (input)
//          The name of the game the file is for.
//
//      fileName (input)
//          The name of the file.
//
//      dwOffset (input)
//          The offset into the file to start loading from.
//
//      dwLength (input)
//          The length to load in.
//
//  Returns:
//
//      TRUE  if the file was loaded.
//      FALSE if the file was not loaded.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
AddrSpace::loadFile(
    const DWord     dwAddress,
    const KString&  gameId,
    const KString&  fileName,
    const DWord     dwOffset /* =0x0000 */,
    const DWord     dwLength /* =0x0000 */
)
{
    //  A pointer to the location to load the buffer into.
    Byte* pbDest;

    //  The number of bytes to load into the file.
    DWord dwBytesToLoad;

 
    //  A file should not be loaded into an already decrypted buffer.
    ASSERT( m_nEncryptType == CRYPT_NONE );


    //  The file must be loaded within the addressable space.
    ASSERT( dwAddress < m_dwAddrSize );

    //  Find the start page that the location is found in.
    pbDest = m_ppbPages[ dwAddress >> m_bPageBits ];

    //  You can't load a ROM into the spare page.
    ASSERT( pbDest != m_pbSparePage );

    //  Now adjust within the page.
    pbDest += ( dwAddress & m_dwPageMask );
    

    //  If the length specified is 0 or if the length would exceed the
    //  end of the buffer then we calculate the number of bytes to load
    //  as the length between the destination and the end of the
    //  address space, otherwise we just use the length.
    if( 
        ( dwLength == 0x0000 ) || 
        ( ( pbDest + dwLength ) > m_pbBuffer + m_dwSize ) 
    )
    {
        dwBytesToLoad = ( m_pbBuffer + m_dwSize ) - pbDest;
    }
    else
    {
        dwBytesToLoad = dwLength;
    }


    //  Load the game file.
    return(
        Replay::s_instance( ).getDisk( )->gameFileLoad(
            gameId, fileName, pbDest, dwBytesToLoad, dwOffset
        )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadFileSkip
//
//  Description:
//
//      This member is called to load a file (or a piece of) into the 
//      buffer at every other byte.  This is typically
//      used for games that store the even bytes in one ROM and the odd
//      bytes in another ROM.
//
//  Parameters:
//
//      dwAddress (input)
//          The address to store the file at.
//
//      gameId (input)
//          The name of the game the file is for.
//
//      fileName (input)
//          The name of the file.
//
//      dwOffset (input)
//          The offset into the file to start loading from.
//
//      dwLength (input)
//          The length to load in.
//
//  Returns:
//
//      TRUE  if the file was loaded.
//      FALSE if the file was not loaded.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
AddrSpace::loadFileSkip(
    const DWord     dwAddress,
    const KString&  gameId,
    const KString&  fileName,
    const DWord     dwOffset /* =0x0000 */,
    const DWord     dwLength /* =0x0000 */
)
{
    //  A pointer to the location to load the buffer into.
    Byte* pbDest;

    //  The number of bytes to load into the file.
    DWord dwBytesToLoad;

 
    //  A file should not be loaded into an already decrypted buffer.
    ASSERT( m_nEncryptType == CRYPT_NONE );


    //  The file must be loaded within the addressable space.
    ASSERT( dwAddress < m_dwAddrSize );

    //  Find the start page that the location is found in.
    pbDest = m_ppbPages[ dwAddress >> m_bPageBits ];

    //  You can't load a ROM into the spare page.
    ASSERT( pbDest != m_pbSparePage );

    //  Now adjust within the page.
    pbDest += ( dwAddress & m_dwPageMask );
    

    //  If the length specified is 0 or if the length would exceed the
    //  end of the buffer then we calculate the number of bytes to load
    //  as the length between the destination and the end of the
    //  address space, otherwise we just use the length.
    if( 
        ( dwLength == 0x0000 ) || 
        ( ( pbDest + 2 * ( dwLength - 1 ) ) >= ( m_pbBuffer + m_dwSize ) )
    )
    {
        dwBytesToLoad = ( ( m_pbBuffer + m_dwSize ) - pbDest + 1 ) / 2;
    }
    else
    {
        dwBytesToLoad = dwLength;
    }


    //  Load the game file.
    return(
        Replay::s_instance( ).getDisk( )->gameFileLoad(
            gameId, fileName, pbDest, dwBytesToLoad, dwOffset, TRUE
        )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: loadBank
//
//  Description:
//
//      This member is called to load a file (or a piece of) into a
//      memory bank.
//
//  Parameters:
//
//      bBank (input)
//          The bank to load the file into.
//
//      gameId (input)
//          The name of the game the file is for.
//
//      fileName (input)
//          The name of the file.
//
//      dwOffset (input)
//          The offset into the file to start loading from.
//
//      dwLength (input)
//          The length to load in.
//
//  Returns:
//
//      TRUE  if the file was loaded.
//      FALSE if the file was not loaded.
//
///////////////////////////////////////////////////////////////////////////////
const
Byte
AddrSpace::loadBank(
    const Byte      bBank,
    const KString&  gameId,
    const KString&  fileName,
    const DWord     dwOffset /* =0x0000 */,
    const DWord     dwLength /* =0x0000 */
)
{
    //  A pointer to the location to load the buffer into.
    Byte* pbDest;

    //  The number of bytes to load into the file.
    DWord dwBytesToLoad;

 
    //  A file should not be loaded into an already decrypted buffer.
    ASSERT( m_nEncryptType == CRYPT_NONE );


    //  The bank must be valid.
    ASSERT( bBank < m_bNumBanks );

    //  Point the destination at the bank.
    pbDest = m_ppbBanks[ bBank ];

    //  If the length specified is 0 or if the length would exceed the
    //  end of the buffer then we calculate the number of bytes to load
    //  as the length between the destination and the end of the
    //  address space, otherwise we just use the length.
    if( 
        ( dwLength == 0x0000 ) || 
        ( ( pbDest + dwLength ) > m_pbBuffer + m_dwSize ) 
    )
    {
        dwBytesToLoad = ( m_pbBuffer + m_dwSize ) - pbDest;
    }
    else
    {
        dwBytesToLoad = dwLength;
    }


    //  Load the game file.
    return(
        Replay::s_instance( ).getDisk( )->gameFileLoad(
            gameId, fileName, pbDest, dwBytesToLoad, dwOffset
        )
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: setEncryptType
//
//  Description:
//
//      This member is called to set the encryption type for the address
//      space.  
//
//  Parameters:
//
//      nEncryptType (input)
//          Indicates the type of encryption present.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AddrSpace::setEncryptType ( 
    const int32 nEncryptType
)
{
    //  Only allow the encryption type to be set once.
    ASSERT( m_nEncryptType == CRYPT_NONE );

    //  For opcode decryption, we need to allocate a seperate buffer for
    //  the decryption since we can't tell which are opcodes and which is
    //  data and therefore decrypt the whole buffer.  So for opcode
    //  decryption, the new decrypt buffer is treated as if everything is
    //  an opcode so that CPU cores can grab opcodes from that buffer and
    //  data from the main buffer.
    if( nEncryptType == CRYPT_OP )
    {
        //  Create the new buffer.
        ASSERT( m_pbDecryptBuffer == m_pbBuffer );
        m_pbDecryptBuffer = new Byte[ m_dwSize ];
        memcpy( ( void* )m_pbDecryptBuffer, ( void* )m_pbBuffer, m_dwSize );

        //  Reassign the decrypt page pointers.
        for( DWord dwI = 0 ; dwI < sm_dwNumPages ; dwI += 1 )
        {
            m_ppbDecryptPages[ dwI ] = 
                m_pbDecryptBuffer + ( m_ppbPages[ dwI ] - m_pbBuffer );
        }

        //  Reassign the bank pointers.
        for( Byte bI = 0 ; bI < m_bNumBanks ; bI += 1 )
        {
            m_ppbDecryptBanks[ bI ] = 
                m_pbDecryptBuffer + ( m_ppbBanks[ bI ] - m_pbBuffer );
        }
    }

    //  Record the encryption type.
    m_nEncryptType = nEncryptType;
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: swapBytes
//
//  Description:
//
//      This member is called to swap consecutive bytes within the address
//      space.  
//
//  Parameters:
//
//      bSwapped (input)
//          Indicates whether the bytes are to be swapped relative to the
//          original or whether they should be returned to their original
//          state.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
AddrSpace::swapBytes ( 
    const Byte bSwapped
)
{
    //  A temporary byte used for swapping.
    Byte bTemp;


    //  Only swap if the swapping state has changed.
    if( bSwapped != m_bSwapped )
    {
        m_bSwapped = bSwapped;
       
        for( DWord dwI = 0 ; dwI < m_dwSize - 1 ; dwI += 2 )
        {
            bTemp                 = m_pbBuffer[ dwI ];
            m_pbBuffer[ dwI ]     = m_pbBuffer[ dwI + 1 ];
            m_pbBuffer[ dwI + 1 ] = bTemp;
            
            if( m_pbDecryptBuffer != m_pbBuffer )
            {
                bTemp                        = m_pbDecryptBuffer[ dwI ];
                m_pbDecryptBuffer[ dwI ]     = m_pbDecryptBuffer[ dwI + 1 ];
                m_pbDecryptBuffer[ dwI + 1 ] = bTemp;
            }
        }
    }
}
