///////////////////////////////////////////////////////////////////////////////
//
//  File:       space.h
//
//  Class:      AddrSpace - Concrete
//  
//  Hierarchy:  AddrSpace - Buffer - RepBase 
//
//  Author:     Kevin Brisley
//
//  Description:
//
//      This class represents an address space.  An address space is simply
//      a buffer that can be used to place data into.  They are most commonly
//      used as a CPU's address space.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef _space_h_
#define _space_h_

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "reptypes.h"
#include "repbase.h"
#include "kstring.h"
#include "buffer.h"
#include "ctrlmap.h"
#include "bitmap.h"


///////////////////////////////////////////////////////////////////////////////
//  AddrSpace Class Definition
///////////////////////////////////////////////////////////////////////////////
class AddrSpace : public Buffer
{
    public:

        //  The possible encryption types.
        enum
        {
            CRYPT_NONE,
            CRYPT_OP,
            CRYPT_FULL
        };

        //  Tokens for setting memory segments.
        enum
        {
            SPACE_START,
            SPACE_RANGE,
            SPACE_BANK,
            SPACE_END
        }; 

        //  Creation/Deletion.
                 AddrSpace ( const KString& iName, const Byte bAddrBits = 16 );
        virtual ~AddrSpace ( );

        //  Base overrides.
        virtual const KString& getClassName ( ) const;

        //  Interface.
        void         setSegments ( const int32 nStart, ... );
        virtual const Byte loadFile ( 
            const DWord    dwAddress, 
            const KString& gameId,
            const KString& fileName,
            const DWord    dwOffset = 0x0000,
            const DWord    dwLength = 0x0000
        );
        virtual const Byte loadFileSkip (
            const DWord    dwAddress,
            const KString& gameId,
            const KString& fileName,
            const DWord    dwOffset = 0x0000,
            const DWord    dwLength = 0x0000
        );
        const Byte         loadBank (
            const Byte     bBank,
            const KString& gameId,
            const KString& fileName,
            const DWord    dwOffset = 0x0000,
            const DWord    dwLength = 0x0000
        );
        void swapBytes              ( const Byte bSwapped );

        //  Access.
        Byte* getPointer       ( const DWord dwAddress );
        Byte  getByte          ( const DWord dwAddress );
        void  setByte          ( const DWord dwAddress, const Byte bValue );
        Byte  getDecryptByte   ( const DWord dwAddress );
        void  setDecryptByte   ( const DWord dwAddress, const Byte bValue );
        void  setEncryptType   ( const int32 nEncryptType );
        int32 getEncryptType   ( ) const;
        Byte* getDecryptBuffer ( ) const;

    protected:

        //  Member data.
        int32  m_nEncryptType;
        Byte*  m_pbDecryptBuffer;
        Byte   m_bAddrBits;
        DWord  m_dwAddrSize;
        Byte   m_bPageBits;
        DWord  m_dwPageSize;
        DWord  m_dwPageMask;
        Byte** m_ppbPages;
        Byte** m_ppbDecryptPages;
        Byte   m_bNumBanks;
        DWord* m_pdwBankSizes;
        Byte** m_ppbBanks;
        Byte** m_ppbDecryptBanks;
        Byte*  m_pbSparePage;
        Byte   m_bSwapped;

        //  Static member data.
        static const DWord sm_dwNumPages;
        static const Byte  sm_bMaxBanks;

    private:

};


///////////////////////////////////////////////////////////////////////////////
//  AddrSpace Inline Functions.
///////////////////////////////////////////////////////////////////////////////
inline
Byte*
AddrSpace::getPointer(
    const DWord dwAddress
)
{
    ASSERT( dwAddress < m_dwAddrSize );
    ASSERT( m_ppbPages[ dwAddress >> m_bPageBits ] != m_pbSparePage );

    return(
        m_ppbPages[ dwAddress >> m_bPageBits ] + ( dwAddress & m_dwPageMask ) 
    );
}

inline
Byte
AddrSpace::getByte(
    const DWord dwAddress
)
{
    ASSERT( dwAddress < m_dwAddrSize );

    CHECK1( 
        m_ppbPages[ dwAddress >> m_bPageBits ] != m_pbSparePage,
        "24 bit address space does not have 0x%06lx allocated for read.",
        dwAddress
    );

    return(
        *( 
            m_ppbPages[ dwAddress >> m_bPageBits ] + 
            ( dwAddress & m_dwPageMask ) 
        ) 
    );
}

inline
void
AddrSpace::setByte(
    const DWord dwAddress,
    const Byte  bValue
)
{
    ASSERT( dwAddress < m_dwAddrSize );

    CHECK1( 
        m_ppbPages[ dwAddress >> m_bPageBits ] != m_pbSparePage,
        "24 bit address space does not have 0x%06lx allocated for write.",
        dwAddress
    );

    *( m_ppbPages[ dwAddress >> m_bPageBits ] + ( dwAddress & m_dwPageMask ) ) =
        bValue;
}

inline
Byte
AddrSpace::getDecryptByte(
    const DWord dwAddress
)
{
    ASSERT( dwAddress < m_dwAddrSize );

    return(
        *( 
            m_ppbDecryptPages[ dwAddress >> m_bPageBits ] + 
            ( dwAddress & m_dwPageMask ) 
        ) 
    );
}

inline
void
AddrSpace::setDecryptByte(
    const DWord dwAddress,
    const Byte  bValue
)
{
    ASSERT( dwAddress < m_dwAddrSize );

    *( 
        m_ppbDecryptPages[ dwAddress >> m_bPageBits ] + 
        ( dwAddress & m_dwPageMask ) 
    ) = bValue;
}

inline
int32
AddrSpace::getEncryptType(
) const
{
    return( m_nEncryptType );
}

inline
Byte*
AddrSpace::getDecryptBuffer(
) const
{
    return( m_pbDecryptBuffer );
}

///////////////////////////////////////////////////////////////////////////////
//  AddrSpace Macros.
///////////////////////////////////////////////////////////////////////////////
#define MEM_START      AddrSpace::SPACE_START
#define MEM_RANGE(a,b) AddrSpace::SPACE_RANGE, a, b
#define MEM_BANK(a)    AddrSpace::SPACE_BANK, a
#define MEM_END        AddrSpace::SPACE_END

#endif
