///////////////////////////////////////////////////////////////////////////////
//
//  File:    _6809pac.cpp
//
//  Class:   Game6809Pac
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract base class for Pac-Man type games on Bally's
//      dual 6809 hardware.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "_6809pac.h"
#include "space.h"
#include "input.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: Game6809Pac
//
//  Description:
//
//      This is the main constructor for the 6809 Pac games.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Game6809Pac::Game6809Pac(
    const KString& iName
)
:
    GameBallyMidway6809 ( iName ),
    m_pbCustom1RAM      ( NULL ),
    m_pbCustom2RAM      ( NULL ),
    m_pbSharedRAM       ( NULL ),
    m_bCredits          ( 0 ),
    m_bCoinsLeft        ( 0 ),
    m_bCoinsRight       ( 0 ),
    m_bCoinLeft         ( 0 ),
    m_bCoinRight        ( 0 ),
    m_bStart1           ( 0 ),
    m_bStart2           ( 0 )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~Game6809Pac
//
//  Description:
//
//      This is the destructor for the 6809 Pac games.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
Game6809Pac::~Game6809Pac(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of the 6809 Pac games.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
Game6809Pac::startUpSpace(
)
{
    //  Allow the base class to perform the address space startup.
    GameBallyMidway6809::startUpSpace( );

    //  Assign some convenient RAM pointers.
    m_pbCustom1RAM = m_pSpaceGame->getBuffer( ) + 0x4800;
    m_pbCustom2RAM = m_pSpaceGame->getBuffer( ) + 0x4810;
    m_pbSharedRAM  = m_pSpaceGame->getBuffer( ) + 0x4040;
}

void
Game6809Pac::startUpMemMap(
)
{
    //  Register the Memory Read Handlers for the Game CPU.
    m_pCPUGame->readMemHandler( 
        "Scratch RAM",    0x0000, 0x1fff, s_readRAM,          NULL
    );
    m_pCPUGame->readMemHandler( 
        "Custom I/O #1",  0x4800, 0x480f, s_readCustom1,      this
    );
    m_pCPUGame->readMemHandler( 
        "Custom I/O #2",  0x4810, 0x481f, s_readCustom2,      this
    );
    m_pCPUGame->readMemHandler( 
        "ROM",            0xa000, 0xffff, s_readROM,          NULL 
    );

    //  Register the Memory Write Handlers for the Game CPU.
    m_pCPUGame->writeMemHandler(
        "Video RAM",      0x0000, 0x03ff, s_writeDirtyByte,    
        m_pDirty->getBuffer( )
    );
    m_pCPUGame->writeMemHandler(
        "Colour RAM",     0x0400, 0x07ff, s_writeDirtyByte,    
        m_pDirty->getBuffer( )
    );
    m_pCPUGame->writeMemHandler(
        "RAM",            0x0800, 0x1fff, s_writeRAM,          NULL
    );
/*
    m_pCPUGame->writeMemHandler(
        "Int Enable",     0x5002, 0x5003, s_writeIntEnable,    m_pCPUGame
    );
*/
    m_pCPUGame->writeMemHandler(
        "Sound Enable",   0x5008, 0x5009, s_writeSoundEnable,  m_pWaveForm
    );
    m_pCPUGame->writeMemHandler(
        "CPU Enable",     0x500a, 0x500b, s_writeCPUEnable,    m_pCPUSound
    );
    m_pCPUGame->writeMemHandler(
        "ROM",            0xa000, 0xffff, s_writeROM,          NULL
    );

    //  Register the Memory Read Handlers for the Sound CPU.
    m_pCPUSound->readMemHandler( 
        "Shared RAM",     0x0040, 0x03ff, s_readSharedRAM,    m_pbSharedRAM
    );
    m_pCPUSound->readMemHandler( 
        "ROM",            0xf000, 0xffff, s_readROM,          NULL
    );

    //  Register the Memory Write Handlers for the Sound CPU.
    m_pCPUSound->writeMemHandler(
        "Shared RAM",     0x0040, 0x03ff, s_writeSharedRAM,     m_pbSharedRAM
    );
    m_pCPUSound->writeMemHandler(
        "Sound Regs.",    0x0000, 0x003f, s_writeSoundRegister, this
    );
    m_pCPUSound->writeMemHandler(
        "Int Enable",     0x2000, 0x2001, s_writeIntEnable,     m_pCPUSound
    );
    m_pCPUSound->writeMemHandler(
        "Sound Enable",   0x2006, 0x2007, s_writeSoundEnable,   m_pWaveForm
    );
    m_pCPUSound->writeMemHandler(
        "ROM",            0xf000, 0xffff, s_writeROM,           NULL         
    );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readCustom1
//
//  Description:
//
//      This member is used to read a value from the first Custom I/O chip.
//      The first custom I/O chip controls coin insertion and game starting.
//
//  Parameters:
//
//      dwAddress (input)
//          The address read from.
//
//      pHandler (input)
//          A pointer to the memory handler. 
//
//  Returns:
//
//      The value of the chip.
//
///////////////////////////////////////////////////////////////////////////////
DWord 
Game6809Pac::s_readCustom1( 
    DWord        dwAddress, 
    ReadHandler* pHandler 
)
{
    //  The following array indicates the number of coins per credit
    //  corresponding to the 4 specific dip switch settings for the
    //  right slot and the 8 specific dip switch settings for the left slot.
    static Byte abRightRatio[ 4 ][ 2 ] =
    {
        { 1, 1 }, { 1, 2 }, { 2, 1 }, { 2, 3 }
    };
    static Byte abLeftRatio[ 8 ][ 2 ] = 
    { 
        { 1, 1 }, { 1, 2 }, { 1, 3 }, { 1, 6 },
        { 1, 7 }, { 2, 1 }, { 2, 3 }, { 3, 1 }
    }; 

    //  The data of the handler contains the pointer to our game.
    Game6809Pac* pThis = ( Game6809Pac* )( pHandler->getData( ) );

    //  The register being accessed.
    DWord dwReg = dwAddress - pHandler->getStart( );


    //  The credit information is found in DSW0 for the right slot and
    //  DSW1 for the left.
    Byte bCoinDipRight = ( pThis->m_pInputDSW0->getValue( ) & 0x30 ) >> 4;
    Byte bCoinDipLeft  = pThis->m_pInputDSW1->getValue( ) & 0x07;

    //  The status of the coin mechs is contained in the low two bits of IN1.
    Byte bMech = pThis->m_pInputIN1->getValue( ) & 0x0f;
    if( bMech & 0x01 )
    {
       if( !( pThis->m_bCoinRight ) )
       {
           pThis->m_bCoinRight   = TRUE;
           pThis->m_bCoinsRight += 1;
           if( pThis->m_bCoinsRight == abRightRatio[ bCoinDipRight ][ 0 ] )
           {
               pThis->m_bCredits    += abRightRatio[ bCoinDipRight ][ 1 ];
               pThis->m_bCoinsRight  = 0;
           }
       }
    }
    else
    {
        pThis->m_bCoinRight = FALSE;
    }
    if( bMech & 0x02 )
    {
        if( !( pThis->m_bCoinLeft ) )
        {
            pThis->m_bCoinLeft   = TRUE;
            pThis->m_bCoinsLeft += 1;
           if( pThis->m_bCoinsLeft == abLeftRatio[ bCoinDipLeft ][ 0 ] )
           {
               pThis->m_bCredits   += abLeftRatio[ bCoinDipLeft ][ 1 ];
               pThis->m_bCoinsLeft  = 0;
           }
        }
    }
    else
    {
        pThis->m_bCoinLeft = FALSE;
    }

    //  The start 1 and start 2 buttons are contained in bits 4&5 of IN1.
    Byte bStart = pThis->m_pInputIN1->getValue( ) >> 4;
    if( bStart & 0x01 )
    {
        if( !( pThis->m_bStart1 ) && pThis->m_bCredits )
        {
            pThis->m_bCredits -= 1;
            pThis->m_bStart1   = TRUE;
        }
    }
    else
    {
        pThis->m_bStart1 = FALSE;
        if( bStart & 0x02 )
        {
            if( !( pThis->m_bStart2 ) && ( pThis->m_bCredits > 1 ) )
            {
                pThis->m_bCredits -= 2;
                pThis->m_bStart2   = TRUE;
            }
        }
        else
        {
            pThis->m_bStart2 = FALSE;
        }
    }

   
    //  Now deal with the custom I/O chip.  The mode of the chip is contained
    //  in location 8 of the custom I/O memory.
    switch( pThis->m_pbCustom1RAM[ 8 ] )
    {
        case 1:
        case 3:
        {
            switch( dwReg )
            {
                case 0:
                {
                    return( pThis->m_pInputIN1->getValue( ) & 0x0f );
                }
                case 1:
                {
                    return( pThis->m_pInputIN0->getValue( ) & 0x0f );
                }
                case 2:
                {
                    return( pThis->m_pInputIN2->getValue( ) & 0x0f );
                }
                case 3:
                {
                    //  This contains the buttons for both players (IN0,IN2)
                    //  along with the player start buttons (IN2) that overlap
                    //  the upper button bits.
                    Byte bRet =
                        ( ( pThis->m_pInputIN0->getValue( ) >> 4 ) & 0x01 ) |
                        ( ( pThis->m_pInputIN0->getValue( ) >> 3 ) & 0x04 ) |
                        ( ( pThis->m_pInputIN2->getValue( ) >> 3 ) & 0x02 ) |
                        ( ( pThis->m_pInputIN2->getValue( ) >> 2 ) & 0x08 ) |
                        ( ( pThis->m_pInputIN1->getValue( ) >> 2 ) & 0x0c );

                    return( bRet );
                }
                case 4:
                case 5:
                case 6:
                case 7:
                {
                    return( 0x0f );
                }
                default:
                {
                    return( pThis->m_pbCustom1RAM[ dwReg ] );
                }
            }
            break;
        }
        case 4:
        {
            switch( dwReg )
            {
                case 0:
                {
                    return( pThis->m_bCredits / 10 );
                }
                case 1:
                {
                    return( pThis->m_bCredits % 10 );
                }
                case 4:
                {
                    return( pThis->m_pInputIN0->getValue( ) & 0x0f );
                }
                case 5:
                {
                    return( pThis->m_pInputIN0->getValue( ) >> 4 );
                }
                case 6:
                {
                    return( pThis->m_pInputIN2->getValue( ) & 0x0f );
                }
                case 7:
                {
                    return( pThis->m_pInputIN2->getValue( ) >> 4 );
                }
                default:
                {
                    return( pThis->m_pbCustom1RAM[ dwReg ] );
                }
            }
            break;
        }
        case 8:
        {
            pThis->m_bCredits    = 0;
            pThis->m_bCoinsRight = 0;
            pThis->m_bCoinsLeft  = 0;
            if( ( dwReg > 8 ) && ( dwReg < 16 ) )
            {
                return( 0 );
            }
            break;
        }
    }

    //  By default, just return the value in the register.
    return( pThis->m_pbCustom1RAM[ dwReg ] );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readCustom2
//
//  Description:
//
//      This member is used to read a value from the second Custom I/O chip.
//      The second custom I/O chip is used to handle player controls.
//
//  Parameters:
//
//      dwAddress (input)
//          The address read from.
//
//      pHandler (input)
//          A pointer to the memory handler. 
//
//  Returns:
//
//      The value of the chip.
//
///////////////////////////////////////////////////////////////////////////////
DWord 
Game6809Pac::s_readCustom2( 
    DWord        dwAddress, 
    ReadHandler* pHandler 
)
{
    //  The data of the handler contains the pointer to our game.
    Game6809Pac* pThis = ( Game6809Pac* )( pHandler->getData( ) );

    //  The register being accessed.
    DWord dwReg = dwAddress - pHandler->getStart( );


    //  Deal with the custom I/O chip.  The mode of the chip is contained
    //  in location 8 of the custom I/O memory.
    switch( pThis->m_pbCustom2RAM[ 8 ] )
    {
        case 3:
        {
            switch( dwReg )
            {
                case 0:
                case 1:
                case 2:
                case 3:
                {
                    return( 0x00 );
                }
                case 4:
                {
                    return( pThis->m_pInputDSW0->getValue( ) & 0x0f );
                }
                case 5:
                {
                    return( pThis->m_pInputDSW1->getValue( ) >> 4 );
                }
                case 6:
                {
                    return( pThis->m_pInputDSW1->getValue( ) & 0x0f );
                }
                case 7:
                {
                    return( ( pThis->m_pInputIN1->getValue( ) & 0xc0 ) >> 4 );
                }
                default:
                {
                    return( pThis->m_pbCustom2RAM[ dwReg ] );
                }
            }
            break;
        }
        case 8:
        {
            pThis->m_bCredits    = 0;
            pThis->m_bCoinsRight = 0;
            pThis->m_bCoinsLeft  = 0;
            if( ( dwReg > 8 ) && ( dwReg < 16 ) )
            {
                return( 0 );
            }
            break;
        }
        case 9:
        {
            switch( dwReg )
            {
                case 0:
                {
                    return( pThis->m_pInputDSW1->getValue( ) & 0x0f );
                }
                case 1:
                {
                    return( pThis->m_pInputDSW1->getValue( ) >> 4 );
                }
                case 2:
                {
                    return( 0x00 );
                }
                case 3:
                {
                    return( pThis->m_pInputDSW0->getValue( ) & 0x0f );
                }
                case 4:
                {
                    return( pThis->m_pInputDSW0->getValue( ) >> 4 );
                }
                case 5:
                {
                    return( 0x00 );
                }
                case 6:
                {
                    return( ( pThis->m_pInputIN1->getValue( ) & 0xc0 ) >> 4 );
                }
                case 7:
                {
                    return( 0x00 );
                }
                default:
                {
                    return( pThis->m_pbCustom2RAM[ dwReg ] );
                }
            }
            break;
        }
    }

    //  By default, just return the value in the register.
    return( pThis->m_pbCustom2RAM[ dwReg ] );
}
