///////////////////////////////////////////////////////////////////////////////
//
//  File:    _spac.cpp
//
//  Class:   GameSuperPacManBase
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This is an abstract base class for Super Pac-Man games.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "_spac.h"
#include "gameinfo.h"
#include "space.h"
#include "input.h"
#include "dip.h"
#include "hiscore.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: GameSuperPacManBase
//
//  Description:
//
//      This is the main constructor for Super Pac-Man.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameSuperPacManBase::GameSuperPacManBase(
    const KString& iName
)
:
    Game6809Pac ( iName )
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~GameSuperPacManBase
//
//  Description:
//
//      This is the destructor for Super Pac-Man.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
GameSuperPacManBase::~GameSuperPacManBase(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: startUpXXXXX
//
//  Description:
//
//      The following member functions are used to start up various aspects
//      of Super Pac-Man.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
GameSuperPacManBase::startUpInput(
)
{
    //  The base class will create the inputs and setup the controls.
    Game6809Pac::startUpInput( );

    //  Set up the dip switches.
    DipSwitch* pDip;
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW0", m_pInputDSW0->getLocation( ), FALSE )
    );
    pDip->addSetting(
        "Difficulty",
        0x0f,
        "Difficulty 0-Standard",
        "Difficulty 1-Easiest",
        "Difficulty 2",
        "Difficulty 3",
        "Difficulty 4",
        "Difficulty 5",
        "Difficulty 6",
        "Difficulty 7",
        "Difficulty 8-Default",
        "Difficulty 9",
        "Difficulty A",
        "Difficulty B-Hardest",
        "Difficulty C-Easiest Auto",
        "Difficulty D-Auto",
        "Difficulty E-Auto",
        "Difficulty F-Hardest Auto"
    );
    pDip->addSetting(
        "Demo Sound",
        0x40,
        "Demo Sound On",
        "Demo Sound Off"
    );
    pDip->addSetting(
        "Operation",
        0x80,
        "Normal Operation",
        "Freeze Video"
    );
    pDip->addSetting(
        "Right Coin",
        0x30,
        "Right 1 Coin/1 Credit",
        "Right 1 Coin/2 Credits",
        "Right 2 Coins/1 Credit",
        "Right 2 Coins/3 Credits"
    );
    pDip = m_dipSwitchList.add(
        new DipSwitch( "DSW1", m_pInputDSW1->getLocation( ), FALSE )
    );
    pDip->addSetting(
        "Left Coin",
        0x07,
        "Left 1 Coin/1 Credit",
        "Left 1 Coin/2 Credits",
        "Left 1 Coin/3 Credits",
        "Left 1 Coin/6 Credits",
        "Left 1 Coin/7 Credits",
        "Left 2 Coins/1 Credit",
        "Left 2 Coins/3 Credits",
        "Left 3 Coins/1 Credit"
    );
    pDip->addSetting(
        "Bonus Life",
        0x38,
        "Bonus Lives at 30k/100k",
        "Bonus Lives at 30k/80k",
        "Bonus Lives at 30k/120k",
        "Bonus Lives at 30k/80k/80k",
        "Bonus Lives at 30k/100k/100k",
        "Bonus Lives at 30k/120k/120k",
        "Bonus Life at 30k",
        "No Bonus Lives"
    );
    pDip->addSetting(
        "Lives",
        0xc0,
        "3 Lives",
        "1 Life",
        "2 Lives",
        "5 Lives"
    );


    //  The Super Pac-Man high scores are stored at:
    //      0x1138: High Score table = 40 bytes.
    //      0x1087: Current High Score = 3 bytes.
    //  Super Pac-Man writes to this area 18 times before settling down.
    m_pHiScore = new HiScore( "HiScore", this );
    m_pHiScore->setTrigger( 18 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x1138, 40 );
    m_pHiScore->addRange( m_pSpaceGame->getBuffer( ) + 0x1087,  3 );
}

void
GameSuperPacManBase::startUpColour(
)
{
    //  The Super Pac-Man colour PROM.  It contains the palette in the first
    //  32 bytes and then contains the character colour lookup table in the 
    //  next 256 bytes followed by the sprite colour lookup table in the final
    //  256 bytes.
    static Byte abColourPROM[ ] =
    {
        0xf6,0xc9,0x3f,0x07,0xef,0xf8,0x2f,0xaf,
        0x3c,0x5d,0x38,0xe7,0x29,0x66,0x54,0x00,
        0x00,0xd8,0x66,0x29,0xe7,0x38,0x5d,0xd5,
        0xaf,0x2f,0xb8,0xef,0x07,0x3f,0xf6,0x00,

        0x00,0x00,0x00,0x00,0x00,0x02,0x02,0x00,
        0x00,0x03,0x01,0x00,0x00,0x0b,0x03,0x0e,
        0x00,0x0b,0x01,0x0e,0x00,0x06,0x01,0x0b,
        0x00,0x0b,0x01,0x08,0x00,0x04,0x01,0x07,
        0x00,0x04,0x01,0x05,0x00,0x02,0x06,0x04,
        0x00,0x02,0x06,0x03,0x00,0x01,0x01,0x01,
        0x00,0x03,0x01,0x0d,0x00,0x02,0x06,0x0d,
        0x00,0x0d,0x09,0x01,0x00,0x09,0x07,0x0a,
        0x00,0x01,0x06,0x09,0x00,0x02,0x09,0x0a,
        0x00,0x03,0x0b,0x0a,0x00,0x01,0x07,0x03,
        0x00,0x04,0x0b,0x0a,0x00,0x0a,0x01,0x0c,
        0x00,0x01,0x04,0x03,0x00,0x06,0x09,0x01,
        0x00,0x02,0x06,0x01,0x00,0x0a,0x01,0x0c,
        0x00,0x03,0x02,0x0e,0x00,0x01,0x0b,0x03,
        0x00,0x00,0x03,0x00,0x00,0x00,0x04,0x00,
        0x00,0x0a,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,

        0x0f,0x0f,0x0f,0x0f,0x0f,0x02,0x02,0x02,
        0x0f,0x00,0x00,0x00,0x0f,0x00,0x04,0x05,
        0x0f,0x03,0x00,0x01,0x0f,0x04,0x00,0x01,
        0x0f,0x05,0x00,0x01,0x0f,0x06,0x00,0x01,
        0x0f,0x01,0x04,0x00,0x00,0x00,0x00,0x00,
        0x0f,0x0f,0x00,0x01,0x0f,0x07,0x00,0x00,
        0x0f,0x0f,0x0f,0x0f,0x0f,0x08,0x0f,0x0f,
        0x0f,0x08,0x08,0x0f,0x0f,0x08,0x08,0x08,
        0x0f,0x03,0x00,0x0d,0x0f,0x02,0x06,0x0d,
        0x0f,0x0d,0x0e,0x00,0x0f,0x09,0x07,0x0a,
        0x0f,0x00,0x06,0x09,0x0f,0x02,0x09,0x0a,
        0x0f,0x03,0x0b,0x08,0x0f,0x00,0x07,0x03,
        0x0f,0x04,0x0b,0x0a,0x0f,0x0a,0x0c,0x00,
        0x0f,0x00,0x04,0x03,0x0f,0x06,0x09,0x00,
        0x0f,0x02,0x00,0x06,0x0f,0x0a,0x00,0x0c,
        0x0f,0x03,0x02,0x01,0x0f,0x00,0x0b,0x03,
        0x0f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x0f,0x00,0x00,0x00,0x0f,0x00,0x00,0x00,
        0x0f,0x05,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x0f,0x02,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x0f,0x02,0x0f,0x0f,0x0f,0x02,0x02,0x0f,
        0x0f,0x02,0x02,0x02,0x0f,0x02,0x02,0x0f,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x0f,0x00,0x04,0x00,0x0f,0x00,0x05,0x00,
        0x0f,0x00,0x06,0x00,0x0f,0x00,0x08,0x00
    };

    //  Assign the colour PROM.
    m_pColourPROM = new Buffer( 
        "Colour PROM", 
        sizeof( abColourPROM ) / sizeof( abColourPROM[ 0 ] ),
        abColourPROM
    );

    //  The base class will initialize the colours.
    Game6809Pac::startUpColour( );
}

void
GameSuperPacManBase::startUpSound(
)
{
    //  The Super Pac-Man sound PROM. 
    static Byte abSoundPROM[ ] =
    {
	    0x07,0x09,0x0a,0x0b,0x0c,0x0d,0x0d,0x0e,
        0x0e,0x0e,0x0d,0x0d,0x0c,0x0b,0x0a,0x09,
	    0x07,0x05,0x04,0x03,0x02,0x01,0x01,0x00,
        0x00,0x00,0x01,0x01,0x02,0x03,0x04,0x05,
	    0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
        0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f, 
        0x0f,0x0e,0x0d,0x0c,0x0b,0x0a,0x09,0x08,
        0x07,0x06,0x05,0x04,0x03,0x02,0x01,0x00,
	    0x07,0x08,0x06,0x09,0x05,0x0a,0x04,0x0b,
        0x03,0x0c,0x02,0x0d,0x01,0x0e,0x00,0x0f,
	    0x00,0x0f,0x01,0x0e,0x02,0x0d,0x03,0x0c,
        0x04,0x0b,0x05,0x0a,0x06,0x09,0x07,0x08,
	    0x07,0x0a,0x0c,0x0d,0x0e,0x0d,0x0c,0x0a,
        0x07,0x04,0x02,0x01,0x00,0x01,0x02,0x04,
	    0x07,0x0b,0x0d,0x0e,0x0d,0x0b,0x07,0x03,
        0x01,0x00,0x01,0x03,0x07,0x0e,0x07,0x00,
	    0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
        0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,0x0e,
	    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	    0x0e,0x0c,0x07,0x0a,0x0c,0x07,0x05,0x0c,
        0x0e,0x0d,0x08,0x0a,0x0b,0x06,0x02,0x07,
	    0x0c,0x08,0x03,0x04,0x06,0x01,0x00,0x03,
        0x09,0x07,0x02,0x04,0x07,0x02,0x00,0x07,
	    0x07,0x0c,0x0e,0x0e,0x0d,0x0b,0x09,0x0a,
        0x0b,0x0b,0x0a,0x09,0x06,0x04,0x03,0x05,
	    0x07,0x09,0x0b,0x0a,0x08,0x05,0x04,0x03,
        0x03,0x04,0x05,0x03,0x01,0x00,0x00,0x02,
	    0x0c,0x0e,0x0e,0x0e,0x0e,0x0c,0x0b,0x09,
        0x07,0x06,0x05,0x05,0x05,0x06,0x08,0x0a,
	    0x0b,0x0b,0x09,0x06,0x04,0x02,0x02,0x02,
        0x03,0x05,0x04,0x03,0x02,0x01,0x03,0x07
    };

    //  Assign the sound PROM.
    m_pSoundPROM = new Buffer( 
        "Sound PROM", 
        sizeof( abSoundPROM ) / sizeof( abSoundPROM[ 0 ] ),
        abSoundPROM
    );

    //  Call the base class.
    Game6809Pac::startUpSound( );
}

void
GameSuperPacManBase::startUpMemMap(
)
{
    //  Set some specific handlers before the default handlers are set.
    m_pCPUGame->writeMemHandler(
        "Score Trigger",  0x115f, 0x115f, s_writeHiScore,      m_pHiScore
    );
    m_pCPUGame->writeMemHandler(
        "Screen Rotate",  0x1085, 0x1085, s_writeScreenRotate, this
    );

    //  Allow the base class to set the remainder of the handlers.
    Game6809Pac::startUpMemMap( );

    //  We place it after the base class because it overrides the default.
    m_pCPUSound->readMemHandler( 
        "Shared RAM",     0x0040, 0x03ff, s_readSharedRAM,    this
    );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_readSharedRAM
//
//  Description:
//
//      This member is called when the sound CPU reads a value from the 
//      shared RAM area.  Super Pac-Man uses a special handler to detect
//      when the sound CPU goes into a tight loop.
//
//  Parameters:
//
//      dwAddress (input)
//          The address read from.
//
//      pHandler (input)
//          A pointer to the memory handler. 
//
//  Returns:
//
//      The value of the RAM location.
//
///////////////////////////////////////////////////////////////////////////////
DWord 
GameSuperPacManBase::s_readSharedRAM( 
    DWord        dwAddress, 
    ReadHandler* pHandler 
)
{
    //  The data of the handler contains the pointer to our game.
    GameSuperPacManBase* pThis = 
        ( GameSuperPacManBase* )( pHandler->getData( ) );

    //  The shared memory is mapped to 0x4040 in the game space and 0x0040
    //  in the sound space.  Get the value of the requested address.
    Byte bValue = pThis->m_pbSharedRAM[ dwAddress - 0x40 ];

    //  The sound CPU may go into a tight loop reading from a particular
    //  location until a command arrives.  Since there won't be a command 
    //  until the other CPU gets to execute, we might as well complete this 
    //  CPU slice if there is no command.
    if( ( dwAddress == 0x00fb ) && ( bValue == 0 ) )
    {
        //  Cease operation of the current CPU so the interrupt can take
        //  effect.
        CPU::sm_pCPU->setCyclesLeft( 0 );
    }
        
    //  Now return the value.
    return( bValue );
}
