///////////////////////////////////////////////////////////////////////////////
//
//  File:    bitmapl.cpp
//
//  Class:   BitmapLinux
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      This class represents a bitmap that can be displayed on the screen
//      for the Linux platform.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  Application Headers.
#include "bitmapl.h"
#include "clip.h"
#include "bmp.h"



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a Linux bitmap object.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the bitmap.
//
//      dwHeight (input)
//          The height of the bitmap.
//
//      bScreen (input)
//          Indicates whether or not this bitmap is a screen bitmap.
//
//      bScale (input)
//          Indicates the scale of the image buffer in relation to the
//          bitmap buffer.  A scale of 0 indicates that the same buffer
//          should be used for both.
//
//      pDisplay (input)
//          The X-Display of the Replay application.
//
//      pScreen (input)
//          The X-Screen of the Replay application.
//
//      pVisual (input)
//          The visual of the screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
BitmapLinux*
BitmapLinux::s_build(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bScreen    /* = FALSE */,
    const Byte      bScale     /* = 0     */,
    Display*        pDisplay   /* = NULL  */,
    Screen*         pScreen    /* = NULL  */,
    XVisualInfo*    pVisual    /* = NULL  */
)
{
    //  Create the new object.
    BitmapLinux* pThis = new BitmapLinux( 
        iName, dwWidth, dwHeight, bScreen, bScale, pDisplay, pScreen, pVisual
    );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new bitmap.
    return( pThis );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: BitmapLinux
//
//  Description:
//
//      This constructor creates a blank bitmap of the specified dimensions.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//      dwWidth (input)
//          The width of the bitmap.
//
//      dwHeight (input)
//          The height of the bitmap.
//
//      bScreen (input)
//          Indicates whether or not this bitmap is a screen bitmap.
//
//      pDisplay (input)
//          The X-Display of the Replay application.
//
//      pScreen (input)
//          The X-Screen of the Replay application.
//
//      pVisual (input)
//          The visual of the screen.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
BitmapLinux::BitmapLinux(
    const KString&  iName,
    const DWord     dwWidth,
    const DWord     dwHeight,
    const Byte      bScreen    /* = FALSE */,
    const Byte      bScale     /* = 0     */,
    Display*        pDisplay   /* = NULL  */,
    Screen*         pScreen    /* = NULL  */,
    XVisualInfo*    pVisual    /* = NULL  */
)
:
    BitmapUnixX ( 
        iName, 
        dwWidth, 
        dwHeight, 
        bScreen, 
        bScale, 
        pDisplay, 
        pScreen, 
        pVisual 
    )
{
    //  Initialization should be done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~BitmapLinux
//
//  Description:
//
//      This is the destructor for a bitmap.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
BitmapLinux::~BitmapLinux(
)
{
    //  Nothing to do.
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the Linux screen bitmap object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
BitmapLinux::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "BitmapLinux" );

    return( className );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: blit
//
//  Description:
//
//      These is a simple blit method that simply places the specified bitmap
//      at the location specified by the mapping for this bitmap.  This is
//      typically used for characters and tiles that have static locations
//      on a bitmap.
//
//  Parameters:
//
//      pSrc (input)
//          The bitmap to blit from.
//
//      dwItem (input)
//          The item as referenced in the map.
//
//      pdwColourTable (input)
//          The table used to map the values in the bitmap to the pixel
//          values.
//  
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
BitmapLinux::blit( 
    Bitmap*      pSrc,
    DWord        dwItem, 
    DWord*       pdwColourTable
)
{
    //  Find the destination of this item.
    int32 nDestX = m_pnMapX[ dwItem ];
    int32 nDestY = m_pnMapY[ dwItem ];


    //  If the item isn't clipped then draw it.
    if( nDestX != MAP_CLIPPED )
    {
        //  Make sure the source fits into the bitmap.
        ASSERT( nDestX + s_getWidth( pSrc )  <= m_dwWidth  );
        ASSERT( nDestY + s_getHeight( pSrc ) <= m_dwHeight );
    
        //  If the canvas is transposed then we need to swap the X/Y locations.
        if( m_pCanvas->getTranspose( ) )
        {
            int32 nSwap = nDestX;
            nDestX      = nDestY;
            nDestY      = nSwap;
        }
    
        //  If the canvas is X flipped then we need to adjust the X coordinate
        //  to the opposite side of the bitmap in the horizontal direction.  
        if( m_pCanvas->getFlipX( ) )
        {
            nDestX = m_dwWidth - s_getWidth( pSrc ) - nDestX;
        }
            
        //  If the canvas is Y flipped then we need to adjust the Y coordinate 
        //  to the opposite side of the bitmap in the vertical direction.
        if( m_pCanvas->getFlipY( ) )
        {
            nDestY = m_dwHeight - s_getHeight( pSrc ) - nDestY;
        }
    

        //  For speed we let an ASM routine handle it.
        blitInfo.DestStart   = m_ppbRows;
        blitInfo.DestWidth   = m_dwWidth;
        blitInfo.SrcStart    = pSrc->getRows( );
        blitInfo.SrcWidth    = s_getWidth( pSrc );
        blitInfo.SrcHeight   = s_getHeight( pSrc );
        blitInfo.DestX       = nDestX;
        blitInfo.DestY       = nDestY;
        blitInfo.ColourTable = pdwColourTable;
        bitmapBlitChar( );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: blit
//
//  Description:
//
//      These is a more complicated blit method that allows flipping and
//      clipping.  This is typically used for sprites or scrolled layers.
//
//  Parameters:
//
//      pSrc (input)
//          The bitmap to blit from.
//
//      nOX (input)
//          The X coordinate to blit to.
//
//      nOY (input)
//          The Y coordinate to blit to.
//
//      pdwColourTable (input)
//          The table used to map the values in the bitmap to the pixel
//          values.
//
//      bFlipX (input)
//          Indicates whether or not to flip the image horizontally.
//
//      bFlipY (input)
//          Indicates whether or not to flip the image vertically.
//    
//      pClipping (input)
//          The clipping region.
//
//      eTransparency (input)
//          The type of transparency.
//  
//      bTransparentColour (input)
//          The transparent colour.
//  
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
BitmapLinux::blit( 
    Bitmap*      pSrc,
    int32        nOX,
    int32        nOY,
    DWord*       pdwColourTable,
    Byte         bFlipX,
    Byte         bFlipY,
    Clipping*    pClipping,
    Transparency eTransparency,
    Byte         bTransparentColour /* = 0x00 */
)
{
    //  Check the parameters.
    ASSERT( pClipping != NULL );

    //  Set the initial values.
    if( m_pCanvas->getTranspose( ) )
    {
        blitInfo.DestX    = nOY;
        blitInfo.DestY    = nOX;
        blitInfo.FlipX    = bFlipY;
        blitInfo.FlipY    = bFlipX;
        blitInfo.ClipMinX = pClipping->m_nMinY;
        blitInfo.ClipMaxX = pClipping->m_nMaxY;
        blitInfo.ClipMinY = pClipping->m_nMinX;
        blitInfo.ClipMaxY = pClipping->m_nMaxX;
    }
    else
    {
        blitInfo.DestX    = nOX;
        blitInfo.DestY    = nOY;
        blitInfo.FlipX    = bFlipX;
        blitInfo.FlipY    = bFlipY;
        blitInfo.ClipMinX = pClipping->m_nMinX;
        blitInfo.ClipMaxX = pClipping->m_nMaxX;
        blitInfo.ClipMinY = pClipping->m_nMinY;
        blitInfo.ClipMaxY = pClipping->m_nMaxY;
    }
    if( m_pCanvas->getFlipX( ) )
    {
        blitInfo.DestX    = m_dwWidth - s_getWidth( pSrc ) - blitInfo.DestX;
        blitInfo.ClipMinX = m_dwWidth - 1 - blitInfo.ClipMaxX;
        blitInfo.ClipMaxX = m_dwWidth - 1 - blitInfo.ClipMinX;
    }
    if( m_pCanvas->getFlipY( ) )
    {
        blitInfo.DestY    = m_dwHeight - s_getHeight( pSrc ) - blitInfo.DestY;
        blitInfo.ClipMinY = m_dwHeight - 1 - blitInfo.ClipMaxY;
        blitInfo.ClipMaxY = m_dwHeight - 1 - blitInfo.ClipMinY;
    }

    //  Finish off setting the information.
    blitInfo.DestStart         = m_ppbRows;
    blitInfo.DestWidth         = m_dwWidth;
    blitInfo.DestHeight        = m_dwHeight;
    blitInfo.SrcStart          = pSrc->getRows( );
    blitInfo.SrcWidth          = s_getWidth( pSrc );
    blitInfo.SrcHeight         = s_getHeight( pSrc );
    blitInfo.ColourTable       = pdwColourTable;
    blitInfo.Transparency      = eTransparency;
    blitInfo.TransparentColour = bTransparentColour;
    bitmapBlitSprite( );
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: blitScrollFull
//
//  Description:
//
//      This member is used to blit a bitmap onto this bitmap scrolling
//      the source bitmap with wraparound in all directions.
//
//  Parameters:
//
//      pSrc (input)
//          The bitmap to blit from.
//
//      nDestX (input)
//          The X coordinate to blit to.
//
//      nDestY (input)
//          The Y coordinate to blit to.
//
//      pClipping (input)
//          The clipping region.
//
//      eTransparency (input)
//          The type of transparency.
//  
//      bTransparentColour (input)
//          The transparent colour.
//  
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
BitmapLinux::blitScrollFull( 
    Bitmap*      pSrc,
    int32        nScrollX,
    int32        nScrollY,
    Clipping*    pClipping,
    Transparency eTransparency,
    Byte         bTransparentColour /* = 0x00 */
)
{
    //  Check the parameters.
    ASSERT( pClipping != NULL );

    //  The adjusted scroll values.
    int32 nScrollXAdjusted;
    int32 nScrollYAdjusted;

    //  Adjust the X scroll value.
    if( nScrollX < 0 )
    {
        nScrollXAdjusted = 
            pSrc->getWidth( ) - ( -nScrollX ) % pSrc->getWidth( );
    }
    else
    {
        nScrollXAdjusted = nScrollX % pSrc->getWidth( );
    }
    
    //  Adjust the Y scroll value.
    if( nScrollY < 0 )
    {
        nScrollYAdjusted = 
            pSrc->getHeight( ) - ( -nScrollY ) % pSrc->getHeight( );
    }
    else
    {
        nScrollYAdjusted = nScrollY % pSrc->getHeight( );
    }

    //  Since we can be scrolling in two directions, the clipping can
    //  result in 4 different regions to blit.  For example, if the screen
    //  is 256x256 and we are scrolling to 128,128 then the original bitmap
    //  and scrolled bitmap look like:
    //
    //            +---+---+ 128    +---+---+
    //            | a | b | --->+  | d | c |
    //            +---+---+     |  +---+---+
    //            | c | d | 128 v  | b | a |
    //            +---+---+        +---+---+

    //  Much of the information is static and can be setup initially.
    blitInfo.DestStart         = m_ppbRows;
    blitInfo.DestWidth         = m_dwWidth;
    blitInfo.DestHeight        = m_dwHeight;
    blitInfo.SrcStart          = pSrc->getRows( );
    blitInfo.SrcWidth          = s_getWidth( pSrc );
    blitInfo.SrcHeight         = s_getHeight( pSrc );
    blitInfo.TransparentColour = bTransparentColour;

    //  Assign the clipping region.
    if( m_pCanvas->getTranspose( ) )
    {
        blitInfo.ClipMinX = pClipping->m_nMinY;
        blitInfo.ClipMaxX = pClipping->m_nMaxY;
        blitInfo.ClipMinY = pClipping->m_nMinX;
        blitInfo.ClipMaxY = pClipping->m_nMaxX;
    }
    else
    {
        blitInfo.ClipMinX = pClipping->m_nMinX;
        blitInfo.ClipMaxX = pClipping->m_nMaxX;
        blitInfo.ClipMinY = pClipping->m_nMinY;
        blitInfo.ClipMaxY = pClipping->m_nMaxY;
    }
    if( m_pCanvas->getFlipX( ) )
    {
        blitInfo.ClipMinX = m_dwWidth - 1 - blitInfo.ClipMaxX;
        blitInfo.ClipMaxX = m_dwWidth - 1 - blitInfo.ClipMinX;
    }
    if( m_pCanvas->getFlipY( ) )
    {
        blitInfo.ClipMinY = m_dwHeight - 1 - blitInfo.ClipMaxY;
        blitInfo.ClipMaxY = m_dwHeight - 1 - blitInfo.ClipMinY;
    }


    //  Perform the blit.
    //  (a)
    blitScrollFullGo( 
        nScrollXAdjusted, 
        nScrollYAdjusted,
        eTransparency
    );
    //  (b)
    blitScrollFullGo( 
        nScrollXAdjusted - pSrc->getWidth( ), 
        nScrollYAdjusted,
        eTransparency
    );
    //  (c)
    blitScrollFullGo(
        nScrollXAdjusted,
        nScrollYAdjusted - pSrc->getHeight( ),
        eTransparency
    );
    //  (d)
    blitScrollFullGo(
        nScrollXAdjusted - pSrc->getWidth( ),
        nScrollYAdjusted - pSrc->getHeight( ),
        eTransparency
    );
}


///////////////////////////////////////////////////////////////////////////////
//
//  Function: blitScrollFullGo
//
//  Description:
//
//      This member is used to perform the actual scrolling for 
//      blitScrollFull().  It expects the blit info structure to be
//      filled except for the destination X,Y.
//
//  Parameters:
//
//      nDestX (input)
//          The X coordinate to blit to.
//
//      nDestY (input)
//          The Y coordinate to blit to.
//
//      eTransparency (input)
//          The type of transparency.
//  
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void  
BitmapLinux::blitScrollFullGo( 
    int32        nDestX,
    int32        nDestY,
    Transparency eTransparency
)
{
    //  Set the X/Y Positions.
    if( m_pCanvas->getTranspose( ) )
    {
        blitInfo.DestX = nDestY;
        blitInfo.DestY = nDestX;
    }
    else
    {
        blitInfo.DestX = nDestX;
        blitInfo.DestY = nDestY;
    }
    if( m_pCanvas->getFlipX( ) )
    {
        blitInfo.DestX = m_dwWidth - blitInfo.SrcWidth - blitInfo.DestX;
    }
    if( m_pCanvas->getFlipY( ) )
    {
        blitInfo.DestY = m_dwHeight - blitInfo.SrcHeight - blitInfo.DestY;
    }
    
    //  Perform the blit.
    if( eTransparency == TRANSPARENCY_NONE )
    {
        bitmapBlitScroll( );
    }
    else
    {
        bitmapBlitScrollTransparent( );
    }
}
