///////////////////////////////////////////////////////////////////////////////
//
//  File:    joyl.cpp
//
//  Class:   JoystickLinux
//
//  Author:  Kevin Brisley
//
//  Description:
//
//      The JoystickLinux class encapsulates joystick access for the Linux
//      platform using the kernel joystick module.
//
//
//  Copyright (c) 1997,1998  Kevin Brisley
//  All rights reserved.
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//  Header Files.
///////////////////////////////////////////////////////////////////////////////

//  System Headers.
#include <linux/joystick.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

//  Application Headers.
#include "joyl.h"


///////////////////////////////////////////////////////////////////////////////
//  Static Member Data Initialization.
///////////////////////////////////////////////////////////////////////////////
const int32 JoystickLinux::sm_nThreshold = 256;



///////////////////////////////////////////////////////////////////////////////
//
//  Function: s_build
//
//  Description:
//
//      This is a factory method to create a Linux Joystick object.
//
//  Parameters:
//
//      iName (input)
//          The name of the object.
//
//  Returns:
//
//      A pointer to the new object.
//
///////////////////////////////////////////////////////////////////////////////
JoystickLinux*
JoystickLinux::s_build(
    const KString& iName
)
{
    //  Create the new object.
    JoystickLinux* pThis = new JoystickLinux( iName );

    //  Initialize the new object.
    pThis->init( );

    //  Send back the new joystick.
    return( pThis );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: JoystickLinux
//
//  Description:
//
//      This is the main constructor for the Linux Joystick object.  This is
//      a protected member.  Clients wishing to create a joystick object should
//      do so through the factory method.
//
//  Parameters:
//
//      iName (input)
//          The instance name of the object. 
//
//  Returns:
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
JoystickLinux::JoystickLinux(
    const KString&  iName
)
:
    Joystick  ( iName ),
    m_nJoy0FD ( -1 ),
    m_nJoy1FD ( -1 )
{
    //  All initialization is done in init( ).
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: init
//
//  Description:
//
//      This is called to initialize the joystick object.  By using an init
//      member we get access to virtual functions that we wouldn't in the
//      constructor.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
void
JoystickLinux::init(
)
{
    //  Call the base class.
    Joystick::init( );

    //  If the joystick is disabled then just return.
    if( !m_bEnabled )
    {
        return;
    }

    //  Open the file descriptors for the two joysticks.
    m_nJoy0FD = open( "/dev/js0", O_RDONLY | O_NONBLOCK );
    m_nJoy1FD = open( "/dev/js1", O_RDONLY | O_NONBLOCK );

    //  If no joysticks are found then disable the joystick.
    if( m_nJoy0FD == 0 )
    {
        m_bEnabled = FALSE;
        return;
    }

    //  Read the initial values of the joystick.
    if( read( m_nJoy0FD, &m_origValues0, sizeof( m_origValues0 ) ) < 0 )
    {
        m_bEnabled = FALSE;
        return;
    }
    if( 
        ( m_nJoy1FD >= 0 ) &&
        ( read( m_nJoy1FD, &m_origValues1, sizeof( m_origValues1 ) ) )
    )
    {
        fatalError( "Could not read joystick device (/dev/js1)." );
    }
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: ~JoystickLinux
//
//  Description:
//
//      This is the destructor for a Linux joystick object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      Nothing.
//
///////////////////////////////////////////////////////////////////////////////
JoystickLinux::~JoystickLinux(
)
{
    //  Close the joystick fd's.
    if( m_nJoy0FD > 0 )
    {
        close( m_nJoy0FD );
    }
    if( m_nJoy1FD > 0 )
    {
        close( m_nJoy1FD );
    }
}



///////////////////////////////////////////////////////////////////////////////
//
//  Function: getClassName
//
//  Description:
//
//      This member returns the name of the joystick object.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      The name of the class.
//
///////////////////////////////////////////////////////////////////////////////
const
KString&
JoystickLinux::getClassName(
) const
{
    //  The name of the class.
    static const KString className( "JoystickLinux" );

    return( className );
}




///////////////////////////////////////////////////////////////////////////////
//
//  Function: update
//
//  Description:
//
//      This member should be called periodically to allow the
//      joystick object to update itself.
//
//  Parameters:
//
//      None.
//
//  Returns:
//
//      TRUE  if a joystick state has changed.
//      FALSE if no joystick state has changed.
//
///////////////////////////////////////////////////////////////////////////////
Byte
JoystickLinux::update(
)
{
    //  The number of bytes read from the joystick device.
    int32 nBytesRead;

    //  The current status of the joystick.
    JS_DATA_TYPE curValues;

    //  The status of the joystick controls.
    Byte abStatus[ JOY_COUNT ];

    //  The differences in the X & Y values of the joystick.
    int32 nXDiff;
    int32 nYDiff;


    //  If the joystick is disabled then return.
    if( !m_bEnabled )
    {
        return( FALSE );
    }

    //  Read the current joystick values for the first device.
    nBytesRead = read( m_nJoy0FD, &curValues, sizeof( curValues ) );
    ASSERT( nBytesRead == sizeof( curValues ) );

    //  Extract the current status of the first joystick.
    nXDiff = m_origValues0.x - curValues.x;
    nYDiff = m_origValues0.y - curValues.y;
    abStatus[ JOY1_LEFT ]  = nXDiff > sm_nThreshold;
    abStatus[ JOY1_RIGHT ] = nXDiff < -sm_nThreshold;
    abStatus[ JOY1_UP ]    = nYDiff > sm_nThreshold;
    abStatus[ JOY1_DOWN ]  = nYDiff < -sm_nThreshold;
    abStatus[ JOY1_B1 ]    = curValues.buttons & 0x01;
    abStatus[ JOY1_B2 ]    = curValues.buttons & 0x02 ? TRUE : FALSE;
    //  How do we get the status of B3, B4?
    abStatus[ JOY1_B3 ]    = FALSE;
    abStatus[ JOY1_B4 ]    = FALSE;
    
    //  If the second joystick is enabled then read the values for it.
    if( m_nJoy1FD >= 0 )
    {
        //  Read the current joystick values for the second device.
        nBytesRead = read( m_nJoy1FD, &curValues, sizeof( curValues ) );
        ASSERT( nBytesRead == sizeof( curValues ) );

        //  Extract the current status of the second joystick.
        nXDiff = m_origValues1.x - curValues.x;
        nYDiff = m_origValues1.y - curValues.y;
        abStatus[ JOY2_LEFT ]  = nXDiff > sm_nThreshold;
        abStatus[ JOY2_RIGHT ] = nXDiff < -sm_nThreshold;
        abStatus[ JOY2_UP ]    = nYDiff > sm_nThreshold;
        abStatus[ JOY2_DOWN ]  = nYDiff < -sm_nThreshold;
        abStatus[ JOY2_B1 ]    = curValues.buttons & 0x01;
        abStatus[ JOY2_B2 ]    = curValues.buttons & 0x02 ? TRUE : FALSE;
        //  How do we get the status of B3, B4?
        abStatus[ JOY2_B3 ]    = FALSE;
        abStatus[ JOY2_B4 ]    = FALSE;
    }
    else
    {
        //  Since the second joystick is not enabled, just mark everything
        //  as "OFF".
        abStatus[ JOY2_LEFT ]  = FALSE;
        abStatus[ JOY2_RIGHT ] = FALSE;
        abStatus[ JOY2_UP ]    = FALSE;
        abStatus[ JOY2_DOWN ]  = FALSE;
        abStatus[ JOY2_B1 ]    = FALSE;
        abStatus[ JOY2_B2 ]    = FALSE;
        abStatus[ JOY2_B3 ]    = FALSE;
        abStatus[ JOY2_B4 ]    = FALSE;
    }
    
    //  Now compare the new states with the old states.
    for( DWord dwI = 1 ; dwI < JOY_COUNT ; dwI += 1 )
    {
        //  If the state has changed then set the new state.
        if( abStatus[ dwI ] != m_switchList[ dwI ]->onPhysically( ) )
        {
            m_switchList[ dwI ]->physicalOn( abStatus[ dwI ] );

            //  If the switch just turned off, then mark this switch as the 
            //  last one that was on.
            if( !abStatus[ dwI ] )
            {
                m_dwLastSwitchOn = dwI;
            }
        }
    }
        
    return( FALSE );
}
